	/*

	Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include "audiosubsys.h"
#include "debug.h"
#include "synth_impl.h"
#include "convert.h"

/*
 * Play module to play sample data, either mono or stereo
 */

class SynthGenericPlay :public SynthModule, ArtsNotifyIO, ASProducer
{
	enum { INVALUE_L, INVALUE_R, PROP_CHANNELS };

	int audiofd;

	unsigned char *outblock;
	unsigned long maxsamples;
	unsigned long channels;

	bool haveSubSys, haveFullDuplex;

	AudioSubSystem *as;

public:
	void genericInitialize(bool fullduplex) {
		artsdebug("SynthGenericPlay: starting full duplex operations\n");
		haveCalculateBlock = true;
		haveFullDuplex = fullduplex;

		as = AudioSubSystem::the();

		haveSubSys = as->attachProducer(this);
		if(!haveSubSys)
		{
			printf("SynthGenericPlay: audio subsystem is already used\n");
			return;
		}

		channels = atoi(getStringProperty(PROP_CHANNELS));

		Arts::RealtimeConfig rtc = Synthesizer->RTConfig();

		audiofd = as->open(rtc.FragmentCount,rtc.FragmentSize,
						samplingRate,channels, fullduplex);
		if(audiofd < 0)
		{
			printf("SynthGenericPlay: audio subsystem init failed\n");
			printf("ASError = %s\n",as->error());
			return;
		}

		maxsamples = 0;
		outblock = 0;
	}
	void getfds(int &infd, int &outfd) {
		if(haveFullDuplex) {
			infd = audiofd;
		}
		else {
			infd = -1;
		}
		outfd = audiofd;
	}
	ArtsNotifyIO *ioHandler()
	{
		return this;
	}
	/*
	 * The IO interaction in full duplex mode is as follows:
	 *
	 * 1. this object (SynthGenericPlay) registers the audio descriptor
	 *    of the audio subsystem for select-watching (read & write), and
	 *    registers itself at the audio subsystem as AudioSubSystemClient
	 *
	 * 2. as soon as a read or write notification arrives, it calls the
	 *    handleIO method of the audio subsystem
	 *
	 * 3. sometimes, this method may see that it hasn't got enough data to
	 *    write in its internal buffers, when this happens, it will call
	 *    needMore of its client (which is the same SynthGenericPlay
	 *    object)
	 *
	 * 4. when needMore is called, Synthesizer->Run is called, which in turn
	 *    activates the aRts flow system
	 *
	 * 5. the flow system will go to all end-modules (modules with no outputs)
	 *    and ensure that they get enough data for RTConfig.Cycles samples,
	 *    normally this is 128 samples
	 *
	 * 6. during this, the CalculateBlock of the SynthGenericPlay module
	 *    and the Synth_FULL_DUPLEX_REC module in the flow system will be
	 *    called
	 *
	 * 7. they call the audio subsystem's read & write methods to read and
	 *    write data from the soundcard. However, the audio subsystem will
	 *    perform some buffering as required (using PipeBuffers).
	 */
	void notifyIO(int fd, long type)
	{
		if(!as->running())
		{
			printf("SynthGenericPlay: got notifyIO while audio subsystem"
				 	"is down\n");
			return;
		}
		assert(fd == audiofd);
		switch(type)
		{
			case ArtsIOEvent::ioRead: type = AudioSubSystem::ioRead;
					break;
			case ArtsIOEvent::ioWrite: type = AudioSubSystem::ioWrite;
					break;
			default: assert(false);
		}
		as->handleIO(type);
	}
	void needMore()
	{
		Synthesizer->Run();
	}
	void Calculate()
	{
		assert(false);
	}
	string getParams() { return("invalue_left,invalue_right,_channels;"); }
	void CalculateBlock(unsigned long cycles);
	void DeInitialize()		{
		artsdebug("... deleting synth_play\n");
		artsdebug("SynthGenericPlay: closing audio fd\n");
		AudioSubSystem::the()->detachProducer();
	}
};

void SynthGenericPlay::CalculateBlock(unsigned long samples)
{
	if(!as->running() || !haveSubSys) return;	// no audio subsystem, no play

	if(samples > maxsamples)
	{
		maxsamples = samples;
		outblock = (unsigned char *)realloc(outblock, maxsamples * 4);
													// 2 channels, 16 bit
	}

	float *left = in[INVALUE_L], *right = in[INVALUE_R];

	assert(channels);

	if(channels == 1)
		convert_mono_float_16le(samples,left,outblock);

	if(channels == 2)
		convert_stereo_2float_i16le(samples,left,right,outblock);

	as->write(outblock,channels * 2 * samples);
}

// ============== PLAY MODULE for non full duplex operation ================

class Synth_PLAY :public SynthGenericPlay {
public:
	void Initialize() { genericInitialize(false); }
	static void *Creator() { return new Synth_PLAY; };
};

ModuleClient MC_Synth_PLAY(SynthModule::get_MS, "Synth_PLAY",
										Synth_PLAY::Creator);

// ============== PLAY MODULE for full duplex operation ====================

class Synth_FULL_DUPLEX_PLAY :public SynthGenericPlay {
public:
	void Initialize() { genericInitialize(true); }
	static void *Creator() { return new Synth_FULL_DUPLEX_PLAY; };
};

ModuleClient MC_Synth_FULL_DUPLEX_PLAY(SynthModule::get_MS,
		"Synth_FULL_DUPLEX_PLAY", Synth_FULL_DUPLEX_PLAY::Creator);


// =============== RECORDING MODULE for full duplex operation ==============

class Synth_FULL_DUPLEX_REC :public SynthModule, ASConsumer
{
	enum { OUTVALUE_L, OUTVALUE_R };

	bool audioinit, scaleerr;

	unsigned char *inblock;
	unsigned long maxsamples;
	unsigned long channels;

	bool haveSubSys;
	AudioSubSystem *as;

public:
	void Initialize() {
		artsdebug("Synth_FULL_DUPLEX_REC: starting full duplex operations\n");
		haveCalculateBlock = true;

		as = AudioSubSystem::the();

		haveSubSys = as->attachConsumer(this);
		if(!haveSubSys)
		{
			printf("Synth_FULL_DUPLEX_REC: audio subsystem is already used\n");
			return;
		}

		channels = as->channels();
		maxsamples = 0;
		inblock = 0;
	}
	void DeInitialize() {
		if(haveSubSys) as->detachConsumer();
	}
	void Calculate()
	{
		assert(false);
	}
	void haveMore() {
		// don't care
	}
		void CalculateBlock(unsigned long cycles);
	string getParams() { return(";outvalue_left,outvalue_right"); }
	static void *Creator() { return new Synth_FULL_DUPLEX_REC; };
};

ModuleClient MC_Synth_FULL_DUPLEX_REC(SynthModule::get_MS,
	"Synth_FULL_DUPLEX_REC",Synth_FULL_DUPLEX_REC::Creator);

void Synth_FULL_DUPLEX_REC::CalculateBlock(unsigned long samples)
{
	float *left = out[OUTVALUE_L], *right = out[OUTVALUE_R];

	if(!as->running() || !haveSubSys)
	{
		// no audio subsystem - so lets record some zeros
		unsigned long l;
		for(l=0;l<samples;l++) left[l] = right[l] = 0.0;
		return;
	}

	if(samples > maxsamples)
	{
		maxsamples = samples;
		inblock = (unsigned char *)realloc(inblock, maxsamples * 4);
													// 2 channels, 16 bit
	}

	unsigned long i,size;

	assert(channels);

	size = channels * 2 * samples;
	as->read(inblock,samples*channels*sizeof(short int));

	if(channels == 1)
	{
		for(i=0;i<size;i+=2)
		{
			long sample;

			sample = (((inblock[i + 1]+128)&0xff) << 8) + inblock[i];

			*left++ = *right++ = (float)(sample-32767)/32768.0;
		}
	}
	else if(channels == 2)
	{
		for(i=0;i<size;i+=4)
		{
			long sample;

			sample = (((inblock[i + 1]+128)&0xff) << 8) + inblock[i];

			*left++ = (float)(sample-32767)/32768.0;

			sample = (((inblock[i + 3]+128)&0xff) << 8) + inblock[i+2];

			*right++ = (float)(sample-32767)/32768.0;
		}
	} else {
		// unknown channel count
		assert(false);
	}
}
