    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "structure.h"

ExecutableStructure::ExecutableStructure(Arts::Synthesizer *Synthesizer)
{
	this->Synthesizer = Synthesizer;
	execID = 0;
}

ExecutableStructure::~ExecutableStructure()
{
	// to make destructor virtual
	// stop execution here?
}

void ExecutableStructure::stopExecute()
{
	assert(execID);
	Synthesizer->freeStructure(execID);
	execID = 0;
}

bool ExecutableStructure::isExecuting()
{
	if(!execID) return(false);
	return(Synthesizer->isExecuting(execID));
}

bool ExecutableStructure::saveSession(const char *filename)
{
	assert(execID);

	Arts::StringSeq_var data;
	printf("saveSession\n");
	if(Synthesizer->saveSession(execID,true,data))
	{
		printf("ok\n");
		FILE *file = fopen(filename,"w");
		if(!file) return false;

		unsigned long i;
		for(i=0;i<data->length();i++) fprintf(file,"%s\n",(char *)(*data)[i]);
		fclose(file);

		return true;
	}
	printf("failed\n");
	return false;
}

Structure::Structure(Arts::Synthesizer *Synthesizer)
						:ExecutableStructure(Synthesizer)
{
	StructureDesc = Synthesizer->createStructureDesc();
	StructureDesc->incRef();
	canvas = 0;
}

void Structure::setCanvas(StructureCanvas *canvas)
{
	this->canvas = canvas;
}

Structure::~Structure()
{
	printf("%ld\n",StructureDesc->obtainID());
	printf("releasing structuredesc from structure\n");
	StructureDesc->decRef();
}

void Structure::startExecute(const Arts::ArtsServerSeq& preferredservers)
{
	assert(!execID);

	// just in case synthesis has been halted before,
	// restart it and hope we'll have enough computing power now
	Synthesizer->Reset();
	//execID = Synthesizer->createStructure1(StructureDesc);
	//execID = Synthesizer->createStructure1(StructureDesc);
	execID = Synthesizer->createStructure(StructureDesc,preferredservers);
	assert(execID);
}

void Structure::publish()
{
	Synthesizer->publishStructureDesc(StructureDesc);
}

bool Structure::valid()
{
	return StructureDesc->valid();
}

char *Structure::name()
{
	return StructureDesc->Name();
}

void Structure::rename(const char *newname)
{
	StructureDesc->Name(newname);
}

void Structure::save(const char *filename)
{
	Arts::StringSeq_var list = StructureDesc->saveToList();
	FILE *file = fopen(filename,"w");
	unsigned long i;

	for(i=0;i<list->length();i++) fprintf(file,"%s\n",(char *)(*list)[i]);
	fclose(file);
}

void Structure::clear()
{
	list<StructureComponent *>::iterator ci;

	printf("clc\n");
/*
	for(ci = ComponentList.begin(); ci != ComponentList.end(); ci++)
		delete (*ci);

	ComponentList.erase(ComponentList.begin(), ComponentList.end());
	ModuleList.erase(ModuleList.begin(), ModuleList.end());
*/
	for(ci = ComponentList.begin(); ci != ComponentList.end(); ci++)
		(*ci)->setSelected(true);
	deleteSelected();

	printf("sdc\n");
	// shouldn't do much, now that we deleted every single component of
	// the structure, but we to it anyway, just to be sure.
	StructureDesc->clear();
}

void Structure::retrieve(const char *pubname)
{
	printf("retrieve... %s\n",pubname);
	Arts::StructureDesc_var psd = Synthesizer->lookupStructureDesc(pubname);

	printf("psdlookup ok\n");
	if(psd)
	{
		printf("starting savetolist\n");
		Arts::StringSeq_var strseq=psd->saveToList();
		printf("savetolist ok\n");
		loadFromList(strseq);
		printf("loadfromlist ok\n");
	}
	printf("retrieve... ok\n");
}

void Structure::load(const char *filename)
{
	FILE *infile = fopen(filename,"r");
	Arts::StringSeq_var strseq = new Arts::StringSeq;

	char line[1024];
	unsigned long i = 0;

	while(fgets(line,1024,infile))
	{
		// cut eventual CR and/or LFs at the end of the line
		while(strlen(line) && line[strlen(line)-1] < 14)
			line[strlen(line)-1] = 0;

		strseq->length(i+1);
		(*strseq)[i++] = CORBA::string_dup(line);
	}
	fclose(infile);

	printf(">>loadfromlist...\n");
	loadFromList(strseq);
	printf("<<loadfromlist...\n");
}

void Structure::loadFromList(Arts::StringSeq *strseq)
{
	unsigned long i;
	printf(">>clear\n");
	clear();
	printf("<<clear\n");

	StructureDesc->loadFromList(*strseq);

	Arts::ModuleDescSeq_var modules = StructureDesc->Modules();

	for(i=0;i<modules->length();i++)
	{
		assert(canvas);
		Module *m = new Module((*modules)[i],StructureDesc,canvas);

		m->show();
		ModuleList.push_back(m);
		ComponentList.push_back(m);
	}

	Arts::StructurePortDescSeq_var ports = StructureDesc->Ports();
	for(i=0;i<ports->length();i++)
	{
		assert(canvas);
		StructurePort *p = new StructurePort(Arts::StructurePortDesc::_duplicate((*ports)[i]),StructureDesc,canvas);

		p->show();
		ComponentList.push_back(p);
	}
}

Module *Structure::createModule(const Arts::ModuleInfo *minfo)
{
	assert(canvas);
	Module *m = new Module(minfo,StructureDesc,canvas);

	ComponentList.push_back(m);
	ModuleList.push_back(m);
	return m;
}

StructurePort *Structure::createStructurePort(const Arts::PortType type)
{
// portname generation
	unsigned long portindex = 1,l,baselen;;
	char name[100];

	string namebase;
	if(type.Direction == Arts::input) {
		// this is a port where our structure puts its results
		// so it is an input port, that is named out
		namebase = "out"; baselen = 3;
    } else {
		// this is a port where our structure gets data from
		// so it is an output port, that is named in
		namebase = "in"; baselen = 2;
	}

	Arts::StructurePortDescSeq_var sps = StructureDesc->Ports();

	for(l=0;l<sps->length();l++) {
		CORBA::String_var thisname = (*sps)[l]->Name();
		if(strncmp((const char *)thisname, namebase.c_str(), baselen) == 0 &&
           strlen((const char *)thisname) > baselen)
		{
			unsigned long index2 = atol(&thisname[baselen]);
			if(index2 >= portindex) portindex = index2+1;
		}
	}

	sprintf(name,"%s%ld",namebase.c_str(),portindex);
	printf("new Portname: %s\n",name);
	Arts::StructurePortDesc* spd =
         StructureDesc->createStructurePortDesc(type,name);
	assert(canvas);
	StructurePort *s = new StructurePort(spd,StructureDesc,canvas);
	ComponentList.push_back(s);
	return s;
}

list<Module *> *Structure::getModuleList()
{
	return(&ModuleList);
}

list<StructureComponent *> *Structure::getComponentList()
{
	return(&ComponentList);
}

long Structure::countSelected()
{
	list<StructureComponent *>::iterator ci;
	long targets = 0;

	for(ci=ComponentList.begin();ci!=ComponentList.end();ci++)
		if((*ci)->selected()) targets++;

	return targets;
}

void Structure::deleteSelected()
{
	printf("deleteSelected...\n");

	// remove module from the ModuleList
	list<Module *>::iterator mi;
	for(mi=ModuleList.begin();mi!=ModuleList.end();)
	{
		if((*mi)->selected())
		{
			ModuleList.erase(mi);
			mi = ModuleList.begin();
		}
		else mi++;
	}

	// and from the ComponentList (the future ;)

	list<StructureComponent *>::iterator ci = ComponentList.begin();

	while(ci!=ComponentList.end())
	{
		if((*ci)->selected())
		{
			delete (*ci);
			ComponentList.erase(ci);
			ci = ComponentList.begin();
		}
		else ci++;
	}

	printf("deleteSelected ok.\n");
}

StructureComponent *Structure::componentAt(long x, long y, bool ignore_selected)
{
	list<StructureComponent *>::iterator ci;

	for(ci=ComponentList.begin();ci!=ComponentList.end();ci++)
	{
		StructureComponent *c = *ci;

		if(x >= c->x() && x < c->x()+c->width() &&
           y >= c->y() && y < c->y()+c->height())
		{
			if((c->selected() && !ignore_selected) || !c->selected()) return c;
		}
	}

	return 0;
}
