	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "guiinstdlg.h"
#include "namedlg.h"
#include "structureport.h"
#include <qcombobox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qframe.h>
#include <kbuttonbox.h>
#include <qbttngrp.h>
#include <qradiobt.h>
#include <kapp.h>
#include <kiconloader.h>
#include <qlined.h>
#include "utils.h"
#include "sequenceutils.h"

static void min_size(QWidget *w) {
  w->setMinimumSize(w->sizeHint());
} 

GuiInstDlg::GuiInstDlg(QWidget *parent, Arts::Synthesizer *Synthesizer) :QDialog(parent,"Props", TRUE)
{
	unsigned long i;
	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);

	setCaption(i18n("Add instrument"));

	QVBoxLayout *mainlayout = new QVBoxLayout(this);
	QGridLayout *contentslayout = new QGridLayout(4,3);

	mainlayout->addLayout(contentslayout);
	contentslayout->addColSpacing(1,15);

// midichannel

	QLabel *labelChannel = new QLabel(this);
	labelChannel->setText(i18n("MidiChannel"));
	min_size(labelChannel);

	contentslayout->addWidget(labelChannel,0,0);

	cbChannel = new QComboBox(this);
	for(i=1;i<=16;i++)
	{
		char text[3];
		sprintf(text,"%ld",i);
		cbChannel->insertItem(text);
	}

	min_size(cbChannel);
	contentslayout->addWidget(cbChannel,0,2);

// instrument

	QLabel *labelInstrument = new QLabel(this);
	labelInstrument->setText(i18n("Instrument"));
	min_size(labelInstrument);

	contentslayout->addWidget(labelInstrument,1,0);

	cbInstrument = new QComboBox(this);
//-------------------- find all possible instruments: -----------------------

	printf("> getting mbroker\n");
	ModuleBroker = Synthesizer->moduleBroker();
	printf("> getting modules\n");
	Arts::StringSeq_var modules = ModuleBroker->publishedModules();

	for(i=0;i<modules->length();i++)
	{
		const char *modname = (*modules)[i];
		const char *prefix = "instrument_";
		printf("> modules[%ld] = %s\n",i,modname);
		if(strncmp(modname,prefix,strlen(prefix)) == 0 &&
		   strcmp(&modname[strlen(modname)-4],"_GUI") != 0)
		{
			cbInstrument->insertItem(&modname[strlen(prefix)]);
			printf("> thats fine\n");
		}
	}

	printf("> getting maps\n");
	Arts::StringSeq_var maps = Synthesizer->listFiles("maps",".arts-map");
	for(i=0;i<maps->length();i++)
	{
		cbInstrument->insertItem(maps[i]);
	}

//----------------------------------------------------------------------------
	min_size(cbInstrument);
	contentslayout->addWidget(cbInstrument,1,2);

// destination

	QLabel *labelDestination = new QLabel(this);
	labelDestination->setText(i18n("Destination"));
	min_size(labelDestination);

	contentslayout->addWidget(labelDestination,2,0);

	cbDestination = new QComboBox(this);

//-------------------- find all possible destinations: -----------------------
	printf("> getting destinations\n");
	Arts::StringSeq_var destinations = Synthesizer->busList();
	printf("> ok \n");

	for(i=0;i<destinations->length();i++)
		cbDestination->insertItem((const char *)(*destinations)[i]);
	min_size(cbDestination);
	printf("> layout \n");
	contentslayout->addWidget(cbDestination,2,2);

//----------------------------------------------------------------------------

//
	QLabel *labelMax = new QLabel(this);
	labelMax->setText(i18n("MaxVoices"));
	min_size(labelMax);

	contentslayout->addWidget(labelMax,3,0);

	eMaxVoices = new QLineEdit(this);
	eMaxVoices->setText("8");
	min_size(eMaxVoices);
	contentslayout->addWidget(eMaxVoices,3,2);

// hruler

	mainlayout->addSpacing(5);
	QFrame *ruler2 = new QFrame(this);
	ruler2->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	ruler2->setMinimumSize(50,2);
	mainlayout->addWidget(ruler2);

// buttons

	QHBoxLayout *buttonlayout = new QHBoxLayout;
	mainlayout->addSpacing(5);
	mainlayout->addLayout(buttonlayout);
	mainlayout->addSpacing(5);

	buttonlayout->addSpacing(5);
	KButtonBox *bbox = new KButtonBox(this);

	bbox->addButton(i18n("Help"));
	bbox->addStretch(1);

	QButton *raise = bbox->addButton(i18n("Cancel"));
	connect( raise, SIGNAL( clicked() ), SLOT( reject() ));

	QButton *okbutton = bbox->addButton(i18n("Okay"));
	connect( okbutton, SIGNAL( clicked() ), SLOT(accept() ) );

/*
	QButton *cancelbutton = bbox->addButton(i18n("Cancel"));
	connect( cancelbutton, SIGNAL( clicked() ), SLOT(reject() ) );
*/
	bbox->layout();
	//min_size(bbox);

	buttonlayout->addWidget(bbox);
	buttonlayout->addSpacing(5);

	//mainlayout->activate();
	mainlayout->freeze();
};

InstListViewItem *GuiInstDlg::create(QListView *parent)
{
	string channelstr = cbChannel->text(cbChannel->currentItem());

	float channel = atof(channelstr.c_str())-1.0;
	if(channel < 0.0) channel = 0.0;

	float maxvoices = atof(eMaxVoices->text());
	if(maxvoices < 1) maxvoices = 8.0;

	return new InstListViewItem(Synthesizer,ModuleBroker,parent,channel,
		maxvoices, cbInstrument->text(cbInstrument->currentItem()),
		cbDestination->text(cbDestination->currentItem()));
}

Arts::StructureDesc_ptr InstListViewItem::createStructure()
{
    unsigned long i;

//--- Implications from user selected parameters
	string instrument = string("instrument_")+instrumentShort;
	string instrumentgui = instrument + string("_GUI");

//--- get a structure

	Arts::StructureDesc_ptr sd = Synthesizer->createStructureDesc();
	assert(sd);
	sd->incRef();

//--- put a midirouter (or midimaprouter) in there

	string mrtype;

	const char *iscs = instrumentShort.c_str();
	if(strlen(iscs) > 9 && strncmp(&iscs[strlen(iscs)-9],".arts-map",9) == 0)
		mrtype = "Synth_MIDI_MAP_ROUTER";
	else
		mrtype = "Synth_MIDI_ROUTER";

	Arts::ModuleInfo_var mrinfo =
		ModuleBroker->lookupModule(mrtype.c_str());
	assert(mrinfo);

    Arts::ModuleDesc_var mrmod = sd->createModuleDesc(mrinfo);
	assert(mrmod);
		 
//--- if its a gui thingy, then add a window for the parameters

	Arts::PortDesc_var winport;
	bool havewinport = false;

/*
 * seems that these var types always transfer "something", so they have
 * no "unassigned value" like 0, NULL or nil
 *
 * FIXME: need to change the ModuleBroker interface later

	Arts::ModuleInfo_var guiinfo = 
		ModuleBroker->lookupModule(instrumentgui.c_str());
	if (guiinfo)
 */

	Arts::StringSeq_var publishedModules = ModuleBroker->publishedModules();

	unsigned long l;
	bool has_gui = false;
	for(l=0;l<publishedModules->length();l++)
	{
		if(strcmp((*publishedModules)[l],instrumentgui.c_str()) == 0)
			has_gui = true;
	}
	if(has_gui)
	{
		printf("instrument %s has a gui %s (building a window)\n",
					instrument.c_str(), instrumentgui.c_str());

		Arts::ModuleInfo_var wininfo =
			ModuleBroker->lookupModule("Gui_WINDOW");
		assert(wininfo);

    	Arts::ModuleDesc_var winmod = sd->createModuleDesc(wininfo);
		assert(winmod);

		Arts::PortDescSeq_var wports = winmod->Ports();

    	for(i=0;i<wports->length();i++)
		{
        	Arts::PortDesc_ptr p = (*wports)[i];
            CORBA::String_var name = p->Name();
            if(strcmp((const char *)name,"id") == 0)
			{
				winport = Arts::PortDesc::_duplicate(p);
				havewinport = true;
			}
            if(strcmp((const char *)name,"caption") == 0)
			{
				char caption[1024];
				sprintf(caption, "Ch#%d:%s",(int)(channel+1.5),
												instrumentShort.c_str());
				p->StringValue(caption);
			}
		}
	}

//--- setup the ports of the midirouter

	printf("creating instrument (%f %s %s)\n",channel,instrument.c_str(),
		destination.c_str());

	Arts::PortDescSeq_var ports = mrmod->Ports();
    for(i=0;i<ports->length();i++)
    {
        Arts::PortDesc_ptr p = (*ports)[i];
        Arts::PortType pt = p->Type();
 
        if(pt.Direction == Arts::input)
        {
            CORBA::String_var name = p->Name();
 
            if(strcmp((const char *)name,"y") == 0)
				p->FloatValue(0);
            if(strcmp((const char *)name,"x") == 0)
				p->FloatValue(0);
            if(strcmp((const char *)name,"parent") == 0)
			{
				if(havewinport)
				{
					p->connectTo(winport);
				}
				else
				{
					// well, it shouldn't have a GUI now, otherwise
					// setting the parent to 0 will break things
					p->FloatValue(0);
				}
			}
            if(strcmp((const char *)name,"maxvoices") == 0)
				p->FloatValue(maxVoices);
            if(strcmp((const char *)name,"channel") == 0)
				p->FloatValue(channel); 
            if(strcmp((const char *)name,"bus") == 0)
				p->StringValue(destination.c_str());
            if(strcmp((const char *)name,"structure") == 0)
				p->StringValue(instrument.c_str());
            if(strcmp((const char *)name,"mapfile") == 0)
				p->StringValue(instrumentShort.c_str());
		}
	}
	return sd;
}
		
InstListViewItem::InstListViewItem(Arts::Synthesizer_ptr Synthesizer,
		Arts::ModuleBroker_ptr ModuleBroker, QListView *parent, float channel,
		float maxVoices, const char *instrumentShort, const char *destination)
		:QListViewItem(parent)
{
	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
	this->ModuleBroker = Arts::ModuleBroker::_duplicate(ModuleBroker);
	this->channel = channel;
	this->maxVoices = maxVoices;
	this->instrumentShort = instrumentShort;
	this->destination = destination;

	Arts::StructureDesc_var sd = createStructure();
	Arts::ArtsServerSeq preferredservers;

	instrumentID = Synthesizer->createStructure(sd,preferredservers);
	assert(instrumentID);

	sd->decRef();
}

InstListViewItem::~InstListViewItem()
{
	Synthesizer->freeStructure(instrumentID);
}

const char *InstListViewItem::key(int column,bool ascending) const
{
	static char buffer[10];
	if(column == 0)
	{
		sprintf(buffer,"%03d",(int)(1.5+channel));
		return buffer;
	}
	return text(column);
}

const char *InstListViewItem::text(int column) const
{
	static char buffer[10];

	switch(column)
	{
		case 0: sprintf(buffer,"%d",(int)(1.5+channel));
				return buffer;
			break;
		case 1: return instrumentShort.c_str();
			break;
		case 2: return destination.c_str();
			break;
		default: return 0;
	}
}

Arts::StringSeq *InstListViewItem::saveSessionParameters(list<long>& IDs)
{
	Arts::StringSeq *result = new Arts::StringSeq;

	sqprintf(result,"instrument_id=%ld",instrumentID);
	IDs.push_back(instrumentID);

	sqprintf(result,"channel=%f",channel);
	sqprintf(result,"maxvoices=%f",maxVoices);
	sqprintf(result,"instrument=%s",instrumentShort.c_str());
	sqprintf(result,"destination=%s",destination.c_str());

	return result;
}

InstListViewItem::InstListViewItem(Arts::Synthesizer_ptr Synthesizer,
		Arts::ModuleBroker_ptr ModuleBroker, QListView *parent,
		const Arts::StringSeq& params, long restoreID) :QListViewItem(parent)
{

	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
	this->ModuleBroker = Arts::ModuleBroker::_duplicate(ModuleBroker);

	printf("* InstListViewItem restoring instrument\n");
   	char *cmd,*param;

	long instrumentOldID = -1;

	unsigned long i;
   	for(i=0;i<params.length();i++)
   	{
   		if(parse_line(params[i],cmd,param))   // otherwise: empty or comment
   		{
   			if(strcmp(cmd,"instrument_id") == 0) {
				instrumentOldID = atol(param);
			} else if(strcmp(cmd,"channel") == 0) {
				channel = atof(param);
			} else if(strcmp(cmd,"maxvoices") == 0) {
				maxVoices = atof(param);
			} else if(strcmp(cmd,"instrument") == 0) {
				instrumentShort = param;
				printf("* got instrumentShort = %s\n",instrumentShort.c_str());
			} else if(strcmp(cmd,"destination") == 0) {
				destination = param;
			}
		}
   	}

	if(instrumentOldID != -1)
	{
		Arts::StructureDesc_var sd = createStructure();
		Arts::ArtsServerSeq preferredservers;

		instrumentID = Synthesizer->restoreStructure(sd,preferredservers,
												restoreID,instrumentOldID);
		assert(instrumentID);

		sd->decRef();
	}
}
