// KreateCD - CD recording software for the K desktop environment
//
// 1998-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "CDWriter.h"
#include "CDTrack.h"

#include <qstring.h>
#include <qmessagebox.h>

#include <kapp.h>
#include <kconfig.h>
#include <klocale.h>

#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include "uidcontrol.h"
#include "pathconfig.h"

extern KLocale *locale;

int CDWriter::burnCD(int ntrack,CDTrack **track,int simmode,int speed,Fixating fix,int eject)
 {
  return(callRecord(ntrack,track,simmode,speed,fix,eject,false));
 }

long int CDWriter::getCDFree(int ntrack,CDTrack **track,Fixating fix)
 {
  int rc;
  rc=callRecord(ntrack,track,1,1,fix,0,true);
  if (rc!=1) return(-1);
  if (ntrack==0)
   {
    return(blocksFree);
   }
   else
   {
    return(blocksRemain);
   }
 }

int CDWriter::callRecord(int ntrack,CDTrack **track,int simmode,int speed,Fixating fix,int eject,bool capread)
 {
  int retval,i;
  CDTrack *xtrack;
  KConfig *config;
  enum CDTrack::TrackType cmode,xmode;

  char devline[32],speedline[32];
  int unit,host;


  getCapacity=capread;

  prepareProcess();
#if QT_VERSION >= 200
  config=kapp->config();
#else
  config=kapp->getConfig();
#endif
  config->setGroup("SCSI");
  host=config->readNumEntry("SCSIWriterHost",0);
  unit=config->readNumEntry("SCSIWriterUnit",-1);
  if (unit==-1) return(0);

  sprintf(devline,"dev=%d,%d,0",host,unit);
  sprintf(speedline,"speed=%d",speed);
  config->setGroup("Path");
  if ROOT_CALLED
   {
    *this<<config->readEntry("PathCdrecord",PATH_CDRECORD)<<"-v"<<devline<<speedline;
   }
   else
   {
    *this<<PATH_CDRECORD<<"-v"<<devline<<speedline;
   }
  if (simmode) *this<<"-dummy";
  switch (fix)
   {
    case Fix_None:
      *this<<"-nofix";
      break;
    case Fix_Session:
      *this<<"-multi";
      break;
    case Fix_CD:
      break;
   }

  if (eject) *this<<"-eject";



  cmode=CDTrack::Track_DataMode1;

  for (i=0;i<ntrack;++i)
   {
    char imagefile[1024];
    xtrack=track[i];

    if (!xtrack->getBurnFile(imagefile))
     {
#if QT_VERSION >= 200
      QMessageBox::critical(0,QString::null,locale->translate("Image file not found!"));
#else
      QMessageBox::critical(0,0,locale->translate("Image file not found!"));
#endif
      return(0);
     }
    xmode=xtrack->getTrackType();
    if ( (xmode!=cmode) || (i==0) )
     {
      if (cmode==CDTrack::Track_AudioPreEmph) *this<<"-nopreemp";
      cmode=xmode;
      switch(xmode)
       {
        case CDTrack::Track_Audio:
          *this<<"-audio";
          break;
        case CDTrack::Track_AudioPreEmph:
          *this<<"-audio"<<"-preemp";
          break;
        case CDTrack::Track_DataMode1:
          *this<<"-data";
          break;
        case CDTrack::Track_DataMode2:
          *this<<"-mode2";
          break;
        case CDTrack::Track_XA1:
          *this<<"-xa1";
          break;
        case CDTrack::Track_XA2:
          *this<<"-xa2";
          break;
        case CDTrack::Track_CDI:
          *this<<"-cdi";
          break;
       }
     }
    *this<<imagefile;
   }

  if (ntrack==0)
   {
    if (!getCapacity)
     {
      closeProcess();
      return(0);
     }
     else
     {
      FILE *duf;
      char dumpath[512];
      KConfig *mconfig;
#if QT_VERSION >= 200
      mconfig=kapp->config();
#else
      mconfig=kapp->getConfig();
#endif
      mconfig->setGroup("Path");
      strcpy(dumpath,mconfig->readEntry("Temporary","/tmp"));
      strcat(dumpath,"/dummyfile");
      *this<<dumpath;
      duf=fopen(dumpath,"wb");
      fclose (duf);
     }
   }


  setWorkText(getCapacity?locale->translate("Reading capacity..."):locale->translate("Preparing burning..."));
  if (!getCapacity) addStatusView();

  burnstate=0;
  burntrack=0;
  burnsize=100;



  retval=startProcess();  // 1=ripping was OK  -1=abort -2=ripping error  0=window hard closed
  closeProcess();

  if (retval==-2)
   {
    switch (burnstate)
     {
#if QT_VERSION >= 200
      case 10000:
        QMessageBox::critical(0,QString::null,locale->translate("Cannot access the drive! (no disc?)"));
        break;
      case 10001:
        QMessageBox::critical(0,QString::null,locale->translate("Cannot open a new session!"));
        break;
      default:
        QMessageBox::critical(0,QString::null,locale->translate("Unrecognized error while writing CD-R!!!"));
        break;
#else
      case 10000:
        QMessageBox::critical(0,0,locale->translate("Cannot access the drive! (no disc?)"));
        break;
      case 10001:
        QMessageBox::critical(0,0,locale->translate("Cannot open a new session!"));
        break;
      default:
        QMessageBox::critical(0,0,locale->translate("Unrecognized error while writing CD-R!!!"));
        break;
#endif
     }
   }

  if (ntrack==0)
   {
    if (getCapacity)
     {
      char dumpath[512];
      KConfig *mconfig;
#if QT_VERSION >= 200
      mconfig=kapp->config();
#else
      mconfig=kapp->getConfig();
#endif
      mconfig->setGroup("Path");
      strcpy(dumpath,mconfig->readEntry("Temporary","/tmp"));
      strcat(dumpath,"/dummyfile");
      remove(dumpath);
     }
   }

  if ( (retval!=1) || (getCapacity))
   {
    // workaround : open close writer device to unlock tray
    FILE *xfile;
    QString qs;
    config->setGroup("SCSI");
    qs=config->readEntry("SCSIWriterDevice","/dev/null");
    xfile=fopen(qs.data(),"rb");
    if (xfile!=0) fclose(xfile);
   }

  if (retval!=1) return(0);

  return(1);
 }

int CDWriter::processExited(void)
 {
  return((burnstate==99999)?1:-2);
 }

bool CDWriter::processStdoutLine(char *linebuffer)
 {
   if (linebuffer[0]==0) return(true);
   if (strncmp("Track ",linebuffer,6)==0)
   {
    char *xptr,*yptr;
    long cur,max,track,fifo;
    xptr=linebuffer+6;
    while (*xptr==' ') ++xptr;
    if (*xptr==0)
     {
      printStatusLine(QString(linebuffer));
      return(true);
     }
    track=strtol(xptr,&yptr,10);
    if (*yptr!=':')
     {
      printStatusLine(QString(linebuffer));
      return(true);
     }
   xptr=yptr+1;
   while (*xptr==' ') ++xptr;
    if ( (*xptr=='0') && (*(xptr+1)==' ') )
     {
      cur=0;
      yptr=xptr+1;
     }
     else
     {
      cur=strtol(xptr,&yptr,10);
     }
    if (strncmp(" of ",yptr,4)!=0)
     {
      printStatusLine(QString(linebuffer));
      return(true);
     }
   xptr=yptr+4;
    while (*xptr==' ') ++xptr;
    if (*xptr==0)
     {
      printStatusLine(QString(linebuffer));
      return(true);
     }
    max=strtol(xptr,&yptr,10);
    fifo=100;
    if (strncmp(" MB written",yptr,11)==0) goto noFifo;
    if (strncmp(" MB written (fifo ",yptr,18)!=0)
     {
      printStatusLine(QString(linebuffer));
      return(true);
     }
   xptr=yptr+18;
   while (*xptr==' ') ++xptr;
    if (*xptr==0)
     {
      printStatusLine(QString(linebuffer));
      return(true);
     }
    fifo=strtoul(xptr,&yptr,10);
    noFifo:
    if (max==0) max=1;
   setProgress(cur,max,true);
    if (burntrack==0)
    {
      char xwork[128];
      sprintf(xwork,locale->translate("Writing track %ld"),track);
      setWorkText(xwork);
     }
    return(true);
   }

  printStatusLine(QString(linebuffer));
  if (strncmp("Starting to write CD",linebuffer,20)==0)
   {
    if (getCapacity)
     {
      burnstate=99999;
      usleep(1500);  // make sure cdrecord is interruptible
      return(false);
     }
    return(true);
   }
  if (strncmp("Starting new track at sector:",linebuffer,29)==0)
   {
    burntrack=0;
    setProgress(0,100);
    return(true);
   }
  if (strncmp("Fixating...",linebuffer,11)==0)
   {
    setWorkText(locale->translate("Fixating..."));
    return(true);
   }
  if (strncmp("Blocks total:",linebuffer,13)==0)
   {
    char *xptr,*yptr;
    xptr=linebuffer+13;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    blocksTotal=strtoul(xptr,&yptr,10);
    if (strncmp(" Blocks current:",yptr,16)!=0) return(true);
    xptr=yptr+16;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    blocksFree=strtoul(xptr,&yptr,10);
    if (strncmp(" Blocks remaining:",yptr,18)!=0) return(true);
    xptr=yptr+18;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    blocksRemain=strtoul(xptr,&yptr,10);
    if (xptr==yptr) return(true);
    return(true);
   }

   return(true);
 }

bool CDWriter::processStderrLine(char *linebuffer)
 {

  printStatusLine(QString(linebuffer));
   {
    char *xptr;
    xptr=linebuffer;
    while ( (*xptr!=0) && (*xptr!=':') ) ++xptr;
    if (*xptr!=0)
     {
     xptr++;
     while (*xptr==' ') ++xptr;
      if (strncmp("fifo had ",xptr,9)==0)
       {
        if (burnstate<10000) burnstate=99999;
        return(true);
       }
      if (strncmp("No disk",xptr,7)==0)
       {
        burnstate=10000;
        return(true);
       }
      if (strncmp("Cannot open new session",xptr,23)==0)
       {
        burnstate=10001;
       }
     }
  }
  return(true);
}
