// KreateCD - CD recording software for the K desktop environment
//
// 1999 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "IsoFile.h"

#include <qstring.h>


#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>


ISOFile::ISOFile(ISOFile::ISOType type,QString *name,QString *reference)
 {
  parentObject=0;
  subObjects=new ISOFileList();
  objectType=type;
  objectName=new QString(name->data());
  if (reference==0)
   {
    objectReference=0;
   }
   else
   {
    objectReference=new QString(reference->data());
   }
 }

ISOFile::~ISOFile(void)
 {
  ISOFile *ifile;
  while ( (ifile=subObjects->first()) !=0)
   {
    delete ifile;
   }

  if (parentObject!=0)
   {
    parentObject->subObjects->remove(this);
   }
  delete objectName;
  if (objectReference!=0) delete objectReference;
  delete subObjects;
 }

ISOFile::ISOFile(const ISOFile &obj)
 {
  ISOFile *subobj,*subobj2;

  objectType=obj.objectType;
#if QT_VERSION >= 200
  objectName=new QString(*obj.objectName);
  if (obj.objectReference!=0)
   {
    objectReference=new QString(*obj.objectReference);
   }
   else
   {
    objectReference=0;
   }
#else
  objectName=new QString(obj.objectName->data());
  if (obj.objectReference!=0)
   {
    objectReference=new QString(obj.objectReference->data());
   }
   else
   {
    objectReference=0;
   }
#endif
  parentObject=0;
  subObjects=new ISOFileList();
  subobj=obj.subObjects->first();
  while ( subobj!= 0)
   {
    subobj2=new ISOFile(*subobj);
    subObjects->append(subobj2);
    subobj2->parentObject=this; 
    subobj=obj.subObjects->next();
   }
 }

void ISOFile::addObject(class ISOFile *object)
 {
  this->subObjects->append(object);
  object->parentObject=this;
 }

const class QString *ISOFile::name(void)
 {
  return (objectName);
 }

const class QString *ISOFile::reference(void)
 {
  return (objectReference);
 }

ISOFile::ISOType ISOFile::type(void)
 {
  return (objectType);
 }

ISOFile *ISOFile::getChildren(ISOFile *walker)
 {
  if (walker==0)
   {
    return(subObjects->first());
   }
  subObjects->findRef(walker);
  return(subObjects->next());
 }


ISOFile *ISOFile::findChildren(const QString *name)
 {
  ISOFile *file;
  file=subObjects->first();
  while (file!=0)
   {
    if (strcmp(name->data(),file->objectName->data())==0) return(file);
    file=subObjects->next();
   }
  return(0);
 }

bool ISOFile::createTree(const char *prefix)
 {
  ISOFile *file;
  QString filename;
  file=subObjects->first();
  while (file!=0)
   {
    filename=prefix;
    filename+="/";
    filename+=file->objectName->data();
    switch (file->objectType)
     {
      case ISO_ISODir:
        if (mkdir(filename,0700)!=0)
         {
          return(false);
         }
        if (!file->createTree(filename.data()))
         {
          return(false);
         }
        break;
      case ISO_RealFile:
      case ISO_RealDir:
        if (symlink(file->objectReference->data(),filename.data())!=0)
         {
          return(false);
         }
        break;
     }
    file=subObjects->next();
   }
  return(true);
 }

bool ISOFile::deleteTree(const char *prefix)
 {
  ISOFile *file;
  QString filename;
  file=subObjects->first();
  while (file!=0)
   {
    filename=prefix;
    filename+="/";
    filename+=file->objectName->data();
    switch (file->objectType)
     {
      case ISO_ISODir:
        if (!file->deleteTree(filename.data()))
         {
          return(false);
         }
        if (rmdir(filename.data())!=0)
         {
          if (errno!=ENOENT) return(false);
         }
        break;
      case ISO_RealFile:
      case ISO_RealDir:
        if (unlink(filename.data())!=0)
         {
          if (errno!=ENOENT) return(false);
         }
        break;
     }
    file=subObjects->next();
   }
  return(true);
 }

ISOFile *ISOFile::getParent(void)
 {
  return parentObject;
 }
