/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <SFString.h>
#include <Transform.h>

/*************************************************************************/
void Transform::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Transform::eventIn (" << DEF << ")\n";
#endif 
  
  /* Transforms only take in events and do not propogate */
  if ((eventType == "translation") || (eventType == "set_translation")) {
    SFVec3f *n = (SFVec3f *) eventData;
    translation = *n;

  } else if ((eventType == "rotation") || (eventType == "set_rotation")) {
    SFRotation *n = (SFRotation *) eventData;
    rotation = *n;

  } else if ((eventType == "scale") || (eventType == "set_scale")) {
    SFVec3f *n = (SFVec3f *) eventData;
    scale = *n;

  } else if ((eventType == "scaleOrientation") || (eventType == "set_scaleOrientation")) {
    SFRotation *n = (SFRotation *) eventData;
    scaleOrientation = *n;

  } else if ((eventType == "center") || (eventType == "set_center")) {
    SFVec3f *n = (SFVec3f *) eventData;
    center = *n;

  } else if (eventType == "addChildren") {
    children.add(eventData);

  } else if (eventType == "removeChildren") {
    children.remove(eventData);

  }

  /* prepare the children */
  createMatrix();

}
/*************************************************************************/

/*************************************************************************/
void Transform::createMatrix()
{

  /* store the current transform */
  glPushMatrix();

  /* load the identity */
  glLoadIdentity();

  /* unadjust the center of rotation and scale */
  glTranslated(translation[0],
	       translation[1], 
	       translation[2]);

  /* adjust for the center */
  glTranslated(center[0], 
	       center[1], 
	       center[2]);

  /* apply the rotation */
  glRotatef(rotation[3], 
	    rotation[0], 
	    rotation[1],
	    rotation[2]);

  /* apply the scale rotation */
  glRotatef(scaleOrientation[3], 
	    scaleOrientation[0], 
	    scaleOrientation[1],
	    scaleOrientation[2]);

  /* apply the scale */
  glScaled(scale[0], 
	   scale[1],
	   scale[2]);

  /* apply the scale rotation */
  glRotatef(-float(scaleOrientation[3]), 
	    scaleOrientation[0], 
	    scaleOrientation[1],
	    scaleOrientation[2]);
  
  /* adjust for the center */
  glTranslated(-float(center[0]), 
	       -float(center[1]), 
	       -float(center[2]));

  /* get the matrix */
  glGetDoublev (GL_MODELVIEW_MATRIX,  m);

  /* restore one level of the stack */
  glPopMatrix();

}
/*************************************************************************/

/*************************************************************************/
void Transform::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Transform::prepare (" << DEF << ")\n";
#endif 

  /* create the matrix */
  createMatrix();

  /* store the current transformation */
  glPushMatrix();

  /* multiply the new matix */
  glMultMatrixd(m);

  /* prepare the children */
  children.prepare(min, max);

  /* restore the matrix stack */
  glPopMatrix();

}
/*************************************************************************/

/*************************************************************************/
void Transform::render(SFRenderInfo &SFRI)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Transform::render (" << DEF << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif 

  /* store the current transformation */
  glPushMatrix();

  /* store the new matrix */
  glMultMatrixd(m);

  /* render the children */
  children.render(SFRI);

  /* restore the matrix stack */
  glPopMatrix();

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone Transform\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void Transform::parse(char *header, istream &InFile)
{

  /* we a re valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tTransform: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "bboxSize") {
      bboxSize.parse(header, InFile);

    } else if (Token == "bboxCenter") {
      bboxCenter.parse(header, InFile);

    } else if (Token == "translation") {
      translation.parse(header, InFile);

    } else if (Token == "rotation") {
      rotation.parse(header, InFile);

    } else if (Token == "scale") {
      scale.parse(header, InFile);

    } else if (Token == "scaleOrientation") {
      scaleOrientation.parse(header, InFile);

    } else if (Token == "center") {
      center.parse(header, InFile);

    } else if (Token == "children") {
      children.parse(header, InFile);

    } else if (Token == "Transform") {
      TokenFound = true;

    } else if (Token == "") {

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/

