/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* miscellaneos headers */
#include <iostream.h>
#include <fstream.h>
#include <stdio.h>

/* local headers */
#include <SFToken.h>
#include <SFImage.h>

/* for reading jpeg images */
extern "C" {
#   define XMD_H
#   include <jpeglib.h>
#   undef XMD_H
#   include <gif_lib.h>
#   include<png.h>
	   }

/*************************************************************************/
static bool isPowTwo(int a) 
{

  /* the number of shifts to do */
  int NumShifts = 8 * sizeof(int);

  /* count ones */
  int NumOnes = 0;
  for (int i = 0; i < NumShifts; i++, a = a >> 1) NumOnes += (a & 1);

  /* we can only have one one */
  return(NumOnes == 1);

}
/*************************************************************************/

/*************************************************************************/
static int MakePowTwo(int a) 
{

  /* the number of shifts to do */
  int NumShifts = 8 * sizeof(int);

  /* count ones */
  int LargestOne = 0;
  for (int i = 0; i < NumShifts; i++, a = a >> 1) if (a & 1) LargestOne = i;

  /* we can only have one one */
  return(1 << (LargestOne + 1));

}
/*************************************************************************/

/********************************************************************/
void SFImage::Resize()
{

  /* determine the new image size */
  int rw = width, rh = height;

  /* if already a power of two we are done */
  if (isPowTwo(width) == false) rw = MakePowTwo(width);
  if (isPowTwo(height) == false) rh = MakePowTwo(height);

#ifdef DEBUG_PREPARE
  cout << "\nEntered SFImage::Resize (" << rw << ", " << rh << ")\n";
#endif 

  /* the old dimensions */
  int nw = width;
  int nh = height;

  /* resize the image */
  width = rw;
  height = rh;
  numBands = 4;

  /* fun with pointers */
  unsigned char *old = data;
  data = (unsigned char *) 0;
  alloc(rw, rh);

  /* First we compute (cx,cy) = original pixel, and (px,py) = relative
  ** position of pixel ex,ey inside of cx,cy as percentages +-50%,
  ** +-50%. 0,0 = middle of pixel. We can save a lot of time by
  ** precomputing cxtab[] and pxtab[], both dwide arrays of ints that
  ** contain values for the equations: cx = (ex * nw) / rw; px = ((ex
  ** * nw * 100) / rw) - (cx * 100) - 50; This procedure is taken out
  ** of the xv source distribution developed and copyrighted by John
  ** Bradely. The code has been modified for the current problem. */
  int *cxtab = new int[rw];
  int *pxtab = new int[rw];
  if ((cxtab == NULL) || (pxtab == NULL)) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (SFImage::Resize)\n";
    cerr << "\tCould not allocate memory\n";
    exit(0);
  } else {
    for (int ex = 0; ex < rw; ex++) {
      if (rw == nw) cxtab[ex] = ex;
      else          cxtab[ex] = (ex * nw) / rw;
      pxtab[ex] = (((ex * nw) * 100) / rw) - (cxtab[ex] * 100) - 50;
    }
  }
  
  /* use the computed arrays to scale the image */
  for (int ey = 0; ey < rh; ey++) {
    int y1 = 0;
    int cy = (ey * nh) / rh;
    int py = (((ey * nh) * 100) / rh) - (cy * 100) - 50;
    if (py < 0) { 
      y1 = cy-1; 
      if (y1 < 0) y1 = 0;
    } else {
      y1 = cy+1; 
      if (y1 > (nh - 1)) y1 = nh - 1;
    }
    
    
    /* visit each pixel */
    for (int ex = 0; ex < rw; ex++) {
      
      /* adjust the temporary arrays */
      int x1 = 0;
      int cx = cxtab[ex];
      int px = pxtab[ex];      
      if (px < 0) {
	x1 = cx - 1;  
	if (x1 < 0) x1 = 0;
      } else { 
	x1 = cx + 1;
	if (x1 > nw-1) x1 = nw - 1;
      }
      
      /* corner pixel */
      int idx = y1 * nw * 4 + x1 * 4;
      unsigned char rA = old[idx + 0];
      unsigned char gA = old[idx + 1];
      unsigned char bA = old[idx + 2];
      unsigned char aA = old[idx + 3];
      
      /* up/down center pixel */
      idx = y1 * nw * 4 + cx * 4;
      unsigned char rB = old[idx + 0];
      unsigned char gB = old[idx + 1];
      unsigned char bB = old[idx + 2];
      unsigned char aB = old[idx + 3];
     
      /* left/right center pixel */
      idx = cy * nw  * 4 + x1 * 4;
      unsigned char rC = old[idx + 0];
      unsigned char gC = old[idx + 1];
      unsigned char bC = old[idx + 2];
      unsigned char aC = old[idx + 3];
      
      /* center pixel */
      idx = cy * nw * 4 + cx * 4;
      unsigned char rD = old[idx + 0];
      unsigned char gD = old[idx + 1]; 
      unsigned char bD = old[idx + 2];
      unsigned char aD = old[idx + 3];
      
      /* compute weighting factors */
      int apx = abs(px);
      int apy = abs(py);
      int pA = (apx * apy) / 100;
      int pB = (apy * (100 - apx)) / 100;
      int pC = (apx * (100 - apy)) / 100;
      int pD = 100 - (pA + pB + pC);
      
      /* store the new pixel values into the resized image */
      idx = ey * rw * 4 + ex * 4;
      data[idx + 0] = (unsigned char)(((pA*rA) + (pB*rB) + (pC*rC) + (pD*rD)) / 100);
      data[idx + 1] = (unsigned char)(((pA*gA) + (pB*gB) + (pC*gC) + (pD*gD)) / 100);
      data[idx + 2] = (unsigned char)(((pA*bA) + (pB*bB) + (pC*bC) + (pD*bD)) / 100);
      data[idx + 3] = (unsigned char)(((pA*aA) + (pB*aB) + (pC*aC) + (pD*aD)) / 100);
      if (data[idx + 3] < 128) data[idx + 3] = 0;
      else if (data[idx + 3] >= 128) data[idx + 3] = 255;
    }
  }
  
  /* free up local memory */
  delete[] cxtab;  
  delete[] pxtab;

  /* remove the old image */
  delete[] old;

#ifdef DEBUG_PREPARE
  cout << "\tExiting SFImage::Resize (" << rw << ", " << rh << ")\n";
#endif 
  
}
/********************************************************************/

/*************************************************************************/
void SFImage::LoadImage(char *FN)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered SFImage::LoadImage (" << FN << ")\n";
#endif 

  /* we are valid */
  isValid() = true;

  /* open the file read the magic number and close */
  ifstream InFile(FN);
  if (InFile.bad()) {
    cerr << "\nError:\n";
    cerr << "\tOccured in (SFImage::InputSFImage)\n";
    cerr << "\tCould not open input file (" << FN << ")\n";
    exit(0);
  } else {
    unsigned char MagNo[30];
    InFile.read(MagNo, sizeof(unsigned char) * 30);
    if (! InFile.good()) {
      cerr << "\nError:\n";
      cerr << "\tOccured in (SFImage::InputSFImage)\n";
      cerr << "\tCould not open input file (" << FN << ")\n";
      exit(0);
    } else {
      InFile.close();
      if ((strncmp((char *) MagNo,"GIF87a", 6)==0)||
	  (strncmp((char *) MagNo,"GIF89a",6)==0)) {
	LoadGIF(FN);

      } else if ((MagNo[0] == 0xff) && 
		 (MagNo[1] == 0xd8) && 
		 (MagNo[2] == 0xff)) {
	LoadJPG(FN);

      } else if (png_check_sig(MagNo, 4)) {
 	LoadPGN(FN);

      } else {	
        cerr << "\nError:\n";
	cerr << "\tOccured in (SFImage::InputSFImage)\n";
	cerr << "\tUnknown image read, only JPEG/GIF/PNG images read\n";
	exit(0);
      }
    }
  }

  /* if we do not have a power of two we must resize */
  if ((isPowTwo(width) == false) || (isPowTwo(height) == false)) {
    Resize();
  }

#ifdef DEBUG_PREPARE
  cout << "\tExiting SFImage::LoadImage (" << FN << ")\n";
#endif 

}
/*************************************************************************/

/*************************************************************************/
void SFImage::LoadPGN(char *FN)
{
  
  /* Alloc and check various png_structures */
  png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (png_ptr == NULL) return;
  
  /* create the info struct */
  png_infop info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == NULL)  {
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    return;
  }
  
  /* open a pointer to the file */
  FILE *fp = fopen(FN, "rb");
  if (fp == NULL) {
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    fclose(fp);
    return;
  }

  /* Set error handling if you are using the setjmp/longjmp method (this is
   * the normal method of doing things with libpng).  REQUIRED unless you
   * set up your own error handlers in the png_create_read_struct() earlier.
   */
  if (setjmp(png_ptr->jmpbuf))   {
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    fclose(fp);
    return;
  }
  
  /* associate the info pointer with the file */
  png_init_io(png_ptr, fp);
  
  /* Read header, filling image info into info_ptr */
  png_read_info(png_ptr, info_ptr);
  
  /* set up the header */
  png_uint_32 w, h;
  int bit_depth, color_type, interlace_type;
  png_get_IHDR(png_ptr, info_ptr, 
	       &w, &h, 
	       &bit_depth, &color_type,
	       &interlace_type, NULL, NULL);
  
  /* allocate the data */
  alloc(w, h);
  numBands = color_type;
  
  /* extract multiple pixels with bit depths of 1, 2, and 4 from a
   * single byte into separate bytes (useful for paletted and
   * grayscale images).  */
  png_set_packing(png_ptr);
  
  /* expand paletted colors into true RGB triplets */
  if (color_type == PNG_COLOR_TYPE_PALETTE) png_set_expand(png_ptr);

  /* expand grayscale images to the full 8 bits from 1, 2, or 4 bits/pixel */
  if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8) png_set_expand(png_ptr);
  
  /* expand paletted or RGB images with transparency to full alpha channels
   * so the data will be available as RGBA quartets */
  if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))  png_set_expand(png_ptr);

  /* read file to image */
  png_start_read_image(png_ptr);  
  
  /* the easiest way to read the image */
  int rowbytes = png_get_rowbytes(png_ptr, info_ptr);
  unsigned char **tmp = new unsigned char *[h];  
  for (int row = 0; row < int(h); row++) tmp[row] = new unsigned char[rowbytes * 4];
  png_read_image(png_ptr, tmp);

  /* copy the image into my struct */
  for(unsigned int rowCtr = 0; rowCtr < h; rowCtr++) {
    for (unsigned int colCtr = 0; colCtr < w; colCtr++) {
      if (numBands == 1) {
	pixel(w - colCtr - 1, rowCtr, 0) = tmp[rowCtr][colCtr];
	pixel(w - colCtr - 1, rowCtr, 1) = tmp[rowCtr][colCtr];
	pixel(w - colCtr - 1, rowCtr, 2) = tmp[rowCtr][colCtr];
	pixel(w - colCtr - 1, rowCtr, 3) = 255;
      } else if (numBands == 3) {
	pixel(w - colCtr - 1, rowCtr, 0) = tmp[rowCtr][colCtr * 3 + 0];
	pixel(w - colCtr - 1, rowCtr, 1) = tmp[rowCtr][colCtr * 3 + 1];
	pixel(w - colCtr - 1, rowCtr, 2) = tmp[rowCtr][colCtr * 3 + 2];
	pixel(w - colCtr - 1, rowCtr, 3) = 255;
      }
    }
  }
  
  /* cleanup */
  png_read_end(png_ptr, info_ptr);
  png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
  
}
/*************************************************************************/

/*************************************************************************/
void SFImage::LoadGIF(char *FN)
{
  
  /* Initialise GIF struct and read init block */
  GifByteType *Extension;
  GifFileType *GifFile = DGifOpenFileName(FN);
  if (GifFile == NULL) return;

  /* The way Interlaced image should. */
  int InterlacedOffset[] = {0, 4, 2, 1}, InterlacedJumps[] = {8, 8, 4, 2};
 
  /* allocate the data */
  alloc(GifFile->SWidth, GifFile->SHeight);
  numBands = 1;

  /* Index of the transparant colour, -1 if no transparent colour */
  int trans= -1, ExtCode;

  /* Allocate the screen as vector of column of rows. We cannt
  ** allocate the all screen at once, as this broken minded CPU can
  ** allocate up to 64k at a time and our image can be bigger than
  ** that: Note this screen is device independent - its the screen as
  ** defined by the GIF file parameters itself.  */
  GifRowType *ScreenBuffer = new GifRowType[GifFile->SHeight];
  if (ScreenBuffer == NULL) return;

  /* Bytes in one row */
  int Size = GifFile->SWidth * sizeof(GifPixelType);

  /* Allocate the other rows, and set their color to background too: */
  for (int i = 0; i < GifFile->SHeight; i++) {
    ScreenBuffer[i] = new unsigned char[Size];
    if (ScreenBuffer[i] == NULL) {
      return;
    } else {
      for (int j = 0; j < GifFile->SWidth; j++) {
	ScreenBuffer[i][j] = GifFile->SBackGroundColor;
      }
    }
  }

  /* Scan the content of the GIF file and load the image(s) in: */
  GifRecordType RecordType; 
  do {

    /* get the current record type */
    if (DGifGetRecordType(GifFile, &RecordType) == GIF_ERROR) {
      PrintGifError();
      return;
    }

    /* based on the record get a piece of data (scanline) */
    if (RecordType == IMAGE_DESC_RECORD_TYPE) {      
      if (DGifGetImageDesc(GifFile) == GIF_ERROR) {
	PrintGifError();
	return;
      }

      /* Image Position relative to Screen. */
      int Row = GifFile->Image.Top;
      int Col = GifFile->Image.Left;
      int Width = GifFile->Image.Width;
      int Height = GifFile->Image.Height;

      if ((Col + Width > GifFile->SWidth) || (Row + Height > GifFile->SHeight)) return;

      if (GifFile->Image.Interlace) {	
	/* Need to perform 4 passes on the images: */
	for (int i = 0; i < 4; i++) {
	  for (int j = Row + InterlacedOffset[i]; j < Row + Height; j += InterlacedJumps[i]) {
	    if (DGifGetLine(GifFile, &ScreenBuffer[j][Col], Width) == GIF_ERROR) return;
          }
	}
      } else {
	for (int i = 0; i < Height; i++) {
	  if (DGifGetLine(GifFile, &ScreenBuffer[Row++][Col], Width) == GIF_ERROR) {
	    PrintGifError();
	    return;
	  }
	}
      }
    } else if (RecordType == EXTENSION_RECORD_TYPE) {
      
      /* Skip any extension blocks in file: */
      if (DGifGetExtension(GifFile, &ExtCode, &Extension) == GIF_ERROR) return;
      if ((ExtCode == 249) && (Extension[1] & 1)) trans= Extension[4];
      do {
	if (DGifGetExtensionNext(GifFile, &Extension) == GIF_ERROR) return;
	if ((Extension != NULL) && 
	    (ExtCode == 249 ) &&
	    (Extension[1] & 1)) trans = Extension[4];
      } while  (Extension != NULL);
    }
  } while (RecordType != TERMINATE_RECORD_TYPE);

  /* Set QImage colour map */    
  ColorMapObject *cmap = (GifFile->Image.ColorMap
			  ? GifFile->Image.ColorMap
			  : GifFile->SColorMap);

  /* Copy image from the buffer */
  for (int j = 0; j < GifFile->SHeight; j++) {
    for (int i = 0; i < GifFile->SWidth; i++) {
      pixel(i, GifFile->SHeight - j - 1, 0) = cmap->Colors[ScreenBuffer[j][i]].Red;
      pixel(i, GifFile->SHeight - j - 1, 1) = cmap->Colors[ScreenBuffer[j][i]].Green;
      pixel(i, GifFile->SHeight - j - 1, 2) = cmap->Colors[ScreenBuffer[j][i]].Blue;
      pixel(i, GifFile->SHeight - j - 1, 3) = (ScreenBuffer[j][i] == trans) ? 0 : 255;
    }
  }

  /* only blend transparent gifs */
  if (trans != -1) blendingNeeded = true;

  /* close the file */
  DGifCloseFile(GifFile);

}
/*************************************************************************/

/*************************************************************************/
void SFImage::LoadJPG(char *FN)
{

  /* open the file */
  FILE *fp = fopen(FN, "r");
  if (fp == NULL) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (SFImage::LoadJPG\n";
    cerr << "\tCould not open file (" << FN << ")\n";
    exit(0);
  }

  /* set up the standard error handler */
  struct jpeg_decompress_struct    cinfo;
  struct jpeg_error_mgr            jerr;
  cinfo.err = jpeg_std_error(&jerr);

  /* start the decompression */
  jpeg_create_decompress(&cinfo);

  /* get the header info */
  jpeg_stdio_src(&cinfo, fp);
  jpeg_read_header(&cinfo, true);
  jpeg_calc_output_dimensions(&cinfo);

  /* allocate the data */
  alloc(cinfo.output_width, cinfo.output_height);
  numBands = cinfo.output_components;

  /* start the decompression */
  jpeg_start_decompress(&cinfo);

  /* a single scan line */
  unsigned char *tmp = new unsigned char[width * height * numBands];
  if (tmp == NULL) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (SFImage::InputJpg)\n";
    cerr << "\tCould not allocate data\n";
    exit(0);
  }

  /* read the scan lines */
  JSAMPROW rowptr[1];
  while (cinfo.output_scanline < cinfo.output_height) {
    rowptr[0] = (JSAMPROW) &tmp[(cinfo.output_height - cinfo.output_scanline - 1) *
			        cinfo.output_width * cinfo.output_components];
    if (jpeg_read_scanlines(&cinfo, rowptr, (JDIMENSION) 1) != 1) {
      cerr << "\nError:\n";
      cerr << "\tOccurred in (SFImage::InputJpg)\n";
      cerr << "\tIncorrect number of scan lines read\n";
      exit(0);
    }
  }

  /* Finish decompression and release memory.*/
  jpeg_finish_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);
  
  /* convert the scan lines */
  for (int j = 0; j < height; j++) {
    for (int i = 0; i < width; i++) {
      if (numBands == 1) {
	pixel(i, j, 0) = tmp[j * width + i];
	pixel(i, j, 1) = tmp[j * width + i];
	pixel(i, j, 2) = tmp[j * width + i];
	pixel(i, j, 3) = 255;
      } else if (numBands == 3) {
	pixel(i, j, 0) = tmp[3 * j * width + 3 * i + 0];
	pixel(i, j, 1) = tmp[3 * j * width + 3 * i + 1];
	pixel(i, j, 2) = tmp[3 * j * width + 3 * i + 2];
	pixel(i, j, 3) = 255;	
      }      
    }
  }

  /* remove the temporary array */
  delete[] tmp;  

  /* close up shop */
  fclose(fp);

}
/*************************************************************************/

/*************************************************************************/
void SFImage::prepare (SFVec3f &, SFVec3f &)
{


#ifdef DEBUG_PREPARE
  cout << "\nEntered SFImage::prepare (" << bool(blendingNeeded) << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* make sure we are prepared to see unsigned chars */
  glPixelStorei  (GL_UNPACK_ALIGNMENT, sizeof(unsigned char));

  /* enable alpha blending */
  if (bool(blendingNeeded)) glEnable(GL_ALPHA_TEST);

  /* store the image */
  glTexImage2D (GL_TEXTURE_2D, 
		0, 
		4,
		width, 
		height,
		0,
		GL_RGBA, 
		GL_UNSIGNED_BYTE, 
		data);

  /* apply the appropriate texture scaling */
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glHint         (GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
  if (bool(blendingNeeded)) {
    glTexEnvi      (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE,  GL_MODULATE);
  } else {
    glTexEnvi      (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE,  GL_DECAL);
  }

  /* enable texture mapping */
  glEnable(GL_TEXTURE_2D);

  /* finish up the list */
  glEndList();

#ifdef DEBUG_PREPARE
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone SFImage\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void SFImage::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered SFImage::render (" << bool(blendingNeeded) << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tImageDims(" << width << ", " << height << ", " << numBands << ")\n";
#endif 

  /* do nothing for the empty textures */
  if ((width <= 0) || (height <= 0)) return;

  /* call the render function */
  glCallList(CallNum);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone SFImage\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
static int CharToHex(char c) 
{

  /* determine the approriate value to return */
  if      (c == '0') return(0x0);
  else if (c == '1') return(0x1);
  else if (c == '2') return(0x2);
  else if (c == '3') return(0x3);
  else if (c == '4') return(0x4);
  else if (c == '5') return(0x5);
  else if (c == '6') return(0x6);
  else if (c == '7') return(0x7);
  else if (c == '8') return(0x8);
  else if (c == '9') return(0x9);
  else if ((c == 'a') || (c == 'A')) return(0xA);
  else if ((c == 'b') || (c == 'B')) return(0xB);
  else if ((c == 'c') || (c == 'C')) return(0xC);
  else if ((c == 'd') || (c == 'D')) return(0xD);
  else if ((c == 'e') || (c == 'E')) return(0xE);
  else if ((c == 'f') || (c == 'F')) return(0xF);
  
  /* the default */
  return(-1);

}
/*************************************************************************/

/*************************************************************************/
void SFImage::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* the image has a fixed format. width comes first */
  width.parse(header, InFile);

  /* followed by the height */
  height.parse(header, InFile);

  /* allocate the texture map */
  alloc(width, height);

  /* followed by the number of bands */
  numBands.parse(header, InFile);

#ifdef DEBUG_PARSE
  cout << "\tSFImage: (";
  cout << width << ", ";
  cout << height << ", ";
  cout << numBands << ")\n";
#endif

  /* a token in the file */
  SFToken Token;

  /* now read each and every pixel in the image */
  for (int i = 0; i < height; i++) {
    for (int j = 0; j < width; j++) {

      /* get the current pixel */
      Token.GetToken(InFile);
      
      /* verify that the number is in HEX */
      if (((Token()[0]) != '0') || (Token()[1]) != 'x') {
	cerr << "\nError:\n";
	cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
	cerr << "\tImage values must be hex (" << Token() << ").\n";
	exit(0);
      }
      
      /* we now need to decode this number */
      int r = 0, g = 0, b = 0, a = 255;
      if (strlen(Token()) == 4)  {
	b = (CharToHex(Token()[2]) << 4) | CharToHex(Token()[3]);

      } else if (strlen(Token()) == 6)  {
	g = (CharToHex(Token()[2]) << 4) | CharToHex(Token()[3]);
	b = (CharToHex(Token()[4]) << 4) | CharToHex(Token()[5]);

      } else if (strlen(Token()) == 8)  {
	r = (CharToHex(Token()[2]) << 4) | CharToHex(Token()[3]);
	g = (CharToHex(Token()[4]) << 4) | CharToHex(Token()[5]);
	b = (CharToHex(Token()[6]) << 4) | CharToHex(Token()[7]);

      } else if (strlen(Token()) == 10) {
	r = (CharToHex(Token()[2]) << 4) | CharToHex(Token()[3]);
	g = (CharToHex(Token()[4]) << 4) | CharToHex(Token()[5]);
	b = (CharToHex(Token()[6]) << 4) | CharToHex(Token()[7]);
	a = (CharToHex(Token()[8]) << 4) | CharToHex(Token()[9]);
	blendingNeeded = true;
      }

      /* use the decoded value to update the pixel */
      pixel(i, j, 0) = r;
      pixel(i, j, 1) = g;
      pixel(i, j, 2) = b;
      pixel(i, j, 3) = a;
      
    }
  }
}
/*************************************************************************/
