/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <PointSet.h>

/*************************************************************************/
void PointSet::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered PointSet::eventIn (" << DEF << ")\n";
#endif 

    /* Determine what action to take based on incoming event */
  if ((eventType == "coord") || (eventType == "set_coord")) {
    Coordinate *n = (Coordinate *) eventData;
    coord = *n;

  } else if ((eventType == "color") || (eventType == "set_color")) {
    Color *n = (Color *) eventData;
    color = *n;

  }

  /* re-generate the call list */
  SFVec3f min, max;
  prepare(min, max);

}
/*************************************************************************/

/*************************************************************************/
void PointSet::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered PointSet::prepare (" << DEF << ")\n";
#endif 

  /* verify that enough points exist */
  if (coord.num() == 0) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (PointSet::Render)\n";
    cerr << "\tEmpty coordinate lists found\n";
    exit(0);
  }

  /* get the modelview matrix */
  GLdouble m[16];
  glGetDoublev (GL_MODELVIEW_MATRIX,  m);

  /* determine the max and min */
  for (int i = 0; i < coord.num(); i++) {
    float ix = coord[i][0];
    float iy = coord[i][1];
    float iz = coord[i][2];
    float ox = (m[0] * ix) + (m[4] * iy) + (m[8]  * iz) + m[12];
    float oy = (m[1] * ix) + (m[5] * iy) + (m[9]  * iz) + m[13];
    float oz = (m[2] * ix) + (m[6] * iy) + (m[10] * iz) + m[14];
    if (min[0] > ox) min[0] = ox;
    if (min[1] > oy) min[1] = oy;
    if (min[2] > oz) min[2] = oz;
    if (max[0] < ox) max[0] = ox;
    if (max[1] < oy) max[1] = oy;
    if (max[2] < oz) max[2] = oz;
  }

  /* free the call list if one already exists */
  if (CallNum != 0) glDeleteLists(CallNum, 1);

  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* set up for drawing all of the points */
  glPointSize(2.0);
  glBegin(GL_POINTS);

  /* visit each coordinate */
  for (int i = 0; i < coord.num(); i++) {

    /* set the color */
    if (color.num() != 0) glColor3f(color[i][0], color[i][1], color[i][2]); 

    /* set the point */
    glVertex3f(coord[i][0], coord[i][1], coord[i][2]);

  }

  /* finish off the pointset */
  glEnd();

  /* finish up the list */
  glEndList();

}
/*************************************************************************/

/*************************************************************************/
void PointSet::render(SFRenderInfo &)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered PointSet::render (" << DEF << ")\n";
#endif 

  /* call the render function */
  glCallList(CallNum);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif
  
}
/*************************************************************************/

/*************************************************************************/
void PointSet::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  char TokenFound  = false;
  char BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tPointSet: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "coord") {
      coord.parse(header, InFile);

    } else if (Token == "color") {
      color.parse(header, InFile);

    } else if (Token == "PointSet") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/
