/*
 *   kwrl - a little VRML 2.0 viewer
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <IndexedFaceSet.h>

/*************************************************************************/
void IndexedFaceSet::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered IndexedFaceSet::prepare (" << DEF << ")\n";
#endif 

  /* verify that enough points exist */
  if ((! coordIndex.isValid()) || (! coord.isValid())) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (IndexedFaceSet::prepare)\n";
    cerr << "\tEmpty coordinate lists found\n";
    exit(0);
  }

  /* Count the number of polygons total */
  int localNumPolygons = 0;
  for (int i = 0; i < coordIndex.num(); i++) {
    if (coordIndex[i] == -1) localNumPolygons += 1;
  }

  /* compute the normals if necessary (may have been specifed by default) */
  if (normal.num() == 0) {
    
    /* allocate the normal node */
    normal.alloc(localNumPolygons);

    /* we are computing normals per face now */
    normalPerVertex = false;
    
    /* visit each face and compute the normal */
    for (int i = 0, f = 0; i < coordIndex.num(); i++, f++) {

      /* visit each point in the face and add contribution of normal */
      int beg = coordIndex[i];
      while (coordIndex[i] != -1) {

	/* the current and index */
	int idx = coordIndex[i];
	int jdx = coordIndex[i + 1];
	if (jdx == -1) jdx = beg;

	/* the components */
	float xi = coord[idx][0];
	float yi = coord[idx][1];
	float zi = coord[idx][2];
	float xj = coord[jdx][0];
	float yj = coord[jdx][1];
	float zj = coord[jdx][2];

	/* update the normal */
	normal[f][0] += (yi - yj) * (zi + zj);
	normal[f][1] += (zi - zj) * (xi + xj);
	normal[f][2] += (xi - xj) * (yi + yj);

	/* move to the next point */
	i += 1;

      }

      /* determine the length of the vector */
      normal[f].normalize();
    }
  }

  /* get the modelview matrix */
  GLdouble m[16];
  glGetDoublev (GL_MODELVIEW_MATRIX,  m);

  /* determine the max and min */
  for (int i = 0; i < coord.num(); i++) {
    float ix = coord[i][0];
    float iy = coord[i][1];
    float iz = coord[i][2];
    float ox = (m[0] * ix) + (m[4] * iy) + (m[8]  * iz) + m[12];
    float oy = (m[1] * ix) + (m[5] * iy) + (m[9]  * iz) + m[13];
    float oz = (m[2] * ix) + (m[6] * iy) + (m[10] * iz) + m[14];
    if (min[0] > ox) min[0] = ox;
    if (min[1] > oy) min[1] = oy;
    if (min[2] > oz) min[2] = oz;
    if (max[0] < ox) max[0] = ox;
    if (max[1] < oy) max[1] = oy;
    if (max[2] < oz) max[2] = oz;
  }

  /* free the call list if one already exists */
  if (CallNum != 0) glDeleteLists(CallNum, 1);

  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* set the attributes about vertex ordering */
  if (ccw) glFrontFace(GL_CCW);
  else     glFrontFace(GL_CW);

  /* set the attributes about solid */
  if (solid) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  else       glPolygonMode(GL_FRONT, GL_FILL);

  /* visit each index */
  for (int i = 0, f = 0; i < coordIndex.num(); f++, i++) {

    /* store the color if in color per face mode */
    if ((! colorPerVertex.isValid()) && (color.isValid())) {      
      if (colorIndex.num() == 0) {
	glColor3f(float(color[f][0]), 
		  float(color[f][1]),
		  float(color[f][2]));
      } else {
	glColor3f(float(color[colorIndex[f]][0]),
		  float(color[colorIndex[f]][1]), 
		  float(color[colorIndex[f]][2]));
      }
    }

    /* store the normal if in normal per face mode */
    if (normalPerVertex == false) {
      if (normalIndex.num() == 0) {
	glNormal3f(float(normal[f][0]), 
		   float(normal[f][1]), 
		   float(normal[f][2]));
      } else {
	int ndx = normalIndex[f];
	glNormal3f(float(normal[ndx][0]), 
		   float(normal[ndx][1]), 
		   float(normal[ndx][2]));
      }
    }

    /* start drawing the face */
    glBegin(GL_POLYGON);

    /* continue for each face */
    while (coordIndex[i] != -1) {
      
      /* the current index */
      int idx = coordIndex[i];
      
      /* store the color if in color per vertex mode */
      if ((colorPerVertex) && (color.isValid())) {
	if (colorIndex.num() == 0) {
	  glColor3f(float(color[i][0]), 
		    float(color[i][1]),
		    float(color[i][2]));
	} else {
	  glColor3f(float(color[colorIndex[i]][0]),
		    float(color[colorIndex[i]][1]), 
		    float(color[colorIndex[i]][2]));
	}
      }
      
      /* store the normal if in normal per vertex mode */
      if (normalPerVertex) {
	if (normalIndex.num() == 0) {
	  glNormal3f(float(normal[i][0]), 
		     float(normal[i][1]),
		     float(normal[i][2]));
	} else {
	  int ndx = normalIndex[i];
	  glNormal3f(float(normal[ndx][0]), 
		     float(normal[ndx][1]),
		     float(normal[ndx][2]));
	}
      }

      /* store the texture map if availiable */
      if ((texCoord.num() != 0) && (texCoordIndex.num() != 0)){
	int ndx = int(texCoordIndex[i]);
	glTexCoord2f(float(texCoord[ndx][0]), float(texCoord[ndx][1]));
      }

      /* add the vertex */
      glVertex3f(float(coord[idx][0]), 
		 float(coord[idx][1]),
		 float(coord[idx][2]));

      /* move to the next point */
      i += 1;
      
    }
    
    /* finish off the polygon */
    glEnd();
    
  }

  /* finish up the list */
  glEndList();

#ifdef DEBUG_PREPARE
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone IndexedFaceSet\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void IndexedFaceSet::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered IndexedFaceSet::render()\n";  
#endif

  /* call the render function */
  glCallList(CallNum);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone IndexedFaceSet\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void IndexedFaceSet::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  char TokenFound  = false;
  char BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tIndexedFaceSet: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "coord") {
      coord.parse(header, InFile);

    } else if (Token == "ccw") {
      ccw.parse(header, InFile);

    } else if (Token == "convex") {
      convex.parse(header, InFile);

    } else if (Token == "texCoord") {
      texCoord.parse(header, InFile);

    } else if (Token == "normal") {
      normal.parse(header, InFile);
      
    } else if (Token == "coordIndex") {
      coordIndex.parse(header, InFile);

    } else if (Token == "texCoordIndex") {
      texCoordIndex.parse(header, InFile);

    } else if (Token == "normalIndex") {
      normalIndex.parse(header, InFile);

    } else if (Token == "creaseAngle") {
      creaseAngle.parse(header, InFile);

    } else if (Token == "normalPerVertex") {
      normalPerVertex.parse(header, InFile);

    } else if (Token == "solid") {
      solid.parse(header, InFile);

    } else if (Token == "colorPerVertex") {
      colorPerVertex.parse(header, InFile);

    } else if (Token == "colorIndex") {
      colorIndex.parse(header, InFile);

    } else if (Token == "color") {
      color.parse(header, InFile);

    } else if (Token == "IndexedFaceSet") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse()\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse()\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }

}
/*************************************************************************/

/*************************************************************************/
void IndexedFaceSet::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered IndexedFaceSet::eventsIn (" << DEF << ")\n";
#endif 

  /* Determine what action to take based on incoming event */
  if        ((eventType == "coord")    || (eventType == "set_coord")) {
    Coordinate *n = (Coordinate *) eventData;
    coord = *n;

  } else if ((eventType == "texCoord")    || (eventType == "set_texCoord")) {
    TextureCoordinate *n = (TextureCoordinate *) eventData;
    texCoord = *n;

  } else if ((eventType == "color")    || (eventType == "set_color")) {
    Color *n = (Color *) eventData;
    color = *n;

  } else if ((eventType == "normal")    || (eventType == "set_normal")) {
    Normal *n = (Normal *) eventData;
    normal = *n;

  } else if (eventType == "set_colorIndex") {
    MFInt32 *n = (MFInt32 *) eventData;
    colorIndex = *n;

  } else if (eventType == "set_coordIndex") {
     MFInt32 *n = (MFInt32 *) eventData;
     coordIndex = *n;

  } else if (eventType == "normalIndex") {
    MFInt32 *n = (MFInt32 *) eventData;
    normalIndex = *n;

  } else if (eventType == "texCoordIndex") {
    MFInt32 *n = (MFInt32 *) eventData;
    texCoordIndex = *n;

  }

  /* re-generate the call list */
  SFVec3f min, max;
  prepare(min, max);

}
/*************************************************************************/
