/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <ElevationGrid.h>

/*************************************************************************/
void ElevationGrid::prepare(SFVec3f &, SFVec3f &)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered ElevationGrid::prepare (" << DEF << ")\n";
#endif 

  /* verify that enough points exist */
  if (height.num() == 0) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (ElevationGrid::prepare)\n";
    cerr << "\tEmpty coordinate lists found\n";
    exit(0);
  }
  
  /* compute the normals if necessary (may have been specifed by default) */
  if (normal.num() == 0) {
    
    /* allocate the normal node */
    normal.alloc(xDimension() * zDimension());

    /* we are computing normals per face now */
    normalPerVertex = false;

    /* visit each point and compute the normal as a cross product */
    for (int i = 0; i < (xDimension() - 1); i++) {
      for (int j = 0; j < (zDimension() - 1); j++) {

	/* get three points */
	/* p1 */
	float x1 = xSpacing * i;
	float y1 = height[i + xDimension() * j];
	float z1 = zSpacing * j;
	
	/* p2 */
	float x2 = xSpacing * (i + 1);
	float y2 = height[(i + 1) + xDimension * j];
	float z2 = zSpacing * j;
	
	/* p3 */
	float x3 = xSpacing * (i + 1);
	float y3 = height[(i + 1) + xDimension * (j + 1)];
	float z3 = zSpacing * (j + 1);
	
	/* form two vectos */
	float v1x = x3 - x2;
	float v1y = y3 - y2;
	float v1z = z3 - z2;
	float v2x = x1 - x2;
	float v2y = y1 - y2;
	float v2z = z1 - z2;
	
	/* cross them */
	float nx = v1y * v2z - v1z * v2y;
	float ny = v1z * v2x - v1x * v2z;
	float nz = v1x * v2y - v1y * v2x;
	
	/* store the new values */
	normal[i + xDimension * j][0] = nx;
	normal[i + xDimension * j][1] = ny;
	normal[i + xDimension * j][2] = nz;
	
	/* normalize */
	normal[i + xDimension * j].normalize();
	
      }
    }
  }

  /* free the call list if one already exists */
  if (CallNum != 0) glDeleteLists(CallNum, 1);

  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* Store the matrix */
  glPushMatrix();

  /* determine the max and min y for centering */
  float min = 10000, max = -10000;
  for (int i = 0; i < (xDimension - 1); i++) {
    for (int j = 0; j < (zDimension - 1); j++) {
      float y1 = height[i + xDimension * j];
      if (y1 > max) max = y1;
      if (y1 < min) min = y1;
    }
  }

  /* generate about the origin */
  glTranslated(-(xSpacing * xDimension) / 2.0,
	       (max - min) / 2.0,
	       -(zSpacing * zDimension) / 2.0);


  /* visit all of the points */
  for (int i = 0; i < (xDimension - 1); i++) {
    glBegin(GL_TRIANGLE_STRIP);
    for (int j = 0; j < (zDimension - 1); j++) {

      /* p1 */
      float x1 = xSpacing * i;
      float y1 = height[i + xDimension * j];
      float z1 = zSpacing * j;

      /* p2 */
      float x2 = xSpacing * (i + 1);
      float y2 = height[(i + 1) + xDimension * j];
      float z2 = zSpacing * j;

      /* store the normals */
      if (normalPerVertex) {
	glNormal3f(normal[(i + 1) + xDimension * j][0],
		   normal[(i + 1) + xDimension * j][1],
		   normal[(i + 1) + xDimension * j][2]);
      } else {
	glNormal3f(normal[i + xDimension * j][0],
		   normal[i + xDimension * j][1],
		   normal[i + xDimension * j][2]);
      }

      /* store the texture map if availiable */
      if (texCoord.num() == 0) {
	glTexCoord2f(float(i + 1) / xDimension, float(j) / zDimension);
      } else {
	glTexCoord2f(texCoord[i + 1][0], texCoord[i + 1][1]);
      }

      /* draw */
      glVertex3f(x2, y2, z2);

      /* store the normals */
      glNormal3f(normal[i + xDimension * j][0],
		 normal[i + xDimension * j][1],
		 normal[i + xDimension * j][2]);

      /* store the texture map if availiable */
      if (texCoord.num() == 0) {
	glTexCoord2f(float(i) / float(xDimension), float(j) / float(zDimension));
      } else {
	glTexCoord2f(texCoord[i][0], texCoord[i][1]);
      }

      /* draw again */
      glVertex3f(x1, y1, z1);

    }

    /* finish up drawing */
    glEnd();

  }

  /* Restore the matrix */
  glPopMatrix();

  /* finish up the list */
  glEndList();

}
/*************************************************************************/

/*************************************************************************/
void ElevationGrid::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered ElevationGrid::render()\n";  
#endif

  /* call the render function */
  glCallList(CallNum);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void ElevationGrid::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered ElevationGrid::eventsIn (" << DEF << ")\n";
#endif 

  /* Determine what action to take based on incoming event */
  if ((eventType == "color") || (eventType == "set_color")) {
    Color *n = (Color *) eventData;
    color = *n;

  } else if ((eventType == "normal") || (eventType == "set_normal")) {
    Normal *n = (Normal *) eventData;
    normal = *n;

  } else if ((eventType == "texCoord") || (eventType == "set_texCoord")) {
    TextureCoordinate *n = (TextureCoordinate *) eventData;
    texCoord = *n;

  } else if ((eventType == "height") || (eventType == "set_height")) {
    MFFloat *n = (MFFloat *) eventData;
    height = *n;

  }

  /* re-generate the call list */
  SFVec3f min, max;
  prepare(min, max);

}
/*************************************************************************/

/*************************************************************************/
void ElevationGrid::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  char TokenFound  = false;
  char BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tElevationGrid: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "height") {
      height.parse(header, InFile);

    } else if (Token == "texCoord") {
      texCoord.parse(header, InFile);

    } else if (Token == "normal") {
      normal.parse(header, InFile);
      
    } else if (Token == "creaseAngle") {
      creaseAngle.parse(header, InFile);

    } else if (Token == "normalPerVertex") {
      normalPerVertex.parse(header, InFile);

    } else if (Token == "solid") {
      solid.parse(header, InFile);

    } else if (Token == "ccw") {
      ccw.parse(header, InFile);

    } else if (Token == "xDimension") {
      xDimension.parse(header, InFile);

    } else if (Token == "zDimension") {
      zDimension.parse(header, InFile);

    } else if (Token == "xSpacing") {
      xSpacing.parse(header, InFile);

    } else if (Token == "zSpacing") {
      zSpacing.parse(header, InFile);

    } else if (Token == "creaseAngle") {
      creaseAngle.parse(header, InFile);

    } else if (Token == "colorPerVertex") {
      colorPerVertex.parse(header, InFile);

    } else if (Token == "color") {
      color.parse(header, InFile);

    } else if (Token == "ElevationGrid") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/

