/*
 *   kwrl - a little VRML 2.0 viewer
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <SFRenderInfo.h>
#include <Children.h>

/* for parsing all nodes */
#include <Anchor.h>
#include <Billboard.h>
#include <Collision.h>
#include <Group.h>
#include <TimeSensor.h>
#include <TouchSensor.h>
#include <VisibilitySensor.h>
#include <ProximitySensor.h>
#include <Background.h>
#include <Fog.h>
#include <Viewpoint.h>
#include <NormalInterpolator.h>
#include <PositionInterpolator.h>
#include <OrientationInterpolator.h>
#include <NavigationInfo.h>
#include <PointLight.h>
#include <PlaneSensor.h>
#include <SpotLight.h>
#include <DirectionalLight.h>
#include <SFEvent.h>
#include <Shape.h>
#include <Sound.h>
#include <Script.h>
#include <WorldInfo.h>
#include <Transform.h>
#include <Inline.h>
#include <Switch.h>

/*************************************************************************/
SFNode *Children::operator [] (int i) 
{

  /* process the other children - if present */
  int n = 0;
  for (Children *C = this; C != (Children *) 0; C = C->next()) {
    if ((C->data() != (SFNode *) 0) && (n == i)) {
      return(C->data());
    } else {
      n += 1;
    }
  }

  /* return nothing */
  return ((SFNode *) 0);

}
/*************************************************************************/

/*************************************************************************/
void Children::push(Children &stack, SFString &stackType)
{

  /* Visit the children */
  for (Children *C = this; C != (Children *) 0; C = C->next()) {
    if (C->data() != (SFNode *) 0) C->data()->push(stack, stackType);
  }
}
/*************************************************************************/

/*************************************************************************/
SFNode *Children::add(SFNode *M) 
{

  /* this node may need a child */
  if (data() == (SFNode *) 0) {
    data() = M;
    return(M);
  }

  /* add to the list if we are at the end */
  if (next() == (Children *) 0) {
    next() =  new Children;
    next()->data() = M;
    return(M);
  }

  /* add to the end of the list */
  return(next()->add(M));

}
/*************************************************************************/

/*************************************************************************/
void Children::remove(SFNode *M) 
{

  /* this node may be it */
  if (data() == M) {
    data() = (SFNode *) 0;
    return;
  }

  /* remove from the list */
  if (next() != (SFNode *) 0) next()->remove(M);

}
/*************************************************************************/

/*************************************************************************/
void Children::prepare(SFVec3f &min, SFVec3f &max) 
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Children::prepare (" << DEF << ")\n";
#endif 

  /* process the other children - if present */
  for (Children *C = this; C != (Children *) 0; C = C->next()) {
    if (C->data() != (SFNode *) 0) C->data()->prepare(min, max);
  }

}
/*************************************************************************/

/*************************************************************************/
void Children::render(SFRenderInfo &SFRI)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Children::render (" << DEF << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif 

  /* save the state before rendering */
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  
  /* process the other children - if present */
  for (Children *C = this; C != (Children *) 0; C = C->next()) {
    if (C->data() != (SFNode *) 0) C->data()->render(SFRI);
  }

  /* save the state before rendering */
  glPopAttrib();

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone Children\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
SFNode *Children::findDEF(SFString &UnsatUSE)
{

  /* check against ourselves first */
  if (UnsatUSE == DEF) return(this);

  /* The value found by searching the list */
  SFNode *M = (SFNode *) 0;

  /* process the other children - if present */
  for (Children *C = this; C != (Children *) 0; C = C->next()) {
    if (C->data() != (SFNode *) 0) {
      M = C->data()->findDEF(UnsatUSE);
      if (M != (SFNode *) 0) return(M);
    }
  }

  /* check the children */
  return(M);

}
/*************************************************************************/

/*************************************************************************/
SFNode *Children::findUSE()
{
  
#ifdef DEBUG_USEDEF
    cout << "\nEntered: Children::findUSE\n";
#endif

  /* check this node */
  if (unsatisfiedUSE()) return(this);

  /* The value found by searching the list */
  SFNode *M = (SFNode *) 0;

  /* process the other children - if present */
  for (Children *C = this; C != (Children *) 0; C = C->next()) {
    if (C->data() != (SFNode *) 0) {
      M = C->data()->findUSE();
      if (M != (SFNode *) 0) return(M);
    }
  }

  /* check the children */
  return(M);

}
/*************************************************************************/

/*************************************************************************/
void Children::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  char BeginBracketFound   = false;

  /* the current def */
  char DefFound = false;
  
  /* used to start over if a def found */
  streampos def = 0;
  
  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    streampos p = Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tChildren: (" << Token() << ")\n";
#endif

    /* if a definition was found move to back to that point */
    if (DefFound) {
      p = def;
      DefFound = false;
    }  
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      DefFound = true;
      def      = p;
      Token.GetToken(InFile);

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      if (BeginBracketFound == false) return;

    } else if (Token == "Anchor") {
      add(new Anchor)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Billboard") {
      add(new Billboard)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Collision") {
      add(new Collision)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "DirectionalLight") {
      add(new DirectionalLight)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Group") {
      add(new Group)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Fog") {
      add(new Fog)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Inline") {
      add(new Inline)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "NavigationInfo") {
      add(new NavigationInfo)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "PointLight") {
      add(new PointLight)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Shape") {
      add(new Shape)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "SpotLight") {
      add(new SpotLight)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Switch") {
      add(new Switch)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Transform") {
      add(new Transform)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Viewpoint") {
      add(new Viewpoint)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "WorldInfo") {
      add(new WorldInfo)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Script") {
      add(new Script)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "TimeSensor") {
      add(new TimeSensor)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "TouchSensor") {
      add(new TouchSensor)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "VisibilitySensor") {
      add(new VisibilitySensor)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "ProximitySensor") {
      add(new ProximitySensor)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Sound") {
      add(new Sound)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "Background") {
      add(new Background)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "NormalInterpolator") {
      add(new NormalInterpolator)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "OrientationInterpolator") {
      add(new OrientationInterpolator)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "PositionInterpolator") {
      add(new PositionInterpolator)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "PlaneSensor") {
      add(new PlaneSensor)->parse(header, InFile.seekg(p));
      if (BeginBracketFound == false) return;

    } else if (Token == "ROUTE") {
      add(new SFEvent)->parse(header, InFile);
      if (BeginBracketFound == false) return;

    } else if (Token == ",") {

    } else if (Token == "") {

    } else if (Token == "[") {
      BeginBracketFound = true;

    } else if (Token == "]") {
      break;

    } else {
      parseWarning(Token());
      
    } 
  }
}
/*************************************************************************/










