/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** spinbox.cpp
*/
#include "spinbox.moc"
#include "spinbox.h"
#include "misc.h"

#include <stdio.h>
#include <stdlib.h>

#include <qsize.h>

SpinBox::SpinBox(QWidget *parent,const char *name, double v) :
	QWidget(parent,name)
{
	char	buffer[256];
	
	value = v + 1;
	value = value - 1;
	step = 0.01;
	origStep=0.01;
	stepcount = 0;
	tab = 0;
	sprintf(buffer,"%g",value);

	up = new QPushButton(this);
	up->setAutoRepeat(true);
	up->setText("+");
	down = new QPushButton(this);
	down->setAutoRepeat(true);
	down->setText("-");
	edit = new QLineEdit(this);

	qdv = new QDoubleValidator(this);	
	qiv = new QIntValidator(this);	

	qdv->setRange(-10E6,10E6,6);
	qiv->setRange(-10E6,10E6);

	edit->setValidator(qdv);
	edit->setText(buffer);

	topV = 1E300;
	downV = -1E300;
	
	connect(up,SIGNAL(clicked()),this,SLOT(upClicked()) );
	connect(down,SIGNAL(clicked()),this,SLOT(downClicked()) );
	connect(down,SIGNAL(released()),this,SLOT(released()) );
	connect(up,SIGNAL(released()),this,SLOT(released()) );
	connect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char *)) );

	label = new QLabel(this);
	label->setText(name);
	
	layout();
}

SpinBox::~SpinBox()
{
}


void	SpinBox::switchToIntMode()
{
	step=(double)((int)step);
	origStep=(double)((int)origStep);
	edit->setValidator(qiv);
}

void	SpinBox::switchToDoubleMode()
{
	edit->setValidator(qdv);
}

void SpinBox::setStep(int x)
{
	origStep=(double)x;
	step=(double)x;
}

void SpinBox::setStep(double x)
{
	origStep=x;
	step=x;
}

double SpinBox::getStep()
{
	return step;
}


int SpinBox::getLabelLength()
{
int l=0;
	if (label) l=label->sizeHint().width()+5;
	return l;
}


int SpinBox::layout(SpinBox *s1,SpinBox *s2,SpinBox *s3,SpinBox *s4,SpinBox *s5)
{
int lmax=0;
	if (s1) lmax=MAX(lmax,s1->getLabelLength() );
	if (s2) lmax=MAX(lmax,s2->getLabelLength() );
	if (s3) lmax=MAX(lmax,s3->getLabelLength() );
	if (s4) lmax=MAX(lmax,s4->getLabelLength() );
	if (s5) lmax=MAX(lmax,s5->getLabelLength() );

	if (s1) s1->setTab(lmax);
	if (s2) s2->setTab(lmax);
	if (s3) s3->setTab(lmax);
	if (s4) s4->setTab(lmax);
	if (s5) s5->setTab(lmax);

	return lmax;
}

void SpinBox::setTab(int t)
{
	tab=t;
}

void SpinBox::setLabel(char *n)
{
	label->setText(n);
}

void SpinBox::newText(const char *n)
{
	value = strtod(edit->text(),0);
	
	emit changed();
	emit valueChanged();
}

void SpinBox::released()
{
	if(up->isDown()||down->isDown())
		return;
		
	step = origStep;
	stepcount = 0;
}

void SpinBox::downClicked()
{
	char	buffer[256];
	
	disconnect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char *)) );
	value -= step;
	if(value <= downV)
		value = downV;
	stepcount++;
	if(stepcount % 10 == 0 && stepcount < 30)
	{
		step = step * 10;
	}
	
	if(fabs(value) < 0.0000001)
		sprintf(buffer,"0");
	else
		sprintf(buffer,"%g",value);
		
	edit->setText(buffer);
	//emit changed();
	emit valueChanged();
	connect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char*)));
}


void SpinBox::upClicked()
{
	char	buffer[256];
	
	disconnect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char*)));
	value += step;
	if(value >= topV)
		value = topV;
	stepcount++;
	if(stepcount % 10 == 0 && stepcount < 30)
	{
		step = step * 10;
	}
	
	if(fabs(value) < 0.0000001)
		sprintf(buffer,"0");
	else
		sprintf(buffer,"%g",value);
		
	edit->setText(buffer);
	//emit changed();
	emit valueChanged();
	connect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char *)) );
}

QSize SpinBox::sizeHint()
{
	QSize	qs;
	int	w;
	int	h;
	
	if(tab != 0)
		w = tab;
	else
		w = label->sizeHint().width();
		
 	h = MAX(26,edit->sizeHint().height()) + 2;
       	w = w + 60 + h / 2 + 10;	
	
	qs.setWidth(w);
	qs.setHeight(h);
	
	return qs;
}

void	SpinBox::layout()
{
	int	x = 0;
	int	y = 0;
	int	w,ww = width();
	int	h,wh = height();
	
	if(tab != 0)
		w = tab;
	else
		w = label->sizeHint().width();
	
	h = wh;
		
	label->setGeometry(x,y,w,h);
	
	x += w + 5;
	w = ww - w - h / 2 - 10;
	edit->setGeometry(x,y,w,h - 5);		

       	x += w + 5;
 	h = wh - 5;

	up->  setGeometry(x,y        ,h / 2 - 1,h / 2 - 1);
	down->setGeometry(x,y + h / 2,h / 2 - 1,h / 2 - 1);
}

void SpinBox::paintEvent(QPaintEvent*)
{
	layout();
}



void SpinBox::setValue(double v)
{
	char	buffer[256];
	
	disconnect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char *)) );
	value=v;
	sprintf(buffer,"%g",value);
	edit->setText(buffer);
	connect(edit,SIGNAL(textChanged(const char*)),this,SLOT(newText(const char *)) );
}


double SpinBox::getValue()
{
	return (double)value;
}

void SpinBox::setEnabled(bool e)
{
	up->setEnabled(e);
	down->setEnabled(e);
	edit->setEnabled(e);
	label->setEnabled(e);
}

void SpinBox::DownTopValue(double a, double b)
{
	topV=b;
	downV=a;	
}
