/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** graph-0.cpp
*/

#include "graph.h"
#include <stdio.h>

knode::knode(graph *g,Vector3 v) :
	Vector3(v)
{
	edgelist.empty();
	edgelist.setUnique(1);

	flags = 0;

	if((parent = g))
		parent->add(this);
	
//	printf("created knode <%g,%g,%g>\n",elements[0],elements[1],elements[2]);
}

knode::knode(graph *g,knode *k) :
	Vector3(*k)
{
	edgelist.empty();

	flags = k->flags;

	if((parent = g))
		parent->add(this);
	
//	printf("created knode <%g,%g,%g>\n",elements[0],elements[1],elements[2]);
}

knode::~knode()
{
	if(parent)
		parent->removeFromList(this);

	printf("deleted knode <%g,%g,%g>\n",elements[0],elements[1],elements[2]);
}

const char	*knode::id()
{
	static char	idstr[50];

	sprintf(idstr,"<%g,%g,%g>",elements[0],elements[1],elements[2]);

	return idstr;
}

void	knode::clearFlags(int f)
{
	flags = f;
}

void	knode::add(edge *e)
{
	edgelist += e;
}

edge	*knode::getEdge(int i)
{
	if(i >= edgelist.length())
		return 0;

	return edgelist[i];
}

int	knode::numEdges()
{
	return edgelist.length();
}

void	knode::removeEdge(edge *t)
{
	int	i;

	i = edgelist.find(t);
	if(i > 0)
	{
		edgelist.deleteAt(i);
	}
}

int	knode::deleteGraph()
{
	int	i,c;

//	printf("asked to delete knode %i\n",id());

	if(flags == 1) return 0;
	flags = 1;
	i = c = 0;
	while(numEdges() > 1 && c < 2)
	{
//		printf("knode %i ask to delete edge %i\n",id(),edgelist[i]->id());
		if(edgelist[i]->deleteGraph() != 0) return -i;
		i = (i + 1) % numEdges();
		if(i == 0) c++;
	}

	delete this;

	return 0;
}

void	knode::replace(knode *k)
{
	int	i;

	for(i = 0;i < k->numEdges();i++)
	{
		k->getEdge(i)->replace(k,this);
	}
}

int	knode::connected(knode *k)
{
	int	i;

	for(i = 0;i < k->numEdges();i++)
	{
		if(this == k->getEdge(i)->knode1())
			return 1;
		if(this == k->getEdge(i)->knode1())
			return 1;
	}

	return 0;
}

#define	MIN(a,b)	(a < b?a:b)
#define	MAX(a,b)	(a > b?a:b)

int	knode::distance(knode *k)
{
	int	i,d;


	if(this == k)
		return 0;

	if(flags == 2)
		return 2000000000;

	flags = 2;

	d = 2000000000;

	for(i = 0;i < edgelist.length();i++)
	{
		if(edgelist[i]->knode1() != this)
			d = MIN(d,edgelist[i]->knode1()->distance(k) + 1);
		if(edgelist[i]->knode2() != this)
			d = MIN(d,edgelist[i]->knode2()->distance(k) + 1);
	}

//	printf("knode %i distance to knode %i is %i\n",id(),k->id(),d);

	return MIN(d,2000000000);	
}

edge::edge(graph *g,knode *kp1,knode *kp2,int id)
{
	k1 = kp1;
	k2 = kp2;
	flags = 0;
	num_id = id;

	if(k1) k1->add(this);
	if(k2) k2->add(this);

	if((parent = g))
		parent->add(this);

	printf("created edge %i between ",num_id);
	printf("knode %s and ",k1->id());
	printf("knode %s\n",k2->id());
}

edge::~edge()
{
	k1->removeEdge(this);
	k2->removeEdge(this);

	if(parent)
		parent->removeFromList(this);

//	printf("deleted edge %i\n",num_id);
}

knode	*edge::knode1()
{
	return k1;
}

knode	*edge::knode2()
{
	return k2;
}

void	edge::replace(knode *kold,knode *knew)
{
	if(k1 == kold)
	{
		k1 = knew;
		k1->removeEdge(this);
	}
	if(k2 == kold)
	{
		k2 = knew;
		k2->removeEdge(this);
	}
}

int	edge::deleteGraph()
{
//	printf("asked to delete edge %i\n",id());

	if(flags == 1) return 0;
	flags = 1;

	if(k1->deleteGraph() != 0) return -1;
	if(k2->deleteGraph() != 0) return -2;

	delete this;

	return 0;
}

int	edge::id()
{
	return num_id;
}

// return 0 if is close, else error
int	edge::closed()
{
	// the edge is close, if the edge is part of exactly two triangles
	// if the graph is small (less than 14 knodes) the edges are not
	// close if the number of triangles is less than two
	int		i,t,n;

	// count the triangles

	n = 0;
	for(i = 0;i < k2->numEdges();i++)
	{
		if(this == k2->getEdge(i))
			continue;
		for(t = 0;t < k2->getEdge(i)->knode1()->numEdges();t++)
		{
			if(this == k2->getEdge(i)->knode1()->getEdge(t))
				continue;
			if(k1 == k2->getEdge(i)->knode1()->getEdge(t)->knode1())
				n++;
			if(k1 == k2->getEdge(i)->knode1()->getEdge(t)->knode2())
				n++;
		}
		for(t = 0;t < k2->getEdge(i)->knode2()->numEdges();t++)
		{
			if(this == k2->getEdge(i)->knode2()->getEdge(t))
				continue;
			if(k1 == k2->getEdge(i)->knode2()->getEdge(t)->knode1())
				n++;
			if(k1 == k2->getEdge(i)->knode2()->getEdge(t)->knode2())
				n++;
		}
	}

	return (n - 2);
}

int	edge::unique()
{
	int	i;

	for(i = 0;i < k1->numEdges();i++)
	{
		if(this != k1->getEdge(i))
		{
			if(k1 == k1->getEdge(i)->knode1() && k2 == k1->getEdge(i)->knode2())
				return 0;
			if(k2 == k1->getEdge(i)->knode1() && k1 == k1->getEdge(i)->knode2())
				return 0;
		}
	}
	for(i = 0;i < k2->numEdges();i++)
	{
		if(this != k2->getEdge(i))
		{
			if(k1 == k2->getEdge(i)->knode1() && k2 == k2->getEdge(i)->knode2())
				return 0;
			if(k2 == k2->getEdge(i)->knode1() && k1 == k2->getEdge(i)->knode2())
				return 0;
		}
	}

	return 1;
}

edge	*edge::swap()
{
	knode*	k;

	k = k1;
	k1 = k2;
	k2 = k;

	return this;
}

graph::graph()
{
	knodelist.setUnique(1);
	edgelist.setUnique(1);
}

graph::~graph()
{
	int		i;
/*
	list<knode*>	segments;

	segments += knodelist[0];

	for(i = 0;i < knodelist.length();i++)
	{
		knodelist[i]->clearFlags();
		e = 0;
		for(t = 0;t < segments.length();t++)
		{
			if(distance(segments[t],knodelist[i]) != -1)
				e = 1;
		}
		if(e)
			segments += knodelist[i];
	}

	for(i = 0;i < segments.length();i++)
	{
		segments[i]->deleteGraph();
	}
*/
	for(i = 0;i < knodelist.length();i++)
	{
		delete knodelist[0];
		knodelist.deleteAt(0);
	}
	for(i = 0;i < edgelist.length();i++)
	{
		delete edgelist[0];
		edgelist.deleteAt(0);
	}

}

int	graph::add(knode *k)
{
	int		i;

	if(!k)
		return 0;
	if(knodelist.find(k) >= 0)
		return 0;

	if(k->parent != this)
	{
		knodelist += new knode(this,k);
	}
	else
	{
		knodelist += k;

		for(i = 0;i < k->numEdges();i++)
		{
			add(k->getEdge(i));
		}
	}

	return 0;
}

int	graph::add(edge *e)
{
	if(!e)
		return 0;
	if(edgelist.find(e) >= 0)
		return 0;

	if(e->parent != this)
	{
		edge	*t;

		edgelist += t = new edge(this,
					new knode(this,e->knode1()),
					new knode(this,e->knode2()),
					e->id());

		add(t->knode1());
		add(t->knode2());
	}
	else
	{
		edgelist += e;

		add(e->knode1());
		add(e->knode2());
	}


	return 0;
}

int	graph::remove(knode *k)
{
	int		i;

	for(i = 0;i < k->numEdges();i++)
	{
		remove(k->getEdge(i));
	}

	delete k;

	return 0;
}

int	graph::remove(edge *e)
{
	delete e;

	return 0;
}

int	graph::removeFromList(knode *k)
{
	return knodelist.deleteAt(knodelist.find(k));
}

int	graph::removeFromList(edge *e)
{
	return edgelist.deleteAt(edgelist.find(e));
}

int	graph::numKnodes()
{
	return knodelist.length();
}

int	graph::numEdges()
{
	return edgelist.length();
}

knode	*graph::getKnode(int i)
{
	return knodelist[i];
}

edge	*graph::getEdge(int i)
{
	return edgelist[i];
}

int	graph::distance(knode *k1,knode *k2)
{
	int		i,r1,r2;

	for(i = 0;i < knodelist.length();i++)
	{
		knodelist[i]->clearFlags();
	}

	r1 = k1->distance(k2);

	for(i = 0;i < knodelist.length();i++)
	{
		knodelist[i]->clearFlags();
	}

	r2 = k2->distance(k1);

	i = MIN(r1,r2);

//	printf("distance between %i %i %i\n",k1->id(),k2->id(),i);

	if(i == 2000000000)
		return -1;

	return i;
}

int	graph::isClose(list<edge*> *l)
{
	int		i,n;

	if(l)
		l->empty();

	for(i = 0;i < edgelist.length();i++)
	{
		if((n = edgelist[i]->closed()) < 0)
		{
			if(l)
				l->append(edgelist[i]);
			else
				return 0;
		}
//		printf("Edge %i is %s close (%i)\n",edgelist[i]->id(),(n?"not":""),n);
	}

	if(l)
		return l->isEmpty();
	else
		return 1;
}

list<edge*>	graph::getOpenPolygone(list<edge*> ncelist)
{
// 1. take the first edge
// 2. take the second edge
// 3. take the next edge closest in the plane discribed by 1. + 2.
// 4. repeate 3. until the polygone is close or there is no match
// 5. is the polygone not close return an empty one

	list<edge*>	rlist;
	int		i,t;
	edge		*best;

	// 1. take the first edge
	rlist += ncelist[0];

	// 2. take the second edge
	for(i = 1;i < ncelist.length();i++)
	{
		if(rlist[0]->knode2() == ncelist[i]->knode1())
		{
			rlist += ncelist[i];
			break;
		}
		if(rlist[0]->knode2() == ncelist[i]->knode2())
		{
			rlist += ncelist[i]->swap();
			break;
		}
	}

	// 4. repeat 3. until the polygone is close or there is no match
	for(t = 0;t < ncelist.length() && rlist.getLast()->knode2() != rlist[0]->knode1();t++)
	{
		best = 0;
		// 3. take the next edge closest in the plane discribed by 1. + 2.
		for(i = 1;i < ncelist.length();i++)
		{
			// is the edge in rlist you cannot insert it again
			if(rlist.find(ncelist[i]) >= 0)
				continue;
			if(rlist.getLast()->knode2() == ncelist[i]->knode1())
			{
				best = ncelist[i];
				break;
			}
			if(rlist.getLast()->knode2() == ncelist[i]->knode2())
			{
				best += ncelist[i]->swap();
				break;
			}
			
		}
		for(;i < ncelist.length();i++)
		{
			// is the edge in rlist you cannot insert it again
			if(rlist.find(ncelist[i]) >= 0)
				continue;
			if(rlist.getLast()->knode2() == ncelist[i]->knode1())
			{
				double	d1,d2;

				d1 = best->distance(rlist[0]->knode1(),rlist[0]->knode2(),rlist[1]->knode2();
				d2 = ncelist[i]->knode2()->distance(rlist[0]->knode1(),rlist[0]->knode2(),rlist[1]->knode2();
				if(d2 < d1)
					 best = ncelist[i];
			}
			if(rlist.getLast()->knode2() == ncelist[i]->knode2())
			{
				double	d1,d2;

				d1 = best->distance(rlist[0]->knode1(),rlist[0]->knode2(),rlist[1]->knode2();
				d2 = ncelist[i]->knode1()->distance(rlist[0]->knode1(),rlist[0]->knode2(),rlist[1]->knode2();
				if(d2 < d1)
					 best = ncelist[i]->swap();
			}
		}
		if(!best)
			break;
		rlist +=  best;
	}

	// 5. is the polygone not close return an empty one
	if(rlist.getLast()->knode2() != rlist[0]->knode1())
		rlist.empty();

	return rlist;
}

int	graph::close()
{
	list<edge*>	enclist;
	int		i,c,c2;

	c = 0;
	while(!isClose(&enclist))
	{
		int		t,tt;
		list<edge*>	polygone;
		edge		*e;

		polygone.setUnique(1);

		tt = 0;
		for(c = 0;c < enclist.length() && tt < enclist.length() - 3;c++)
		{
			t = 1;
			c2 = 0;
			polygone.empty();
			polygone = getOpenPolygone(enclist);
/*
			polygone += enclist[(c * 3) % enclist.length()];
			printf("1. take edge %i out of %i\n",polygone[0]->id(),enclist.length());
			do
			{
				for(i = 0;i < enclist.length();i++)
				{
					e = enclist[(i + c * 3) % enclist.length()];
					if(polygone[t - 1]->knode2() == e->knode1())
					{
						if(polygone.find(e) >= 0)
							continue;
						polygone += e;
						t++;
						tt++;
						printf("%i. take edge %i\n",t,e->id());
						break;
					}
					else if(polygone[t - 1]->knode2() == e->knode2())
					{
						if(polygone.find(e) >= 0)
							continue;
						polygone += e->swap();
						t++;
						tt++;
						printf("%i. take swaped edge %i\n",t,e->id());
						break;
					}
				}
				c2++;
			}
			while(polygone[0]->knode1() != polygone[t - 1]->knode2()
				&& c2 < enclist.length());

*/
			if(polygone[0]->knode1() != polygone[t - 1]->knode2())
				return -1;

			// this is not the best solution
			// make a geometrytest if we have to use polygone[1] and polygon[2] or later pairs,
			// because they may be concave
			new edge(this,polygone[0]->knode1(),polygone[1]->knode2(),rand());
		}
		removeDoubleKnodes();
		removeDoubleEdges();
	}
/*
		for(i = 0;i < enclist.length();i++)
		{
			t = (i + c) % enclist.length();
			knodelist += enclist[t]->knode1();
			knodelist += enclist[t]->knode2();
		}

		if((g = subGraph(knodelist)))
		{
			if(g->shortestCircle(g->getKnode(c % enclist.length()),&polygon))
			{
				Vector3		v;

				for(i = 0;i < polygon.length();i++)
				{
					v += *polygon[i];
				}
				v /= polygon.length();

				k = new knode(this,v);
				for(i = 0;i < polygon.length();i++)
				{
					if((k2 = findKnode(*polygon[i])))
						new edge(this,k2,k,rand());
				}
			}
			delete g;
		}
		c++;
	}
*/

	return 0;
}

int	graph::removeDoubleEdges()
{
	int		i;

	for(i = 0;i < edgelist.length();i++)
	{
		if(!edgelist[i]->unique())
			delete edgelist[i];
	}

	return 0;
}

int	graph::removeDoubleKnodes()
{
	int		i,t;

	for(i = 0;i < knodelist.length() - 1;i++)
	{
		for(t = i + 1;t < knodelist.length();t++)
		{
			if((knodelist[i] != knodelist[t]) &&
			  ((Vector3)*knodelist[i] == (Vector3)*knodelist[t]))
			{
				knodelist[i]->replace(knodelist[t]);
				delete knodelist[t];
			}
		}
	}

	return 0;
}

int	graph::shortestPath(knode *kstart,knode *kend,list<knode*> *l)
{
	list<knode*>	way,best;
	int		i;

	best.empty();
	best += kstart;
	if(!shortestPath(best,kend))
		return 0;

	for(i = 0;i < kstart->numEdges();i++)
	{
		way.empty();
		way += kstart->getEdge(i)->knode1();
		if(shortestPath(way,kend))
		{
			if(way.length() < best.length())
			{
				best = way;
			}
		}
		way.empty();
		way += kstart->getEdge(i)->knode2();
		if(shortestPath(way,kend))
		{
			if(way.length() < best.length())
			{
				best = way;
			}
		}
	}

	if(l)
		*l = best;

	return 1;
}

int	graph::shortestPath(list<knode*> &now,knode *kend)
{
	list<knode*>	way,best;
	int		i;
	knode		*last;

	last = now.getLast();
	if(last == kend)
	{
		printf("best: ");
		for(i = 0;i < best.length();i++)
		{
			printf("%s ",best[i]->id());
		}
		printf("\n");
		return 1;
	}

	for(i = 0;i < last->numEdges();i++)
	{
		best.empty();
		if(now.find(last->getEdge(i)->knode1()) < 0)
		{
			best = now + last->getEdge(i)->knode1();
			if(shortestPath(best,kend))
				break;
		}
		if(now.find(last->getEdge(i)->knode2()) < 0)
		{
			best = now + last->getEdge(i)->knode2();
			if(shortestPath(best,kend))
				break;
		}
	}
	if(best.length() <= now.length())
		return 0;
	for(;i < last->numEdges();i++)
	{
		way.empty();
		if(now.find(last->getEdge(i)->knode1()) < 0)
		{
			way = now + last->getEdge(i)->knode1();
			if(shortestPath(way,kend))
				if(way.length() < best.length())
				{
					best = way;
				}
		}
		way.empty();
		if(now.find(last->getEdge(i)->knode2()) < 0)
		{
			way = now + last->getEdge(i)->knode2();
			if(shortestPath(way,kend))
				if(way.length() < best.length())
				{
					best = way;
				}
		}
	}

	if(best.length() > now.length())
		now = best;

	return 1;
}

int	graph::shortestCircle(knode *kstart,list<knode*> *l)
{
	list<knode*>	way,best;
	int		i;

	for(i = 0;i < kstart->numEdges();i++)
	{
		way.empty();
		way += kstart;
		if(kstart != kstart->getEdge(i)->knode1())
		{
			way += kstart->getEdge(i)->knode1();
			if(shortestCircle(way,kstart))
			{
				if(best.length() == 0 || way.length() < best.length())
				{
					best = way;
				}
			}
			way.empty();
			way += kstart;
		}
		if(kstart != kstart->getEdge(i)->knode2())
		{
			way += kstart->getEdge(i)->knode2();
			if(shortestCircle(way,kstart))
			{
				if(best.length() == 0 || way.length() < best.length())
				{
					best = way;
				}
			}
		}
	}

	if(l)
	{
		l->empty();
		for(i = 0;i < best.length() - 1;i++)
		{
			l->append(best[i]);
		}
	}

	printf("best: ");
	for(i = 0;i < best.length();i++)
	{
		printf("%s ",best[i]->id());
	}
	printf("\n");

	return 1;
}

int	graph::shortestCircle(list<knode*> &now,knode *kend)
{
	list<knode*>	way,best;
	int		i;
	knode		*last;

	last = now.getLast();
	if(last == kend)
	{
/*
		printf("now: ");
		for(i = 0;i < now.length();i++)
		{
			printf("%s ",now[i]->id());
		}
		printf("\n");
*/
		return 1;
	}

	for(i = 0;i < last->numEdges();i++)
	{
		way.empty();
		if(now.find(last->getEdge(i)->knode1()) < 1)
		{
			way = now;
			way += last->getEdge(i)->knode1();
			if(shortestCircle(way,kend))
				if((best.length() == 0
					|| way.length() < best.length())
					&& way.length() > 4)
				{
					best = way;
				}
		}
		way.empty();
		if(now.find(last->getEdge(i)->knode2()) < 1)
		{
			way = now;
			way += last->getEdge(i)->knode2();
			if(shortestCircle(way,kend))
				if((best.length() == 0
					|| way.length() < best.length())
					&& way.length() > 4)
				{
					best = way;
				}
		}
	}

	if(best.length() > now.length())
		now = best;


	return 1;
}

graph	*graph::subGraph(list<knode*> kl)
{
	graph	*g;
	edge	*e;
	int	i,t,c1,c2;

/*
	for(i = 0;i < edgelist.length();i++)
	{
		printf("%i. %i (%s,",i,
			edgelist[i]->id(),
			edgelist[i]->knode1()->id());
		printf("%s)\n",edgelist[i]->knode2()->id());
	}

	for(i = 0;i < kl.length();i++)
	{
		printf("%i. %s\n",i,kl[i]->id());
	}
*/

	g = new graph();
	for(i = 0;i < edgelist.length();i++)
	{
		e = edgelist[i];
		c1 = c2 = 0;
		for(t = 0;t < kl.length();t++)
		{
			if(e->knode1() == kl[t])
				c1 = t + 1;
		}
		for(t = 0;t < kl.length();t++)
		{
			if(e->knode2() == kl[t])
				c2 = t + 1;
		}
		if(c1 && c2)
		{
			g->add(e);
		}
	}

	g->removeDoubleKnodes();
	g->removeDoubleEdges();

	for(i = 0;i < g->knodelist.length();i++)
	{
		printf("%i. %s\n",i,g->knodelist[i]->id());
	}
	for(i = 0;i < g->edgelist.length();i++)
	{
		e = g->edgelist[i];
		e->knode1()->add(e);
		e->knode2()->add(e);
	}

	return g;
}

knode	*graph::findKnode(Vector3 v)
{
	int	i;

	for(i = 0;i < knodelist.length();i++)
	{
		if(*knodelist[i] == v)
			return knodelist[i];
	}

	return 0;
}

