/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** smesh.cpp
*/


#include "smesh.h"
#include "view.h" 

#include <objects.h>


smesh::smesh(base *p,char *n,texture *t) :
	csgobj(p,n,t)
{
	numtype = NUM_SMESH;

	if(parent)
		addToParent(parent);
}


smesh::smesh(base *p,char *n,Vector3 &v1,Vector3 &v2,Vector3 &v3):
	csgobj(p,n)
{
	numtype = NUM_SMESH;

	if(parent)
		addToParent(parent);

	addSegment(addNode(v1),addNode(v2),addNode(v3));
}

smesh::smesh(base *p,char *n,texture *t,Vector3 &v1,Vector3 &v2,Vector3 &v3) :
	csgobj(p,n,t)
{
	numtype = NUM_SMESH;

	if(parent)
		addToParent(parent);

	addSegment(addNode(v1),addNode(v2),addNode(v3));
}

smesh::smesh(base *p,smesh *mc) :
	csgobj(p,mc)
{
	int		i,s0,s1,s2;
	Vector3		v0,v1;

	numtype = NUM_SMESH;

	if(parent)
		addToParent(parent);

	for(i = 0;i < mc->vectorlist.length();i++)
	{
		if(mc->getNode(i,v0,v1) >= 0)
			addNode(v0,v1);
	}
	for(i = 0;i < mc->trianglelist.length();i++)
	{
		if(mc->getSegment(i,s0,s1,s2) >= 0)
			addSegment(s0,s1,s2);
	}
}

smesh::smesh(cmesh &m) : 	// Typumwandlung
	csgobj(m.getParent(),"Unnamed")
{
	printf("Typecast not implemented\n");
}


smesh::~smesh()
{
}

base		*smesh::copy(base *p)
{
	return new smesh(p,this);
}

int		smesh::addNode(double x, double y, double z,double nx, double ny, double nz)
{
	return addNode(Vector3(x,y,z),Vector3(nx,ny,nz));
}

int		smesh::addNode(Vector3 v,Vector3 n)
{
	dvec		dv;
	int		error;

	dv.position = v;
	dv.direction = n;

	if((error = vectorlist.append(dv)) != 0)
		return error;

	return vectorlist.length() - 1;
}

int	smesh::addSegment(int p1,int p2,int p3)
{
	seg		s;
	int		i,r1,r2,r3,r,l,m;

	if(p1 < 0 || p1 >= vectorlist.length())
		return -1; // p1 out of range
	if(p2 < 0 || p2 >= vectorlist.length())
		return -2; // p2 out of range
	if(p3 < 0 || p3 >= vectorlist.length())
		return -3; // p3 out of range

	if(p1 < p2)
	{
		if(p2 < p3)
		{
			// p1 < p2 < p3
			r1 = p1;r2 = p2;r3 = p3;
		}
		else // p3 < p2
		{
			if(p1 < p3)
			{
				// p1 < p3 < p2
				r1 = p1;r2 = p3;r3 = p2;
			}
			else // p3 < p1
			{
				// p3 < p1 < p2
				r1 = p3;r2 = p1;r3 = p2;
			}
		}
	}
	else // p2 < p1
	{
		if(p2 < p3)
		{
			if(p1 < p3)
			{
				// p2 < p1 < p3
				r1 = p2;r2 = p1;r3 = p3;
			}
			else // p3 < p1
			{
				// p2 < p3 < p1
				r1 = p2;r2 = p3;r3 = p1;
			}
		}
		else // p3 < p2
		{
			// p3 < p2 < p1
			r1 = p3;r2 = p2;r3 = p1;
		}
	}
	

	l = 0;r = trianglelist.length();m = 0;
	if(l + 1 == r)
	{
		s = trianglelist[0];
		if(r1 < s.pointnum[0])
		{
			m = 0;
		}
		else if(r1 > s.pointnum[0])
		{
			m = 1;
		}
		else	
		{
			if(r2 < s.pointnum[1])
			{
				m = 0;
			}
			else if(r2 > s.pointnum[1])
			{
				m = 1;
			}
			else	
			{
				if(r3 < s.pointnum[2])
				{
					m = 0;
				}
				else if(r3 > s.pointnum[2])
				{
					m = 1;
				}
				else	
				{
					// error double segment
					m = 1;
				}
			}
		}
		
	}
	while(l + 1 < r)
	{
		m = (l + r) / 2;
		s = trianglelist[m];
		if(r1 < s.pointnum[0])
		{
			r = m;
		}
		else if(r1 > s.pointnum[0])
		{
			l = m;
		}
		else	
		{
			if(r2 < s.pointnum[1])
			{
				r = m;
			}
			else if(r2 > s.pointnum[1])
			{
				l = m;
			}
			else	
			{
				if(r3 < s.pointnum[2])
				{
					r = m;
				}
				else if(r3 > s.pointnum[2])
				{
					l = m;
				}
				else	
				{
					// error double segment
					r = l = m;
				}
			}
		}
	}
	

	s.pointnum[0] = r1;
	s.pointnum[1] = r2;
	s.pointnum[2] = r3;
	
	if((i = trianglelist.insert(s,m)) != 0)
		return i;

	return trianglelist.length() - 1;
}

bool	smesh::hasSegment(int p1,int p2,int p3)
{
	return (findSegment(p1,p2,p3) >= 0);
}

int	smesh::findSegment(int p1,int p2,int p3)
{
	seg		s;
	int		r1,r2,r3,r,l,m;

	if(p1 < 0 || p1 >= vectorlist.length())
		return -1; // p1 out of range
	if(p2 < 0 || p2 >= vectorlist.length())
		return -1; // p2 out of range
	if(p3 < 0 || p3 >= vectorlist.length())
		return -1; // p3 out of range

	if(p1 < p2)
	{
		if(p2 < p3)
		{
			// p1 < p2 < p3
			r1 = p1;r2 = p2;r3 = p3;
		}
		else // p3 < p2
		{
			if(p1 < p3)
			{
				// p1 < p3 < p2
				r1 = p1;r2 = p3;r3 = p2;
			}
			else // p3 < p1
			{
				// p3 < p1 < p2
				r1 = p3;r2 = p1;r3 = p2;
			}
		}
	}
	else // p2 < p1
	{
		if(p2 < p3)
		{
			if(p1 < p3)
			{
				// p2 < p1 < p3
				r1 = p2;r2 = p1;r3 = p3;
			}
			else // p3 < p1
			{
				// p2 < p3 < p1
				r1 = p2;r2 = p3;r3 = p1;
			}
		}
		else // p3 < p2
		{
			// p3 < p2 < p1
			r1 = p3;r2 = p2;r3 = p1;
		}
	}
	

	l = 0;r = trianglelist.length();m = (l + r) / 2;
	while(l + 1 < r)
	{
		m = (l + r) / 2;
		s = trianglelist[m];
		if(r1 < s.pointnum[0])
		{
			r = m;
		}
		else if(r1 > s.pointnum[0])
		{
			l = m;
		}
		else	
		{
			if(r2 < s.pointnum[1])
			{
				r = m;
			}
			else if(r2 > s.pointnum[1])
			{
				l = m;
			}
			else	
			{
				if(r3 < s.pointnum[2])
				{
					r = m;
				}
				else if(r3 > s.pointnum[2])
				{
					l = m;
				}
				else	
				{
					// error double segment
					r = l = m;
				}
			}
		}
	}
		
	s = trianglelist[m];
	if(s.pointnum[0] == p1 && s.pointnum[1] == p2 && s.pointnum[2] == p3)
		return m;
	s = trianglelist[m + 1];
	if(s.pointnum[0] == p1 && s.pointnum[1] == p2 && s.pointnum[2] == p3)
		return m + 1;
	
	return -1;
}

int		smesh::getNode(int n,Vector3 &v,Vector3 &nv)
{
	if(n < 0 || n > vectorlist.length())
		return -1;

	v = vectorlist[n].direction;
	nv = vectorlist[n].position;

	return 0;
}

Vector3		&smesh::getPoint(int n)
{
	return vectorlist[n].position;
}

Vector3		&smesh::getNormal(int n)
{
	return vectorlist[n].direction;
}

int		smesh::getSegment(int n,int &s0,int &s1,int &s2)
{
	if(n < 0 || n > trianglelist.length()) 
		return  -1;

	s0 = trianglelist[n].pointnum[0]; 
	s1 = trianglelist[n].pointnum[1]; 
	s2 = trianglelist[n].pointnum[2];

	return 0;
}

int		smesh::removeSegment(int segnum)
{
	return trianglelist.deleteAt(segnum);
}

int		smesh::removeNode(int nodenum)
{
	int	i,s[3];

	for(i = 0;i < trianglelist.length();i++)
	{
		getSegment(i,s[0],s[1],s[2]);

		if(nodenum == s[0] || nodenum == s[1] || nodenum == s[2])
			return -1;
	}

	return vectorlist.deleteAt(nodenum);
}

int		smesh::getNumNodes()
{
	return vectorlist.length();
}

int		smesh::getNumSegments()
{
	return trianglelist.length();
}


int		smesh::simplify(double)
{
	return -1;
}


int		smesh::subdevide()
{
	seg		s;
	int		n,i,p1,p2,p3,pn;
	Vector3		v;
	double		l1,l2,l3;
	simplelist<seg>	seglist;

/*
	for(n = 0;n < trianglelist.length();n++)
	{
		seglist.append(trianglelist[n]);
	}
*/
	seglist = trianglelist;
		
	// take every triangle and devide it into 2 or 3
	for(n = 0;n < seglist.length();n++)
	{
		s = seglist[n];
		p1 = s.pointnum[0];
		p2 = s.pointnum[1];
		p3 = s.pointnum[2];
		l1 = vectorlist[p1].position.distance(vectorlist[p2].position);
		l2 = vectorlist[p2].position.distance(vectorlist[p3].position);
		l3 = vectorlist[p3].position.distance(vectorlist[p1].position);
		i = findSegment(p1,p2,p3);
		if(l1 * 1.4 > l2 + l3)
		{
			v = (vectorlist[p1].position + vectorlist[p2].position) / 2;
			pn = addNode(v);
			removeSegment(i);
			addSegment(p3,p1,pn);
			addSegment(p2,p3,pn);
		}
		else if(l2 * 1.4 > l3 + l1)
		{
			v = (vectorlist[p2].position + vectorlist[p3].position) / 2;
			pn = addNode(v);
			removeSegment(i);
			addSegment(p1,p2,pn);
			addSegment(p3,p1,pn);
		}
		else if(l3 * 1.4 > l1 + l2)
		{
			v = (vectorlist[p3].position + vectorlist[p1].position) / 2;
			pn = addNode(v);
			removeSegment(i);
			addSegment(p2,p3,pn);
			addSegment(p1,p2,pn);
		}
		else
		{
			v = (vectorlist[p1].position + vectorlist[p2].position + vectorlist[p3].position) / 3;
			pn = addNode(v);
			removeSegment(i);
			addSegment(p1,p2,pn);
			addSegment(p2,p3,pn);
			addSegment(p3,p1,pn);		
		}
	}

	return 0;
}

int		smesh::subdevideClever(int base,double grade)
{
	// take every tiangle and devide it into base * base
	// triangles
	// take advantage of the normalvectors
	
	return 0;
}



int		smesh::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int		smesh::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

void		smesh::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("smesh: %s\n",name);
}

int		smesh::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	int		i;

	printTab(fp,tab); 
	fprintf(fp,"mesh\n"); 
	printTab(fp,tab); 
	fprintf(fp,"{\n"); 
	for(i = 0;i < trianglelist.length();i++)
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"triangle\n");
		printTab(fp,tab + tabsize);
		fprintf(fp,"{\n");
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>,\n",
			vectorlist[trianglelist[i].pointnum[0]].position[0],
			vectorlist[trianglelist[i].pointnum[0]].position[1],
			vectorlist[trianglelist[i].pointnum[0]].position[2]);
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>,\n",
			vectorlist[trianglelist[i].pointnum[1]].position[0],
			vectorlist[trianglelist[i].pointnum[1]].position[1],
			vectorlist[trianglelist[i].pointnum[1]].position[2]);
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>\n",
			vectorlist[trianglelist[i].pointnum[2]].position[0],
			vectorlist[trianglelist[i].pointnum[2]].position[1],
			vectorlist[trianglelist[i].pointnum[2]].position[2]);

		printTab(fp,tab + tabsize);
		fprintf(fp,"} \n");
	}
	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	transform::exportPOV(fp,tab + tabsize,tabsize,anim);

	printTab(fp,tab);
	fprintf(fp,"}\n\n");

	return 0;
}


int		smesh::save(media *m,int ver)
{
	int	i;

	if(!m)
		return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);

			writeChunk("SMSH");
			writeNameChunk(name);

			saveFlags(m);
			anim::save(m);
			transform::save(m,ver);

			writeInt(vectorlist.length());
			for(i = 0;i < vectorlist.length();i++)
			{
				writeVector(vectorlist[i].position);
				writeVector(vectorlist[i].direction);
			}
			writeInt(trianglelist.length());
			for(i = 0;i < trianglelist.length();i++)
			{
				writeInt(trianglelist[i].pointnum[0]);
				writeInt(trianglelist[i].pointnum[1]);
				writeInt(trianglelist[i].pointnum[2]);
			}

			saveTexture(m);

			writeChunkLen();
 		break;
		default:
			return -2;
	}

	return 0;
}

int		smesh::load(media *m,int l,int ver)
{
	int		i,n,s0,s1,s2;
	int		pos;
	Vector3		v,vn;

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);

			pos = m->tell();

			loadFlags(m,l);

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);

			n = readInt();
			for(i = 0;i < n;i++)
			{
				v = readVector();
				vn = readVector();

				addNode(v,vn);
			}

			n = readInt();
			for(i = 0;i < n;i++)
			{
				s0 = readInt();
				s1 = readInt();
				s2 = readInt();
				addSegment(s0,s1,s2);
			}

			if(l - (m->tell() - pos) > 0)
			{
				loadTexture(m,l - (m->tell() - pos));
			}
		break;
		default:
			return -2;
	}

	return 0;
}


int		smesh::draw(view *v,Matrix44 m,int anim)
{
	int		i;
	seg		s;
	Vector3		v0,v1,v2;

	if(isFlag(HIDE)) return 0;

	transformMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		for(i = 0;i < trianglelist.length();i++)
		{
			s = trianglelist[i];
			v0 = vectorlist[s.pointnum[0]].position;
			v1 = vectorlist[s.pointnum[1]].position;
			v2 = vectorlist[s.pointnum[2]].position;
			v->drawLine(v0,v1,m,anim);
			v->drawLine(v1,v2,m,anim);
			v->drawLine(v2,v0,m,anim);
		}
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);


	return 0;
}

int		smesh::calculate(int)
{
	int		i;
	double		x,y,z;
	double		nx,ny,nz;
	double		xx,xy,xz;

	nx = xx = vectorlist[0].position[0];
	ny = xy = vectorlist[0].position[1];
	nz = xz = vectorlist[0].position[2];
	
	for(i = 1;i < vectorlist.length();i++)
	{
		x = vectorlist[i].position[0];
		y = vectorlist[i].position[1];
		z = vectorlist[i].position[2];
		nx = MIN(x,nx);
		ny = MIN(y,ny);
		nz = MIN(z,nz);
		xx = MAX(x,xx);
		xy = MAX(y,xy);
		xz = MAX(z,xz);
	}
	setMin(Vector3(nx,ny,nz));
	setMax(Vector3(xx,xy,xz));

	return 0;
}

int		smesh::addChild(box *b)
{
	vectorlist.empty();
	trianglelist.empty();

	addNode(-0.5,-0.5,-0.5);
	addNode( 0.5,-0.5,-0.5);
	addNode(-0.5, 0.5,-0.5);
	addNode( 0.5, 0.5,-0.5);
	addNode(-0.5,-0.5, 0.5);
	addNode( 0.5,-0.5, 0.5);
	addNode(-0.5, 0.5, 0.5);
	addNode( 0.5, 0.5, 0.5);

	addSegment(0,1,3);
	addSegment(0,3,2);
	addSegment(0,4,6);
	addSegment(0,6,2);
	addSegment(0,1,5);
	addSegment(0,4,5);
	addSegment(7,6,4);
	addSegment(7,4,5);
	addSegment(7,3,1);
	addSegment(7,1,5);
	addSegment(7,6,2);
	addSegment(7,2,3);

	return 0;
}

int		smesh::addChild(cone *c)
{
	int		i,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a;

	vectorlist.empty();
	trianglelist.empty();


	addNode(0,0,0.5);
	addNode(0,0,-0.5);
	for(i = 0;i < s;i++)
	{
		a = (double)i / (double)s * 2 * PI;
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			0.5);
		addNode(v);
		v = Vector3(c->capRadius() * sin(a),
			c->capRadius() * cos(a),
			-0.5);
		addNode(v);
	}

	for(i = 0;i < s;i++)
	{
		p1 = i * 2 + 2;
		p2 = i * 2 + 1 + 2;
		p3 = (i * 2 + 2) % (s * 2) + 2;
		p4 = (i * 2 + 3) % (s * 2) + 2;
		addSegment(0,p1,p3);
		addSegment(p1,p2,p3);
		addSegment(p1,p3,p4);
		addSegment(1,p2,p4);
	}

	return 0;
}

int		smesh::addChild(cylinder *c)
{
	int		i,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a;

	vectorlist.empty();
	trianglelist.empty();


	addNode(0,0,0.5);
	addNode(0,0,-0.5);
	for(i = 0;i < s;i++)
	{
		a = (double)i / (double)s * 2 * PI;
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			0.5);
		addNode(v);
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			-0.5);
		addNode(v);
	}

	for(i = 0;i < s;i++)
	{
		p1 = i * 2 + 2;
		p2 = i * 2 + 1 + 2;
		p3 = (i * 2 + 2) % (s * 2) + 2;
		p4 = (i * 2 + 3) % (s * 2) + 2;
		addSegment(0,p1,p3);
		addSegment(p1,p2,p3);
		addSegment(p1,p3,p4);
		addSegment(1,p2,p4);
	}

	return 0;
}

int		smesh::addChild(sphere *c)
{
	int		i,t,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a,b;

	vectorlist.empty();
	trianglelist.empty();


	addNode(0,0.5,0);
	addNode(0,-0.5,0);
	for(t = 1;t < (s / 2);t++)
	{
		b = (double)t / (double)s * 2 * PI;
		for(i = 0;i < s;i++)
		{
			a = (double)i / (double)s * 2 * PI;
			v = Vector3(0.5 * sin(a) * sin(b),
				0.5 * cos(b),
				0.5 * cos(a) * sin(b));
			addNode(v);
		}
	}

	for(i = 0;i < s;i++)
	{
		p1 = i + 2;
		p2 = (i + 1) % s + 2;
		addSegment(0,p1,p2);
	}
	for(t = 0;t < (s / 2) - 1;t++)
	{
		for(i = 0;i < s;i++)
		{
			p1 = t * s + i + 2;
			p2 = (t + 1) * s + i + 2;
			p3 = t * s + (i + 1) % s + 2;
			p4 = (t + 1) * s + (i + 1) % s + 2;
			addSegment(p1,p2,p4);
			addSegment(p1,p3,p4);
		}
	}
	for(i = 0;i < s;i++)
	{
		p1 = i + s * (s / 2 - 2) + 2;
		p2 = (i + 1) % s + s * (s / 2 - 2) + 2;
		addSegment(1,p1,p2);
	}

	return 0;
}

int		smesh::addChild(text *c)
{
//	int			i;

	vectorlist.empty();
	trianglelist.empty();

/*
	if((i = c->getMesh(this)) != 0)
		return i;
*/
/*
	printf("Got %i nodes and %i edges\n",vectorlist.length(),trianglelist.length() * 3);

	for(i = 0;i < vectorlist.length();i++)
		printf("%i. <%g,%g,%g>\n",i,vectorlist[i].position[0],vectorlist[i].position[1],vectorlist[i].position[2]);
	for(i = 0;i < trianglelist.length();i++)
		printf("%i. <%i,%i,%i>\n",i,trianglelist[i].pointnum[0],trianglelist[i].pointnum[1],trianglelist[i].pointnum[2]);
*/

//	close();

	return 0;
}

int		smesh::addChild(csgobj *co)
{
	switch(co->type())
	{
		case NUM_BOX:
			addChild((box*)co);
		break;
		case NUM_CONE:
			addChild((cone*)co);
		break;
		case NUM_CYLINDER:
			addChild((cylinder*)co);
		break;
		case NUM_SPHERE:
			addChild((sphere*)co);
		break;
		case NUM_TEXT:
			addChild((text*)co);
		break;
		default:
			return -1;
	}

	return -1;
}

int		smesh::addChild(blobobj *co)
{
	return addChild((csgobj*)co);
}

void		smesh::empty()
{
	vectorlist.empty();
	trianglelist.empty();
}
