/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** heightfield.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "heightfield.h"
#include "num.h"

#include "view.h"

typedef	int (heightfield::*dmethodfn)(param**,list<param*>*);

heightfield::heightfield(base *p,char *n,char *fn,int  h,int s,double wl,texture *t) :
	csgobj(p,n,t)
{
	numtype = NUM_HEIGHTFIELD;
	filename = strdup(fn);
	hierarchy = h;
	smooth = s;
	water_level = wl;
	field = 0;
	field_w = 0;
	field_h = 0;

	dmethodfn	d;

	d = &agetHeight;
	addMethod("getHeight",(methodfn)d);
	addMethod("getheight",(methodfn)d);
	
	if (parent) addToParent(parent);
}

heightfield::heightfield(base *p,char *n,char *fn,int  h,int s,double wl,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,vs,vr,vt)
{
 	numtype = NUM_HEIGHTFIELD;
	filename = strdup(fn);
	hierarchy = h;
	smooth = s;
	water_level = wl;
	field = 0;
	field_w = 0;
	field_h = 0;

	dmethodfn	d;

	d = &agetHeight;
	addMethod("getHeight",(methodfn)d);
	addMethod("getheight",(methodfn)d);

	if (parent) addToParent(parent);
}

heightfield::heightfield(base *p,char *n,char *fn,int  h,int s,double wl,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,t,vs,vr,vt)
{
 	numtype = NUM_HEIGHTFIELD;
	filename = strdup(fn);
	hierarchy = h;
	smooth = s;
	water_level = wl;
	field = 0;
	field_w = 0;
	field_h = 0;

	dmethodfn	d;

	d = &agetHeight;
	addMethod("getHeight",(methodfn)d);
	addMethod("getheight",(methodfn)d);

	if (parent) addToParent(parent);
}

heightfield::heightfield(base *p,heightfield *hc) :
	csgobj(p,(csgobj*)hc)
{
 	numtype = NUM_HEIGHTFIELD;
	if(hc->filename)
		filename = strdup(hc->filename);
	else
		filename = 0;
	hierarchy = hc->hierarchy;
	smooth = hc->smooth;
	water_level = hc->water_level;

	setField(hc->field,hc->field_w,hc->field_h);

	dmethodfn	d;

	d = &agetHeight;
	addMethod("getHeight",(methodfn)d);
	addMethod("getheight",(methodfn)d);

	if (parent) addToParent(parent);
}

heightfield::~heightfield()
{
	if(filename)
		free(filename);

	if(field)
		free(field);

	removeFromParent();
}

base	*heightfield::copy(base *p)
{
	return new heightfield(p,this);
}

int	heightfield::setField(float *f,int w,int h) 
{
	int	t;

	if(field)
		free(field);

	field = (float*)malloc(w * h * sizeof(float));
	if(!field && f)
		return -1;

	field_w = w;
	field_h = h;

	for(t = 0;t < h * w;t++)
	{
		field[t] = f[t];
	}

	return 0;
}

void		heightfield::setWaterLevel(double w)
{
	water_level = w;
}

double		heightfield::getWaterLevel()
{
	return water_level;
}

void		heightfield::setFileName(const char *f)
{
	if(filename)
		free(filename);

	filename = strdup(f);
}

char		*heightfield::getFileName()
{
	return filename;
}

bool		heightfield::getSmoothness()
{
	return smooth;
}

void		heightfield::setSmoothness(bool s)
{
	smooth = s;
}


int heightfield::addToParent(base *p) 
{ 
	if (!p) return -2;
	parent = p;
	return p->addChild(this);
}

int heightfield::removeFromParent() 
{ 
	if (!parent) return -2; 
	return parent->removeChild(this); 
} 
 
void heightfield::dumpNames(int tab,int)
{
	printTab(stderr, tab);
	printf("heightfield: %s\n",name);
}

int	heightfield::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	char *filetype;
	
	if(isFlag(NO_EXPORT)) return 0;

	filetype = strrchr(filename,'.');
	if(filetype == 0)
	{
		filetype = "sys";
	}
	else
	{
		filetype++;
	}

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = heightfield\n");
	printTab(fp,tab);
	fprintf(fp,"height_field\n"); 
	printTab(fp,tab); 
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize); 
	fprintf(fp,"%s \"%s\"\n",filetype,filename); 


	printTab(fp,tab + tabsize); 
	if(hierarchy == FALSE)
	{
		fprintf(fp,"hierarchy false\n");
	}
	else
	{
		fprintf(fp,"hierarchy true\n");
	}

	if(smooth == FALSE)
	{
//		printTab(fp,tab + tabsize); 
//		fprintf(fp,"smooth off\n");
	}
	else
	{
		printTab(fp,tab + tabsize); 
//		fprintf(fp,"smooth on\n");
		fprintf(fp,"smooth\n");
	}

	printTab(fp,tab + tabsize);
	fprintf(fp,"water_level %f\n",water_level);

	if(texptr)
		texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	printTab(fp,tab + tabsize);
	fprintf(fp,"translate <-0.5,-0.5,-0.5>\n");

	transform::exportPOV(fp,tab + tabsize,tabsize,anim);

	if(isFlag(HOLLOW_ON))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow on\n");
	}
	else if(isFlag(HOLLOW_OFF))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow off\n");
	}
	
	printTab(fp,tab);
	fprintf(fp,"}\n\n");

	return 0;
}

int	heightfield::save(media *m,int ver)
{
	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);

			writeChunk("HFLD");
			writeNameChunk(name);

			saveFlags(m);

			writeInt(hierarchy);
			writeInt(smooth);
			writeDouble(water_level);
			writeName(filename);
			anim::save(m);
			transform::save(m,ver);
			saveTexture(m);
			writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}

int	heightfield::load(media *m,int l,int ver)
{
	int	pos = m->tell();

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			loadFlags(m,l);

			hierarchy = readInt();
			smooth = readInt();
			water_level = readDouble();
			filename = readName();

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);
 			if(l > m->tell() - pos)
			{
				loadTexture(m,l - (m->tell() - pos));
			}
		break;
		default:
			return -2;
	}

	return 0;
}

int	heightfield::draw(view *v,Matrix44 m,int anim)
{
	Vector3		p,vx,vy,vz,a,b,c,d;
	int		x,y;

	if(isFlag(HIDE)) return 0;

	p = Vector3(0,0,0);
	vx = Vector3(1,0,0);
	vy = Vector3(0,1,0);
	vz = Vector3(0,0,1);

	transformMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		v->drawBox(p,vx * 0.5,vy * 0.5,vz * 0.5,m,anim);
		if(field)
		{
			int	sx,sy;

			sx = field_w / 32;
			sy = field_h / 32;
			for(x = 0;x < field_w - sx;x += sx)
			{
				for(y = 0;y < field_h - sy;y += sy)
				{
					a = Vector3(((double)x / (double)field_w - 0.5),
						MAX(field[x + y * field_w],water_level) - 0.5,
						((double)y / (double)field_h - 0.5));
					b = Vector3(((double)(x + sx) / (double)field_w - 0.5),
						MAX(field[(x + sx) + y * field_w],water_level) - 0.5,
						((double)y / (double)field_h - 0.5));
					c = Vector3(((double)(x + sx) / (double)field_w - 0.5),
						MAX(field[(x + sx) + (y + sy) * field_w],water_level) - 0.5,
						((double)(y + sy) / (double)field_h - 0.5));
/*
					d = Vector3(((double)x / (double)field_w - 0.5),
						MAX(field[x + (y + sy) * field_w],water_level) - 0.5,
						(((double)y + sy) / (double)field_h - 0.5));
*/
					v->drawLine(a,b,m,anim);
					v->drawLine(b,c,m,anim);
//					v->drawLine(c,d,m,anim);
//					v->drawLine(d,a,m,anim);
				}
			}
			for(x = 0;x < field_w - sx;x += sx)
			{
				y = field_h - sy;
       				a = Vector3(((double)x / (double)field_w - 0.5),
       					MAX(field[x + y * field_w],water_level) - 0.5,
       					((double)y / (double)field_h - 0.5));
       				b = Vector3(((double)(x + sx) / (double)field_w - 0.5),
       					MAX(field[(x + sx) + y * field_w],water_level) - 0.5,
       					((double)y / (double)field_h - 0.5));
       				c = Vector3(((double)(x + sx) / (double)field_w - 0.5),
       					MAX(field[(x + sx) + (y + sy) * field_w],water_level) - 0.5,
       					((double)(y + sy) / (double)field_h - 0.5));
       				d = Vector3(((double)x / (double)field_w - 0.5),
       					MAX(field[x + (y + sy) * field_w],water_level) - 0.5,
       					(((double)y + sy) / (double)field_h - 0.5));
       				v->drawLine(a,b,m,anim);
       				v->drawLine(b,c,m,anim);
       				v->drawLine(c,d,m,anim);
       				v->drawLine(d,a,m,anim);
			}
       			for(y = 0;y < field_h - sy;y += sy)
       			{
 				x = field_w - sx;
       				a = Vector3(((double)x / (double)field_w - 0.5),
       					MAX(field[x + y * field_w],water_level) - 0.5,
       					((double)y / (double)field_h - 0.5));
       				b = Vector3(((double)(x + sx) / (double)field_w - 0.5),
       					MAX(field[(x + sx) + y * field_w],water_level) - 0.5,
       					((double)y / (double)field_h - 0.5));
       				c = Vector3(((double)(x + sx) / (double)field_w - 0.5),
       					MAX(field[(x + sx) + (y + sy) * field_w],water_level) - 0.5,
       					((double)(y + sy) / (double)field_h - 0.5));
       				d = Vector3(((double)x / (double)field_w - 0.5),
       					MAX(field[x + (y + sy) * field_w],water_level) - 0.5,
       					(((double)y + sy) / (double)field_h - 0.5));
       				v->drawLine(a,b,m,anim);
       				v->drawLine(b,c,m,anim);
       				v->drawLine(c,d,m,anim);
       				v->drawLine(d,a,m,anim);
			}
		}
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	heightfield::calculate(int)
{
	setMin(Vector3(-0.5,-0.5,-0.5));
	setMax(Vector3(0.5,0.5,0.5));

	return 0;
}

int	heightfield::agetHeight(param **p,list<param*> *pl)
{
	double	x,y,z,dx,dz,y1,y2,y3;
        int	n,e,s,w;

	if(p == 0) return -1;
	if(pl->length() < 2 || pl->at(0)->type() != param::PM_REAL || pl->at(1)->type() != param::PM_REAL) return -2;

	*p = new pmreal;
	
	x = MIN(1,MAX(0,((pmreal*)pl->at(0))->real()));
	z = MIN(1,MAX(0,((pmreal*)pl->at(1))->real()));

	dx = x * (double)field_w;
	dz = z * (double)field_h;
	w = (int)floor(dx);
	e = (int)ceil(dx);
	s = (int)floor(dz);
	n = (int)ceil(dz);
	dx = dx - (double)w;
	dz = dz - (double)s;

	// which triangle to use
	if(dx + dz <= 1)
	{
		y1 = field[w + s * field_w];
		y2 = field[w + n * field_w];
		y3 = field[e + s * field_w];
		
		y = y1 * (1 - dx - dz)
		  + y2 * dz
		  + y3 * dx;
	}
	else
	{
		y1 = field[e + s * field_w];
		y2 = field[w + n * field_w];
		y3 = field[e + s * field_w];
		
		y = y1 * (dx + dz - 1)
		  + y2 * (1 - dx)
		  + y3 * (1 - dz);
	}
	
	((pmreal*)*p)->setReal(y);

	return 0;
}
