/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** matmath.cpp
*/


#include "matmath.h"

Matrix44::Matrix44() 
{
	int	x,y;

	for(y = 0;y < 4;y++)
		for(x = 0;x < 4;x++)
			elements[x][y] = 0;
}

Matrix44::Matrix44(Matrix44 &copy)
{
	int	x,y; 

	for(x = 0;x < 4;x++)
		for(y = 0;y < 4;y++)
			elements[x][y] = copy.elements[x][y];
}

Matrix44 Matrix44::operator+(const Matrix44 &right)
{
	Matrix44	temp(*this);
	int		x,y; 

	for(x = 0;x < 4;x++)
		for(y = 0;y < 4;y++)
			temp.elements[x][y] = temp.elements[x][y] + right.elements[x][y];

	return temp;
}

Matrix44 Matrix44::operator-(const Matrix44 &right) 
{ 
	Matrix44	temp(*this); 
	int		x,y; 

	for(x = 0;x < 4;x++)
		for(y = 0;y < 4;y++)
			temp.elements[x][y] = temp.elements[x][y] - right.elements[x][y];

	return temp; 
} 
 
Matrix44 Matrix44::operator-() 
{ 
	Matrix44	temp(*this); 
	int		x,y; 

	for(x = 0;x < 4;x++)
		for(y = 0;y < 4;y++)
			temp.elements[x][y] = -temp.elements[x][y];

	return temp; 
} 
 
Matrix44 Matrix44::operator*(const Matrix44 &right)
{
	Matrix44	prod;
	int		i,j,k;

	for(i = 0; i < 4; i++)
	{
		for(j = 0; j < 4; j++)
		{
			for(k = 0; k < 4; k++)
				prod.elements[i][j] += elements[k][j] * right.elements[i][k];
		}
	}

	return prod;
}

Matrix44 Matrix44::operator*(const double &right)
{
	Matrix44	prod;

	for(int i = 0; i < 4; i++)
	{
		for(int j = 0; j < 4; j++)
		{
			prod.elements[i][j] *= right;
		}
	}

	return prod; 
} 

Matrix44	&Matrix44::operator=(const Matrix44 &right)
{
	int	i,j;

	for(i = 0; i < 4; i++) 
	{ 
		for(j = 0; j < 4; j++) 
		{ 
			elements[i][j] = right.elements[i][j]; 
		} 
	} 

	return *this;
}


double	&Matrix44::operator()(int r,int c)
{
	return elements[r][c];
}

double	Matrix44::operator()(int r,int c) const
{ 
	return elements[r][c]; 
} 

/*
Matrix44 Matrix44::operator~()
// Inverse durch Gauss-Elimination mit Pivotsuche
{
	int i, j, k = 0, n;
	int pivrow, tarrow;
	double pivelt, tarelt;

	Matrix44 aug;
	Matrix44 aug2; 
	Matrix44 inverse;

	aug = *this;
	aug2.unify();
	for(pivrow = 0; pivrow < 4; pivrow++)
	{
		pivelt = aug.elements[pivrow][pivrow];
		if(pivelt == 0)
		{
			k = pivrow + 1;	// Pivotsuche
			while(pivelt == 0 && k <= 4)
			{
				pivelt = aug.elements[k][pivrow];
				k++;
			}
			if(pivelt == 0)
			{
				fprintf(stderr,"Fehler - Matrix singulaer\n");
				exit(1);
			}
		}
		else
		{
			k--;
			double		dum[8];

			for(i = 0; i < 4; i++)
				dum[i] = aug.elements[pivrow][i];
			for(i = 0; i < 4; i++)
				dum[i + 4] = aug2.elements[pivrow][i];

			for(i = 0; i < 4; i++)
			{
				aug.elements[pivrow][i] = aug.elements[k][i];
				aug2.elements[pivrow][i] = aug2.elements[k][i];
			}

			for (i = 0; i < 4; i++)
				aug.elements[k][i] = dum[i];
			for (i = 0; i < 4; i++)
				aug2.elements[k][i] = dum[i + 4];
		}
	}
	for(j = 0; j < 4; j++)
	{
		aug.elements[pivrow][j] = aug.elements[pivrow][j] / pivelt;
		aug2.elements[pivrow][j] = aug2.elements[pivrow][j] / pivelt;
	}
	for(tarrow = 0; tarrow < n; tarrow++)
		if(tarrow != pivrow)
		{
			tarelt = aug.elements[tarrow][pivrow];
			for (j = 0;j < 4; j++)
			{
				aug.elements[tarrow][j] = aug.elements[tarrow][j] - aug.elements[pivrow][j] * tarelt;
				aug2.elements[tarrow][j] = aug2.elements[tarrow][j] - aug2.elements[pivrow][j] * tarelt;
			}
		}
		for (i = 0; i < 4; i++)
			for (j = 0; j < 4; j++)
				inverse.elements[i][j] = aug2.elements[i][j];
	return inverse;
}
*/

Matrix44 Matrix44::operator~() 
{
	double		a;
	Matrix44	mh,mr;
	int		x,y,z;

	mr.unify();
	mh = *this;

	for(y = 0;y < 4;y++)
	{
		// Pruefen ob Nullspalte vorhanden => nicht invertierbare Matrix
		for(x = y;(x < 4)&&(mh(x,y) == 0);x++);
		if(x == 4) exit(-1);
		// Vertauschen von Zeilen
		if(x != y)
		{
			for(z = 0;z < 4;z++)
			{
				a = mh(x,z);
				mh(x,z) = mh(y,z);
				mh(y,z) = a;
				a = mr(x,z);
				mr(x,z) = mr(y,z);
				mr(y,z) = a;
			}
		}
		/* Dividieren einer Zeile durch einen Divisor */
		if(mh(y,y) != 1)
		{
			a = mh(y,y);
			for(z = 0;z < 4;z++)
			{
				mh(y,z) /= a;
				mr(y,z) /= a;
			}
		}
		/* Multiplizieren einer Zeile mit einem Faktor und anschliessendes Subtrahieren zweier Zeilen */
		for(x = 1;x < 4;x++)
		{
			if(mh((y + x) % 4,y) != 0)
			{
				a = mh((y + x) % 4,y);
				for(z = 0;z < 4;z++)
				{
					mh((y + x) % 4,z) -= mh(y,z) * a;
					mr((y + x) % 4,z) -= mr(y,z) * a;
				}
			}
		}
	}
	return mr;
}


Matrix44 &Matrix44::operator+=(const Matrix44 &m)
{
	Matrix44	t;

	t = (*this) + m;

	return (*this = t);
}

Matrix44 &Matrix44::operator-=(const Matrix44 &m)
{
	Matrix44	t;

	t = (*this) - m;

	return (*this = t);
}

Matrix44 &Matrix44::operator*=(const Matrix44 &m)
{
	Matrix44	t;

	t = (*this) * m;

	return (*this = t);
}

Matrix44 &Matrix44::operator*=(const double &d)
{
	Matrix44	t1,t2;

	t2 = *this;
	t1 = t2 * d;

	return (*this = t1);
}

void	Matrix44::unify()
{
	int	x,y;

	for(x = 0;x < 4;x++)
	{
		for(y = 0;y < 4;y++)
		{
			elements[x][y] = (x == y ? 1 : 0);
		}
	}
}

Matrix44	&Matrix44::scaleVector(Vector3 &v)
{
	this->unify();
	elements[0][0] = v[0];
	elements[1][1] = v[1];
	elements[2][2] = v[2];

	return *this;
}

Matrix44	&Matrix44::transposeVector(Vector3 &v)
{
	this->unify();
	elements[0][3] = v[0];
	elements[1][3] = v[1];
	elements[2][3] = v[2];

	return *this;
}

Matrix44	&Matrix44::rotateVector(Vector3 &v)
{
	Matrix44	h;
	double		cx,sx,cy,sy,cz,sz;

	cx = cos(v[0]);
	sx = sin(v[0]);
	cy = cos(v[1]);
	sy = sin(v[1]);
	cz = cos(v[2]);
	sz = sin(v[2]);

	this->unify();

	h.unify();
	h.elements[1][1] = cx;
	h.elements[1][2] = -sx;
	h.elements[2][1] = sx;
	h.elements[2][2] = cx;
	(*this) *= h;

	h.unify();
	h.elements[2][2] = cy;
	h.elements[2][0] = -sy;
	h.elements[0][2] = sy;
	h.elements[0][0] = cy;
	(*this) *= h;

	h.unify();
	h.elements[0][0] = cz;
	h.elements[0][1] = -sz;
	h.elements[1][0] = sz;
	h.elements[1][1] = cz;
	(*this) *= h;

	return *this;
}




Vector4::Vector4() 
{
	elements[0] = 0;
	elements[1] = 0;
	elements[2] = 0;
	elements[3] = 0;
}

Vector4::Vector4(double a,double b,double c,double d) 
{
	elements[0] = a;
	elements[1] = b;
	elements[2] = c;
	elements[3] = d;
}

Vector4::Vector4(const Vector4 &copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = copy.elements[2];
	elements[3] = copy.elements[3];
}

Vector4::Vector4(Vector3 copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = copy.elements[2];
	elements[3] = 1;
}

Vector4::Vector4(Vector3 copy,double d) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = copy.elements[2];
	elements[3] = d;
}

Vector4::Vector4(Vector2 copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = 0;
	elements[3] = 1;
}

Vector4::Vector4(Vector2 copy,double d1,double d2) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = d1;
	elements[3] = d2;
}

Vector4 Vector4::operator+(const Vector4 &right) const
{
	Vector4	temp(*this);

	temp.elements[0] += right.elements[0];
	temp.elements[1] += right.elements[1];
	temp.elements[2] += right.elements[2];
	temp.elements[3] += right.elements[3];

	return temp; 
} 
 
Vector4 Vector4::operator-(const Vector4 &right) const
{ 
	Vector4	temp(*this); 

	temp.elements[0] -= right.elements[0];
	temp.elements[1] -= right.elements[1];
	temp.elements[2] -= right.elements[2];
	temp.elements[3] -= right.elements[3];

	return temp; 
} 
 
Vector4 Vector4::operator-() const
{ 
	Vector4	v(-elements[0],-elements[1],-elements[2],-elements[3]); 

	return v; 
} 
 
Vector4 Vector4::operator*(const double &right) const
{
	Vector4	temp(*this); 

	temp.elements[0] *= right;
	temp.elements[1] *= right;
	temp.elements[2] *= right;
	temp.elements[3] *= right;

	return temp; 
} 

Vector4 Vector4::operator/(const double &right) const
{
	Vector4	temp(*this); 

	temp.elements[0] /= right;
	temp.elements[1] /= right;
	temp.elements[2] /= right;
	temp.elements[3] /= right;

	return temp; 
} 

Vector4	&Vector4::operator=(const Vector4 &right)
{
	int	i;

	for(i = 0; i < 4; i++) 
	{ 
		elements[i] = right.elements[i]; 
	} 

	return *this; 
} 
 
double	&Vector4::operator()(int r) 
{ 
	return elements[r]; 
} 

double	&Vector4::operator[](int r) 
{ 
	return elements[r]; 
} 

Vector4 &Vector4::operator+=(const Vector4 &m)
{
	Vector4	t;

	t = (*this) + m;

	return (*this = t);
}

Vector4 &Vector4::operator-=(const Vector4 &m)
{
	Vector4	t;

	t = (*this) - m;

	return (*this = t);
}

Vector4 &Vector4::operator*=(const double &d)
{
	Vector4	t1,t2;

	t2 = *this;
	t1 = t2 * d;

	return (*this = t1);
}

Vector4 &Vector4::operator/=(const double &d)
{
	Vector4	t1,t2;

	t2 = *this;
	t1 = t2 / d;

	return (*this = t1);
}

void	Vector4::normalize()
{
	*this /= length();
}

double	Vector4::length()
{
	return sqrt(scalarprod(*this));
}

double	Vector4::scalarprod(const Vector4 &v)
{
	return elements[0] * v.elements[0]
		 + elements[1] * v.elements[1]
		 + elements[2] * v.elements[2]
		 + elements[3] * v.elements[3];
}

Vector4	Vector4::operator*(const Matrix44 &m) const
{
	Vector4		v;
	int		i,j;

	for(i = 0;i < 4;i++)
	{
		for(j = 0;j < 4;j++)
		{
			v.elements[i] += m(i,j) * elements[j];
		}
	}

	return v;
}

Vector4	&Vector4::operator*=(const Matrix44 &m)
{
	Vector4		v;
	int		i,j;

	for(i = 0;i < 4;i++)
	{
		for(j = 0;j < 4;j++)
		{
			v.elements[i] += m(i,j) * elements[j];
		}
	}

	return (*this = v);
}



Vector3::Vector3() 
{
	elements[0] = 0;
	elements[1] = 0;
	elements[2] = 0;
}

Vector3::Vector3(double a,double b,double c) 
{
	elements[0] = a;
	elements[1] = b;
	elements[2] = c;
}

Vector3::Vector3(Vector4 copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = copy.elements[2];
}

Vector3::Vector3(const Vector3 &copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = copy.elements[2];
}

Vector3::Vector3(Vector3 &copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = copy.elements[2];
}

Vector3::Vector3(Vector2 copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = 0;
}

Vector3::Vector3(Vector2 copy,double d) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
	elements[2] = d;
}

Vector3 Vector3::operator+(const Vector3 &right) const
{
	Vector3	temp = *this;

	temp.elements[0] += right.elements[0];
	temp.elements[1] += right.elements[1];
	temp.elements[2] += right.elements[2];

	return temp;
} 
 
Vector3 Vector3::operator-(const Vector3 &right) const
{ 
	Vector3	temp = *this; 

	temp.elements[0] -= right.elements[0];
	temp.elements[1] -= right.elements[1];
	temp.elements[2] -= right.elements[2];

	return temp;
} 
 
Vector3 Vector3::operator-() const
{ 
	Vector3	temp(-elements[0],-elements[1],-elements[2]); 

	return temp; 
} 
 
Vector3 Vector3::operator*(const double &right) const
{
	Vector3	prod(*this);

	prod.elements[0] *= right;
	prod.elements[1] *= right;
	prod.elements[2] *= right;

	return prod; 
} 

Vector3 Vector3::operator*(const Vector3 &right) const
{
	Vector3	prod(*this);

	prod.elements[0] = right.elements[1] * elements[2] - right.elements[2] * elements[1];
	prod.elements[1] = right.elements[2] * elements[0] - right.elements[0] * elements[2];
	prod.elements[2] = right.elements[0] * elements[1] - right.elements[1] * elements[0];

	return prod; 
} 

Vector3 Vector3::operator/(const double &right) const
{
	Vector3	prod(*this);

	prod.elements[0] /= right;
	prod.elements[1] /= right;
	prod.elements[2] /= right;

	return prod; 
} 

Vector3	&Vector3::operator=(const Vector3 &right)
{
	elements[0] = right.elements[0]; 
	elements[1] = right.elements[1]; 
	elements[2] = right.elements[2]; 

	return *this; 
} 

double	&Vector3::operator()(int r) 
{ 
	return elements[r]; 
} 

double	&Vector3::operator[](int r) 
{ 
	return elements[r]; 
} 

Vector3 &Vector3::operator+=(const Vector3 &m)
{
	Vector3	t;

	t = (*this) + m;

	return (*this = t);
}

Vector3 &Vector3::operator-=(const Vector3 &m)
{
	Vector3	t;

	t = (*this) - m;

	return (*this = t);
}

Vector3 &Vector3::operator*=(const Vector3 &m)
{
	Vector3	t;

	t = (*this) * m;

	return (*this = t);
}

Vector3 &Vector3::operator*=(const double &d)
{
	Vector3	t1,t2;

	t2 = *this;
	t1 = t2 * d;

	return (*this = t1);
}

Vector3 &Vector3::operator/=(const double &d)
{
	Vector3	t1,t2;

	t2 = *this;
	t1 = t2 / d;

	return (*this = t1);
}

int	Vector3::operator==(const Vector3 &v)
{
	return (v.elements[0] == elements[0]
		&& v.elements[1] == elements[1]
		&& v.elements[2] == elements[2]);
}

int	Vector3::operator!=(const Vector3 &v)
{
	return (v.elements[0] != elements[0]
		|| v.elements[1] != elements[1]
		|| v.elements[2] != elements[2]);
}

void	Vector3::normalize()
{
	*this /= length();
}

double	Vector3::length()
{
	return sqrt(scalarprod(*this));
}

double	Vector3::scalarprod(const Vector3 &v)
{
	return elements[0] * v.elements[0]
		 + elements[1] * v.elements[1]
		 + elements[2] * v.elements[2];
}

double	Vector3::distance(const Vector3 &v1)
{
	return ((*this - v1).length());
}

double	Vector3::distance(const Vector3 &v1,const Vector3 &v2)
{
	Vector3		n;

	n = ((v1 - v2) * (v1 - *this));
	n.normalize();

	return distance(v1,v2,v2 + n);
}

double	Vector3::distance(const Vector3 &v1,const Vector3 &v2,const Vector3 &v3)
{
	Vector3		n;

	n = ((v1 - v2) * (v1 - v3));
	n.normalize();

	return (n * (*this - v1)).length();
}





Vector2::Vector2() 
{
	elements[0] = 0;
	elements[1] = 0;
}

Vector2::Vector2(double a,double b) 
{
	elements[0] = a;
	elements[1] = b;
}

Vector2::Vector2(Vector4 copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
}

Vector2::Vector2(Vector3 copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
}

Vector2::Vector2(const Vector2 &copy) 
{
	elements[0] = copy.elements[0];
	elements[1] = copy.elements[1];
}

Vector2 Vector2::operator+(const Vector2 &right) const
{
	Vector2	temp(*this);

	temp.elements[0] += right.elements[0];
	temp.elements[1] += right.elements[1];

	return temp; 
} 
 
Vector2 Vector2::operator-(const Vector2 &right) const
{ 
	Vector2	temp(*this); 

	temp.elements[0] -= right.elements[0];
	temp.elements[1] -= right.elements[1];

	return temp; 
} 
 
Vector2 Vector2::operator-() const
{ 
	Vector2	temp(-elements[0],-elements[1]); 

	return temp; 
} 
 
Vector2 Vector2::operator*(const double &right) const
{
	Vector2	prod(*this);

	prod.elements[0] *= right;
	prod.elements[1] *= right;

	return prod; 
} 

Vector2 Vector2::operator/(const double &right) const
{
	Vector2	prod(*this);

	prod[0] /= right;
	prod[1] /= right;

	return prod;
} 

Vector2	&Vector2::operator=(const Vector2 &right)
{
	int	i;

	for(i = 0; i < 2; i++) 
	{ 
		elements[i] = right.elements[i]; 
	} 

	return *this; 
} 

double	&Vector2::operator()(int r)
{
	return elements[r];
}

double	&Vector2::operator[](int r)
{
	return elements[r];
}

Vector2 &Vector2::operator+=(const Vector2 &m)
{
	Vector2	t;

	t = (*this) + m;

	return (*this = t);
}

Vector2 &Vector2::operator-=(const Vector2 &m)
{
	Vector2	t;

	t = (*this) - m;

	return (*this = t);
}

Vector2 &Vector2::operator*=(const double &d)
{
	Vector2	t1,t2;

	t2 = *this;
	t1 = t2 * d;

	return (*this = t1);
}

Vector2 &Vector2::operator/=(const double &d)
{
	Vector2	t1,t2;

	t2 = *this;
	t1 = t2 / d;

	return (*this = t1);
}

void	Vector2::normalize()
{
	*this /= length();
}

double	Vector2::length()
{
	return sqrt(scalarprod(*this));
}

double	Vector2::scalarprod(const Vector2 &v)
{
	return elements[0] * v.elements[0] + elements[1] * v.elements[1];
}



