/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** viewwidget.cpp
*/
#include "viewwidget.h"
#include "viewwidget.moc"

#include <misc.h>
#include "../misc/preferences.h"

ViewWidget::ViewWidget(QWidget *p,const char *n,world *wptr) :
	QFrame(p,n), view(wptr,n)
{
	qpm = new QPixmap(width(),height());
	qp = new QPainter(qpm);

	configure();


	peye = Vector3(0,0,200);
	vview = Vector3(0,0,-1);
	vup = Vector3(0,1,0);
	vright = Vector3(1,0,0);

/*
        translate(Vector3(0,0,-200));
        rotate(Vector3(PI,0,0));
*/

	setFOV(0.1);
	
	xo = 0;yo = 0;
	viewmode = 1;
	axemode = 7;

	ppmfp = 0;
}

ViewWidget::~ViewWidget()
{
//	qp->end();
	
	delete qp;
}

int	ViewWidget::drawLine(Vector3 a,Vector3 b,Matrix44 m,int anim)
{
	if(drawselected)
		return drawLine(a,b,m,anim,selcolor);

	return drawLine(a,b,m,anim,drawcolor);
}

int	ViewWidget::drawLine(Vector3 a,Vector3 b,Matrix44 m,int anim,QColor color)
{
	Vector2	a2,b2,c2;

	if(project(a,a2,m,anim)) return -1;
	if(project(b,b2,m,anim)) return -1;

	qp->setPen(color);

	if(a2[0] > 32000.0)
	{
		if(a2[0] == b2[0]) return -2;
		c2[0] = 32000;
		c2[1] = ((a2[1] - b2[1]) * (c2[0] - b2[0])) / (a2[0] - b2[0]) - b2[1];
		a2 = c2;
	}
	if(a2[0] < -32000.0)
	{
		if(a2[0] == b2[0]) return -2;
		c2[0] = -32000;
		c2[1] = ((a2[1] - b2[1]) * (c2[0] - b2[0])) / (a2[0] - b2[0]) - b2[1];
		a2 = c2;
	}
	if(a2[1] > 32000.0)
	{
		if(a2[1] == b2[1]) return -2;
		c2[1] = 32000;
		c2[0] = ((a2[0] - b2[0]) * (c2[1] - b2[1])) / (a2[1] - b2[1]) - b2[0];
		a2 = c2;
	}
	if(a2[1] < -32000.0)
	{
		if(a2[1] == b2[1]) return -2;
		c2[1] = -32000;
		c2[0] = ((a2[0] - b2[0]) * (c2[1] - b2[1])) / (a2[1] - b2[1]) - b2[0];
		a2 = c2;
	}

	if(b2[0] > 32000.0)
	{
		if(b2[0] == a2[0]) return -2;
		c2[0] = 32000;
		c2[1] = ((b2[1] - a2[1]) * (c2[0] - a2[0])) / (b2[0] - a2[0]) - a2[1];
		b2 = c2;
	}
	if(b2[0] < -32000.0)
	{
		if(b2[0] == a2[0]) return -2;
		c2[0] = -32000;
		c2[1] = ((b2[1] - a2[1]) * (c2[0] - a2[0])) / (b2[0] - a2[0]) - a2[1];
		b2 = c2;
	}
	if(b2[1] > 32000.0)
	{
		if(b2[1] == a2[1]) return -2;
		c2[1] = 32000;
		c2[0] = ((b2[0] - a2[0]) * (c2[1] - a2[1])) / (b2[1] - a2[1]) - a2[0];
		b2 = c2;
	}
	if(b2[1] < -32000.0)
	{
		if(b2[1] == a2[1]) return -2;
		c2[1] = -32000;
		c2[0] = ((b2[0] - a2[0]) * (c2[1] - a2[1])) / (b2[1] - a2[1]) - a2[0];
		b2 = c2;
	}

	qp->drawLine((int)a2[0],(int)a2[1],(int)b2[0],(int)b2[1]);

	return 0;
}

int	ViewWidget::drawCross(Vector3 a,Matrix44 m,int anim)
{
	Vector2	a2;

	if(project(a,a2,m,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2[0] - 10,(int)a2[1] - 10,(int)a2[0] + 10,(int)a2[1] + 10);
	qp->drawLine((int)a2[0] + 10,(int)a2[1] - 10,(int)a2[0] - 10,(int)a2[1] + 10);

	return 0;
}

int		ViewWidget::drawTriangle(Vector3 a,Vector3 b,Vector3 c,Matrix44 m,int anim)
{
	Vector2	a2,b2,c2;

	if(project(a,a2,m,anim)) return -1;
	if(project(b,b2,m,anim)) return -1;
	if(project(c,c2,m,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2[0],(int)a2[1],(int)b2[0],(int)b2[1]);
	qp->drawLine((int)b2[0],(int)b2[1],(int)c2[0],(int)c2[1]);
	qp->drawLine((int)c2[0],(int)c2[1],(int)a2[0],(int)a2[1]);

	return 0;
}

int	ViewWidget::drawTrapezium(Vector3 a,Vector3 b,Vector3 c,Matrix44 m,int anim)
{
	Vector3		d;
	Vector2	a2,b2,c2,d2;

	d = b;
	d -= a;
	d += c;

	if(project(a,a2,m,anim)) return -1;
	if(project(b,b2,m,anim)) return -1;
	if(project(c,c2,m,anim)) return -1;
	if(project(d,d2,m,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	qp->drawLine((int)a2(0),(int)a2(1),(int)b2(0),(int)b2(1));
	qp->drawLine((int)b2(0),(int)b2(1),(int)c2(0),(int)c2(1));
	qp->drawLine((int)c2(0),(int)c2(1),(int)d2(0),(int)d2(1));
	qp->drawLine((int)d2(0),(int)d2(1),(int)a2(0),(int)a2(1));

	return 0;
}

int	ViewWidget::drawCircle(Vector3 m,Vector3 a,Vector3 b,Matrix44 ma,int anim)
{
	Vector3		d,n;
	Vector2	n2,d2;
	int		t;
	double		alpha;

	alpha = (double)-1 / 16 * PI;
	n = m;
	n += a * cos(alpha) + b * sin(alpha);
	if(project(n,n2,ma,anim)) return -1;

/*
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);
*/
	for(t = 0;t < 32;t++)
	{
		alpha = (double)t / 16 * PI;
		d = m;
		d += a * cos(alpha) + b * sin(alpha);

		if(project(d,d2,ma,anim)) return -1;

		qp->drawLine((int)n2(0),(int)n2(1),(int)d2(0),(int)d2(1));

		n2 = d2;
	}

	return 0;
}

int		ViewWidget::drawSemiCircle(Vector3 m,Vector3 a,Vector3 b,Matrix44 ma,int anim)
{
	Vector3		d,n;
	Vector2		n2,d2;
	int		t;
	double		alpha;

	alpha = (double)-1 / 16 * PI;
	n = m;
	n += a * cos(alpha) + b * sin(alpha);
	if(project(n,n2,ma,anim)) return -1;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	for(t = 0;t < 16;t++)
	{
		alpha = (double)t / 16 * PI;
		d = m;
		d += a * cos(alpha) + b * sin(alpha);

		if(project(d,d2,ma,anim)) return -1;

		qp->drawLine((int)n2(0),(int)n2(1),(int)d2(0),(int)d2(1));

		n2 = d2;
	}

	return 0;
}

int		ViewWidget::drawSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m,a,b,ma,anim);
	drawCircle(m,b,c,ma,anim);
	drawCircle(m,c,a,ma,anim);

	return 0;
}

int		ViewWidget::drawSemiSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawCircle(m,a,b,ma,anim);
	drawSemiCircle(m,b,c,ma,anim);
	drawSemiCircle(m,c,a,ma,anim);

	return 0;
}

int		ViewWidget::drawBox(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] -= (a + b + c);
	v[1] -= (a - b + c);
	v[2] += (a + b - c);
	v[3] += (a - b - c);
	v[4] -= (a + b - c);
	v[5] -= (a - b - c);
	v[6] += (a + b + c);
	v[7] += (a - b + c);

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[1],ma,anim);
	drawLine(v[1],v[2],ma,anim);
	drawLine(v[2],v[3],ma,anim);
	drawLine(v[3],v[0],ma,anim);
	drawLine(v[4],v[5],ma,anim);
	drawLine(v[5],v[6],ma,anim);
	drawLine(v[6],v[7],ma,anim);
	drawLine(v[7],v[4],ma,anim);
	drawLine(v[0],v[4],ma,anim);
	drawLine(v[1],v[5],ma,anim);
	drawLine(v[2],v[6],ma,anim);
	drawLine(v[3],v[7],ma,anim);

	return 0;
}

int		ViewWidget::drawCone(Vector3 m,Vector3 a,Vector3 b,Vector3 c,double d,Matrix44 ma,int anim)
{
	Vector3		v[8];

	d *= 2;

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a + c;
	v[1] += b + c;
	v[2] += a + c;
	v[3] += -b + c;
	v[4] += -a * d - c;
	v[5] += b * d - c;
	v[6] += a * d - c;
	v[7] += -b * d - c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[4],ma,anim);
	drawLine(v[1],v[5],ma,anim);
	drawLine(v[2],v[6],ma,anim);
	drawLine(v[3],v[7],ma,anim);

	drawCircle(c,a,b,ma,anim);
	drawCircle(m,a * (1 + d) / 2,b * (1 + d) / 2,ma,anim);
	drawCircle(-c,a * d,b * d,ma,anim);

	return 0;
}

int		ViewWidget::drawCylinder(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a - c;
	v[1] += b - c;
	v[2] += a - c;
	v[3] += -b - c;
	v[4] += -a + c;
	v[5] += b + c;
	v[6] += a + c;
	v[7] += -b + c;

	if(drawselected) qp->setPen(selcolor);
	else qp->setPen(drawcolor);

	drawLine(v[0],v[4],ma,anim);
	drawLine(v[1],v[5],ma,anim);
	drawLine(v[2],v[6],ma,anim);
	drawLine(v[3],v[7],ma,anim);

	drawCircle(-c,a,b,ma,anim);
	drawCircle(m,a,b,ma,anim);
	drawCircle(c,a,b,ma,anim);

	return 0;
}

int		ViewWidget::drawTorus(Vector3 m,Vector3 a,Vector3 b,Vector3 c,double d,Matrix44 ma,int anim)
{
	if(drawselected)
		qp->setPen(selcolor);
	else
		qp->setPen(drawcolor);

	// top
	drawCircle(m + b,a,c,ma,anim);
	// bottom
	drawCircle(m - b,a,c,ma,anim);
	// inner
	drawCircle(m,a / a.length() * (a.length() - d),c / c.length() * (c.length() - d),ma,anim);
	// outer
	drawCircle(m,a / a.length() * (a.length() + d),c / c.length() * (c.length() + d),ma,anim);

	// west
	drawCircle(m + a,a / a.length() * d,b,ma,anim);
	// east
	drawCircle(m - a,a / a.length() * d,b,ma,anim);
	// north
	drawCircle(m + c,c / c.length() * d,b,ma,anim);
	// south
	drawCircle(m - c,c / c.length() * d,b,ma,anim);

	return 0;
}

int		ViewWidget::drawBlobSphere(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	qp->setPen(blobcolor);

	drawCircle(m,a,b,ma,anim);
	drawCircle(m,b,c,ma,anim);
	drawCircle(m,c,a,ma,anim);

	return 0;
}

int		ViewWidget::drawBlobCylinder(Vector3 m,Vector3 a,Vector3 b,Vector3 c,Matrix44 ma,int anim)
{
	Vector3		v[8];

	v[0] = v[1] = v[2] = v[3] = v[4] = v[5] = v[6] = v[7] = m;
	v[0] += -a - c;
	v[1] += b - c;
	v[2] += a - c;
	v[3] += -b - c;
	v[4] += -a + c;
	v[5] += b + c;
	v[6] += a + c;
	v[7] += -b + c;

	drawLine(v[0],v[4],ma,anim,blobcolor);
	drawLine(v[1],v[5],ma,anim,blobcolor);
	drawLine(v[2],v[6],ma,anim,blobcolor);
	drawLine(v[3],v[7],ma,anim,blobcolor);

	qp->setPen(blobcolor);

	drawCircle(-c,a,b,ma,anim);
	drawCircle(m,a,b,ma,anim);
	drawCircle(c,a,b,ma,anim);

	return 0;
}

int		ViewWidget::drawIcon(Vector3 v,QPixmap p,int x,int y,Matrix44 m,int anim)
{
	Vector2	v2;

	if(project(v,v2,m,anim)) return -1;

	qp->drawPixmap((int)v2[0] + x,(int)v2[1] + y,p);

	return 0;
}

int		ViewWidget::drawSymbol(Vector3 v,const char *name,Matrix44 m,int anim)
{
#include "icons/tree/lightpoint.xpm"
#include "icons/tree/spotlight.xpm"
//#include "icons/tree/arealight.xpm"

	QPixmap			pm_lp((const char**)lightpoint_xpm);
	QPixmap			pm_sl((const char**)spotlight_xpm);

	if(strcmp(name,"lightpoint") == 0)
		return drawIcon(v,pm_lp,12,12,m,anim);
	if(strcmp(name,"spotlight") == 0)
		return drawIcon(v,pm_sl,12,12,m,anim);

	return -1;
}

int		ViewWidget::drawAxis(Matrix44 m,int anim)
{
	Vector3		vm(0,0,0),vx(1,0,0),vy(0,1,0),vz(0,0,1);

	drawLine(vm,vx,m,anim,xaxecolor);//QColor(200,0,0)

	drawLine(vm,vy,m,anim,yaxecolor);//QColor(0,200,0)

	drawLine(vm,vz,m,anim,zaxecolor);//QColor(100,0,200)

	return 0;
}

int		ViewWidget::drawDragvector(Matrix44 m,Vector3 p,Vector3 d,int anim)
{
	Vector2	a;

	if(anim) return 0;
	
	drawLine(p,p + d,m,0,QColor(128,0,64));

	if(project(p + d,a,m,0)) return -1;

	qp->setPen(QColor(128,0,64));
	qp->drawEllipse((int)a[0] - 3,(int)a[1] - 3,7,7);

	return 0;
}

void		ViewWidget::drawanim()
{
	draw(1);
}

void		ViewWidget::draw()
{
	draw(0);
}

void		ViewWidget::draw(int anim)
{
	Matrix44	m;
	Vector3		vm(0,0,0),vx(1,0,0),vy(0,1,0),vz(0,0,1),a,b;
	int		t;

	stopRendering();

	m.unify();

	qp->fillRect(1,1,width() - 2,height() - 2,backcolor);
	qp->setPen(QColor(0,0,0));
	qp->drawLine(0,0,width(),0);
	qp->drawLine(0,0,0,height());
	qp->drawLine(1,1,width() - 2,1);
	qp->setPen(QColor(255,255,255));
	qp->drawLine(width() - 1,0,width() - 1,height() - 1);
	qp->drawLine(0,height() - 1,width() - 1,height() - 1);

	qp->setPen(gridcolor);

	switch(projmode)
	{
		case PERSPECTIVE:
			for(t = -gridlines_xz;grid_xz && t <= gridlines_xz;t++)
			{
				a = (vx * (double)t + vz * -gridlines_xz) * gridspace_xz;
				b = (vx * (double)t + vz * gridlines_xz) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
				a = (vz * (double)t + vx * -gridlines_xz) * gridspace_xz;
				b = (vz * (double)t + vx * gridlines_xz) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
			}
			for(t = -gridlines_xy;grid_xy && t <= gridlines_xy;t++)
			{
				a = (vx * (double)t + vy * -gridlines_xy) * gridspace_xy;
				b = (vx * (double)t + vy * gridlines_xy) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
				a = (vy * (double)t + vx * -gridlines_xy) * gridspace_xy;
				b = (vy * (double)t + vx * gridlines_xy) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
			}
			for(t = -gridlines_zy;grid_zy && t <= gridlines_zy;t++)
			{
				a = (vz * (double)t + vy * -gridlines_zy) * gridspace_zy;
				b = (vz * (double)t + vy * gridlines_zy) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
				a = (vy * (double)t + vz * -gridlines_zy) * gridspace_zy;
				b = (vy * (double)t + vz * gridlines_zy) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vx * 10,m,anim,xaxecolor);//QColor(200,0,0)

			drawLine(vx * 10 + vx + vz,vx * 10 - vz,m,anim,letterxcolor);//QColor(100,0,0));
			drawLine(vx * 10 + vx - vz,vx * 10 + vz,m,anim,letterxcolor);

			drawLine(vm,vy * 10,m,anim,yaxecolor);//QColor(0,200,0)

			drawLine(vy * 10 + vy + vx,vy * 10 + vy * 0.5,m,anim,letterycolor);//QColor(0,100,0));
			drawLine(vy * 10 +      vx,vy * 10 + vy * 0.5,m,anim,letterycolor);
			drawLine(vy * 10 + vy * 0.5,vy * 10 + vy * 0.5 - vx,m,anim,letterycolor);

			drawLine(vm,vz * 10,m,anim,zaxecolor);//QColor(100,0,200)

			drawLine(vz * 10 + vz + vx,vz * 10 + vx,m,anim,letterzcolor);//QColor(50,0,100));
			drawLine(vz * 10 + vz - vx,vz * 10 - vx,m,anim,letterzcolor);
			drawLine(vz * 10 + vz - vx,vz * 10 + vx,m,anim,letterzcolor);
		break;
		case PARALLEL_XY:
			for(t = -gridlines_xy;t <= gridlines_xy;t++)
			{
				a = (vx * (double)t + vy * -10) * gridspace_xy;
				b = (vx * (double)t + vy * 10) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
				a = (vy * (double)t + vx * -10) * gridspace_xy;
				b = (vy * (double)t + vx * 10) * gridspace_xy;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vx * 10,m,anim,xaxecolor);
			drawLine(vm,vy * 10,m,anim,yaxecolor);
		break;
		case PARALLEL_ZY:
			for(t = -gridlines_zy;t <= gridlines_zy;t++)
			{
				a = (vz * (double)t + vy * -10) * gridspace_zy;
				b = (vz * (double)t + vy * 10) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
				a = (vy * (double)t + vz * -10) * gridspace_zy;
				b = (vy * (double)t + vz * 10) * gridspace_zy;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vz * 10,m,anim,zaxecolor);
			drawLine(vm,vy * 10,m,anim,yaxecolor);
		break;
		case PARALLEL_XZ:
			for(t = -gridlines_xz;t <= gridlines_xz;t++)
			{
				a = (vx * (double)t + vz * -10) * gridspace_xz;
				b = (vx * (double)t + vz * 10) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
				a = (vz * (double)t + vx * -10) * gridspace_xz;
				b = (vz * (double)t + vx * 10) * gridspace_xz;
				drawLine(a,b,m,anim,gridcolor);
			}

			drawLine(vm,vz * 10,m,anim,zaxecolor);
			drawLine(vm,vx * 10,m,anim,xaxecolor);
		break;
	}

	qp->setPen(drawcolor);
	
	if(worldptr)
		worldptr->draw(this,m,anim);
	else
		printf("no world!\n");

	flush();
}

void		ViewWidget::setWorld(world *ww)
{
	view	*v;
	
	v = ww->getView(0);
	
	if(v)
	{
		ww->removeView(v);
		take(v);

		delete v;
	}
	view::setWorld(ww);
	
	draw();
}

void		ViewWidget::rotateView(double x,double y,double z)
{
//	rotate(Vector3(x,y,z));
	
	Vector3		vr(x,y,z);
	Matrix44	m;
	Vector4		v,u,r;

	m.rotateVector(vr);
	v = vview;
	u = vup;
	r = vright;

	v *= m;
	u *= m;
	r *= m;

	vview = v;
	vup = u;
	vright = r;
	
	calcVectors(0);
}

void		ViewWidget::rotate0View(double x,double y,double z)
{
//	rotateView(x,y,z);
//	calcVectors(0);
	
	Vector3		vr(x,y,z);
	Matrix44	m;
	Vector4		v4;

	m.rotateVector(vr);
	v4 = peye;
	v4 *= m;
	peye = v4;

	rotateView(x,y,z);
}

void		ViewWidget::moveView(double x,double y,double z)
{
//	translate(Vector3(x,y,z));

	Vector3		v(x,y,z);

	setEye(peye + v);
	
	calcVectors(0);
}

void		ViewWidget::changeFOV(double d)
{
	setFOV(fov + d);
}

void		ViewWidget::setViewMode(int vm)
{
	viewmode = vm;
}

void		ViewWidget::setProjectionMode(int pm)
{
	projmode = pm;
        draw();
}

int		ViewWidget::projectionMode()
{
	return projmode;
}

void		ViewWidget::setAxeMode(int am)
{
	axemode = am;
}

void		ViewWidget::setSelected(base *bp)
{
	selectedptr = bp;
	draw();
}

void		ViewWidget::mouseMoveEvent(QMouseEvent *me)
{
	int		dx,dy;
	Vector3		v(1,1,1);

	if(projmode == PERSPECTIVE && selectedptr)
	{
		if(selectedptr->changeDragvector(this,selectedptr->getMatrix(),xo,yo,me->x(),me->y()) == 0)
		{
			draw();

			xo = me->x();
			yo = me->y();

			return;
		}
	}

	dx = me->x() - xo;
	dy = me->y() - yo;

	switch(viewmode)
	{
		case 1:
			changeFOV(((double)(dx + dy)) / 5000.0);
		break;
		case 2:
			if(me->state() == LeftButton)
				moveView((double)dx / 10.0,(double)dy / 10.0,0);
			else
				moveView(0,(double)dy / 10.0,(double)dx / 10.0);
		break;
		case 3:
			if(me->state() == LeftButton)
				rotate0View((double)dx / 1800 * PI,0,0);
			else if(me->state() == RightButton)
				rotate0View(0,0,(double)dx / 1800 * PI);
			else
				rotate0View(0,(double)dx / 1800 * PI,0);
		break;
		case 4: // Object scale
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;

			if(axemode & 1)
				v(0) = exp((double)dx / -100);
			if(axemode & 2)
				v(1) = exp((double)dx / -100);
			if(axemode & 4)
				v(2) = exp((double)dx / -100);
			((dim*)selectedptr)->scale(v);
			draw();
		break;
		case 5: // Object translate
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;
			if(me->state() == LeftButton)
				((dim*)selectedptr)->translate(Vector3((double)dx / 10.0,(double)dy / 10.0,0));
			else
				((dim*)selectedptr)->translate(Vector3(0,(double)dy / 10.0,(double)dx / 10.0));
			draw();
		break;
		case 6: // Object rotate
			if(!selectedptr || selectedptr->getType() == NUM_WORLD) return;
			if(axemode & 1)
				((dim*)selectedptr)->rotate(Vector3((double)dx / 1800 * PI,0,0));
			if(axemode & 2)
				((dim*)selectedptr)->rotate(Vector3(0,0,(double)dx / 1800 * PI));
			if(axemode & 4)
				((dim*)selectedptr)->rotate(Vector3(0,(double)dx / 1800 * PI,0));
			draw();
		break;
	}
	draw();

	xo = me->x();
	yo = me->y();
}

void		ViewWidget::mousePressEvent(QMouseEvent *me)
{
	xo = me->x();
	yo = me->y();
}


void		ViewWidget::paintEvent(QPaintEvent*)
{
	setScreenSize(width(),height());
	createScreen(width(),height());

	qp->end();
	qpm->resize(width(),height());
	qp->begin(qpm);

	draw();
	
	emit resizedView(width(),height());
}


void		ViewWidget::flush()
{
	QPainter	paint(this);

	qp->end();
	paint.drawPixmap(0,0,*qpm);
	qp->begin(qpm);
}

void		ViewWidget::startRendering(int quality)
{
	char		str[16];
	QString		qstr,tmp;
	Preferences	pref;

	killTimers();

	qstr = "Start rendering with ";
	qstr += pref.getRenderProgram();
	emit report(qstr);
	
	tmp = pref.getTempPath();

	qstr = tmp + "/tmp.pov";
	emit exportPOV(qstr);

//	qstr = "rm -f " + tmp + "/tmp.ppm";
//	system(qstr);

	proc.clearArguments();

	proc << pref.getRenderProgram();
	sprintf(str,"+W%i",width());
	proc << str;
	sprintf(str,"+H%i",height());
	proc << str << "-GA";
	qstr = "-O" + tmp + "/tmp.ppm";
	proc << qstr;
	qstr = "-I" + tmp + "/tmp.pov";
	proc << qstr << "+FP";
	qstr.sprintf("-Q%i",quality);
	proc << qstr;
	
	proc.start(KProcess::DontCare,KProcess::Stdout);

	x = y = 0;

	tcounter = 0;
	ppmfp = 0;
	lc = 0;

//	sleep(1);
	startTimer(50);
}

void		ViewWidget::stopRendering()
{
	QString		qstr,tmp;
	Preferences	pref;
	
	if(!ppmfp) return;

	qstr = "Ready";
	emit report(qstr);
	
	killTimers();
	proc.kill();
	fclose(ppmfp);
	flush();
	ppmfp = 0;
	

	killTimers();
	
	tmp = pref.getTempPath();
	qstr = "rm -f " + tmp + "/tmp.ppm";
	system(qstr);
}

void		ViewWidget::timerEvent(QTimerEvent*)
{
	unsigned char	buffer[3 * 1024];
	int		i,t,r,g,b,p;
	Preferences	pref;
	QString		qstr,tmpstr;

	if(ppmfp == 0)
	{
		qstr = pref.getTempPath();

		tcounter++;
		// 400 * 50ms = 20000ms = 20sec timeout
		if(tcounter > 400)
		{
			stopRendering();
			return;
		}
		ppmfp = fopen(qstr + "/tmp.ppm","rb");
		if(ppmfp == 0) return;
//		sleep(1);
//		printf("File opened\n");
		fseek(ppmfp,0,SEEK_SET);
	}
	if(lc < 3)
	{
//		printf("Try new line\n");
		p = ftell(ppmfp);
		fseek(ppmfp,0,SEEK_END);
//		printf("diff %i - %i = %i\n",ftell(ppmfp),p,ftell(ppmfp) - p);
		if(ftell(ppmfp) <= p)
		{
			fseek(ppmfp,p,SEEK_SET);
			return;
		}
		fseek(ppmfp,p,SEEK_SET);
		
		do
		{
			fread(buffer,1,1,ppmfp);
//			printf("%c",buffer[0]);
		}
		while(buffer[0] != '\n');
		lc++;
//		printf("\nNew Line\n");
		return;
	}

	p = ftell(ppmfp);
	fseek(ppmfp,0,SEEK_END);
	if(ftell(ppmfp) <= p)
	{
		fseek(ppmfp,p,SEEK_SET);
		return;
	}
	fseek(ppmfp,p,SEEK_SET);
	
	i = fread(buffer,3,1024,ppmfp);

	for(t = 0;t < i;t++)
	{
		r = buffer[t * 3];
		g = buffer[t * 3 + 1];
		b = buffer[t * 3 + 2];
		qp->setPen(QColor(r,g,b));
		qp->drawPoint(x++,y);
		if(x >= width())
		{
			x = 0;
			y++;
			if(y >= height())
			{
				stopRendering();
				return;
			}
		}
	}
	
	i = fread(buffer,3,1024,ppmfp);

	for(t = 0;t < i;t++)
	{
		r = buffer[t * 3];
		g = buffer[t * 3 + 1];
		b = buffer[t * 3 + 2];
		qp->setPen(QColor(r,g,b));
		qp->drawPoint(x++,y);
		if(x >= width())
		{
			x = 0;
			y++;
			if(y >= height())
			{
				stopRendering();
				return;
			}
		}
	}
	flush();

	tmpstr.setNum(((x + y * width()) * 100) / (width() * height()));
	qstr = "Rendered ";
	qstr += tmpstr + "%";
	emit report(qstr);
}

int		ViewWidget::setName(const char *n)
{
	return view::setName(n);
}

void		ViewWidget::reconfigure()
{
	configure();
	draw();
}

void		ViewWidget::configure()
{
	Preferences	prefs;

	backcolor = prefs.getBackgroundColor(); 
	drawcolor = prefs.getLinesColor(); 
	selcolor = prefs.getSelectedLinesColor(); 
	boxcolor = prefs.getBoxColor(); 

	gridcolor = prefs.getGridColor(); 

	xaxecolor = prefs.getXAxisColor();
	yaxecolor = prefs.getYAxisColor();
	zaxecolor = prefs.getZAxisColor();

	letterxcolor = prefs.getLetterXColor();
	letterycolor = prefs.getLetterYColor();
	letterzcolor = prefs.getLetterZColor();

	blobcolor = prefs.getBlobColor();

	setAspectRatio(prefs.getAspectRatio());

	gridlines_xy = prefs.getGridLinesXY();
	gridlines_xz = prefs.getGridLinesXZ();
	gridlines_zy = prefs.getGridLinesZY();
	gridpoints_xy = prefs.getGridPointsXY();
	gridpoints_xz = prefs.getGridPointsXZ();
	gridpoints_zy = prefs.getGridPointsZY();
	gridspace_xy = prefs.getGridSpaceXY();
	gridspace_xz = prefs.getGridSpaceXZ();
	gridspace_zy = prefs.getGridSpaceZY();
	grid_xy = prefs.getGridXY();
	grid_xz = prefs.getGridXZ();
	grid_zy = prefs.getGridZY();
}



