/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** triangle.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "triangle.h"
#include "num.h" 
#include "view.h" 

triangle::triangle(base *p,char *n,
	double x1,double y1,double z1,
	double x2,double y2,double z2,
	double x3, double y3,double z3,
	int s,texture *t) :
		nonsolid(p,n,t)
{
	dragvector	*dv;

	numtype = NUM_TRIANGLE;
	edge1 = Vector3(x1,y1,z1);
	edge2 = Vector3(x2,y2,z2);
	edge3 = Vector3(x3,y3,z3);
	n1 = Vector3(0,0,0);
	n2 = Vector3(0,0,0);
	n3 = Vector3(0,0,0);
	smooth=s;
	if(parent) addToParent(parent);

	dv = new dragvector(edge1,&edge1);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);

	dv = new dragvector(edge2,&edge2);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

triangle::triangle(base *p,char *n,Vector3 &v1,Vector3 &v2,Vector3 &v3,texture *t) :
	nonsolid(p,n,t)
{
	dragvector	*dv;

	numtype = NUM_TRIANGLE;
	edge1 = v1;
	edge2 = v2;
	edge3 = v3;
	n1 = Vector3(0,0,0);
	n2 = Vector3(0,0,0);
	n3 = Vector3(0,0,0);
	smooth = 0;

	if(parent) addToParent(parent); 

	dv = new dragvector(edge1,&edge1);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);

	dv = new dragvector(edge2,&edge2);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

triangle::triangle(base *p,char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	nonsolid(p,n,vs,vr,vt)
{
	dragvector	*dv;

	numtype = NUM_TRIANGLE;
	edge1 = Vector3(1,0,0);
	edge2 = Vector3(0,1,0);
	edge3 = Vector3(0,0,1);
	n1 = Vector3(0,0,0);
	n2 = Vector3(0,0,0);
	n3 = Vector3(0,0,0);
	smooth = 0;

	if(parent) addToParent(parent); 

	dv = new dragvector(edge1,&edge1);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);

	dv = new dragvector(edge2,&edge2);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

triangle::triangle(base *p,char *n,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	nonsolid(p,n,t,vs,vr,vt)
{
	dragvector	*dv;

	numtype = NUM_TRIANGLE;
	edge1 = Vector3(1,0,0);
	edge2 = Vector3(0,1,0);
	edge3 = Vector3(0,0,1);
	n1 = Vector3(0,0,0);
	n2 = Vector3(0,0,0);
	n3 = Vector3(0,0,0);
	smooth = 0;

	if(parent) addToParent(parent); 

	dv = new dragvector(edge1,&edge1);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);

	dv = new dragvector(edge2,&edge2);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

triangle::triangle(base *p,triangle *tc) :
	nonsolid(p,(nonsolid*)tc)
{
	dragvector	*dv;

	numtype = NUM_TRIANGLE;
	edge1 = tc->edge1;
	edge2 = tc->edge2;
	edge3 = tc->edge3;
	n1 = tc->n1;
	n2 = tc->n2;
	n3 = tc->n3;
	smooth = tc->smooth;

	if(parent) addToParent(parent); 

	dv = new dragvector(edge1,&edge1);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);

	dv = new dragvector(edge2,&edge2);
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

triangle::~triangle()
{
	removeFromParent();
}

base	*triangle::copy(base *p)
{
	return new triangle(p,this);
}

int	triangle::setNormal(Vector3 v1, Vector3 v2, Vector3 v3,int s)
{
	n1=v1;
	n2=v2;
	n3=v3;
	smooth=s;

	return 0;
}

int	triangle::setNormal(double x1, double y1, double z1, double x2, double y2, double z2, double x3, double y3, double z3, int s)
{
	n1 = Vector3(x1,y1,z1);
	n2 = Vector3(x2,y2,z2);
	n3 = Vector3(x3,y3,z3);
	smooth=s;

	return 0;
}

int	triangle::addToParent(base *p)
{
	if (!p) return -2;
	parent = p;
	return p->addChild(this);
}

int triangle::removeFromParent() 
{ 
	if (!parent) return -2;
	return parent->removeChild(this); 
}

void triangle::dumpNames(int tab, int)
{
	printTab(stderr,tab);
	printf("triangle: %s\n",name);
}

int	triangle::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = triangle\n");

	printTab(fp,tab);
	if(!smooth)
	{
		fprintf(fp,"triangle {\n");
	}
	else
	{
		fprintf(fp,"smooth_triangle {\n");
	}

	printTab(fp,tab+tabsize);
	fprintf(fp,"<%g, %g, %g>, ",edge1[0],edge1[1],edge1[2]);
	if(smooth)
	{
		fprintf(fp,"<%f, %f, %f>, ",n1[0],n1[1],n1[2]);
	}
	fprintf(fp,"\n");

	printTab(fp,tab+tabsize);
	fprintf(fp,"<%g, %g, %g>, ",edge2[0],edge2[1],edge2[2]);
	if(smooth)
	{
		fprintf(fp,"<%f, %f, %f>, ",n1[0],n1[1],n1[2]);
	}
	fprintf(fp,"\n");

	printTab(fp,tab+tabsize);
	fprintf(fp,"<%g, %g, %g> ",edge3[0],edge3[1],edge3[2]);
	if(smooth)
	{
		fprintf(fp,", <%g, %g, %g> ",n1[0],n1[1],n1[2]);
	}
	fprintf(fp,"\n");

	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	dim::exportPOV(fp,tab,tabsize,anim);

	printTab(fp,tab);
	fprintf(fp,"}\n\n");	

	return 0;
}

int triangle::save(media *m)
{
	if(!m) return -1;
	setMedia(m);
	
	writeChunk("TRIA"); 
	writeNameChunk(name);

	saveFlags(m);

	writeVector(edge1);
	writeVector(edge2);
	writeVector(edge3);

	if(smooth)
	{
		writeInt(1);
		writeVector(n1);
		writeVector(n2);
		writeVector(n3);
	}
	else
	{
		writeInt(0); 
	}

	anim::save(m);
	dim::save(m);

	saveTexture(m);
	writeChunkLen();

	return 0;
}

int triangle::load(media *m,int l)
{
	int	pos = m->tell();

	loadFlags(m,l);

	edge1 = readVector();
	edge2 = readVector();
	edge3 = readVector();
	smooth = readInt();
	if(smooth)
	{
		n1 = readVector();
		n2 = readVector();
		n3 = readVector();
	}

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

 	if(l > (m->tell() - pos))
	{
		loadTexture(m,l - (m->tell() - pos));
	}

	return 0;
}

int	triangle::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p1,p2,p3;

	//printf("Triangle\n");

	if(isFlag(HIDE)) return 0;

	p1 = edge1;
	p2 = edge2;
	p3 = edge3;

	dimMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		v->drawLine(p1,p2,m,anim);
		v->drawLine(p2,p3,m,anim);
		v->drawLine(p3,p1,m,anim);
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	triangle::calculate(int)
{
	vmin[0] = MIN3(edge1[0],edge2[0],edge3[0]);
	vmin[1] = MIN3(edge1[1],edge2[1],edge3[1]);
	vmin[2] = MIN3(edge1[2],edge2[2],edge3[2]);
	vmax[0] = MAX3(edge1[0],edge2[0],edge3[0]);
	vmax[1] = MAX3(edge1[1],edge2[1],edge3[1]);
	vmax[2] = MAX3(edge1[2],edge2[2],edge3[2]);

	return 0;
}


