 
#include "main.h"

#include "worm.h"
#include "playground.h"

#include <string.h>
#include <qsize.h>

#include <stdio.h>

#define L_WORM_ADD_LEN 3

#define WORMPOINT(p) LPtoDP(p) + QPoint(FDWIDTH / 2,FDHEIGHT / 2)

CWorm::CWorm(CWormPlayground* pg)
{
	_playground = pg;
	_points = new CPointList;
	_points->setAutoDelete(true);
	QObject::connect(_playground,SIGNAL(sigTimer(long)),this,SLOT(slotTimer(long)));
	QObject::connect(_playground,SIGNAL(sigStart()),this,SLOT(slotStart()));
	QObject::connect(_playground,SIGNAL(sigRestart()),this,SLOT(slotStart()));
	QObject::connect(_playground,SIGNAL(sigDie(char)),this,SLOT(slotDie(char)));
	QObject::connect(_playground,SIGNAL(sigNewDir(char)),this,SLOT(slotNewDir(char)));
	_dirList = new QQueue<char>;
	_dirList->setAutoDelete(true);
}

CWorm::~CWorm()
{
	delete _points;
	delete _dirList;
}

QPoint CWorm::LPtoDP(QPoint pt) 
{ 
	return _playground->LPtoDP(pt); 
}

QRect CWorm::LPtoDP(QRect rct) 
{ 
	return _playground->LPtoDP(rct); 
}

QPoint CWorm::DPtoLP(QPoint pt) 
{ 
	return _playground->DPtoLP(pt); 
}

QRect CWorm::DPtoLP(QRect rct) 
{ 
	return _playground->DPtoLP(rct); 
}

#define POINT_BETWEEN(p,a,b) (((p).x() == (a).x() && (((p).y() <= (a).y() && (b).y() <= (p).y()) || \
        ((p).y() >= (a).y() && (b).y() >= (p).y()))) || ((p).y() == (a).y() && (((p).x() <= (a).x() && (b).x() <= (p).x()) || \
        ((p).x() >= (a).x() && (b).x() >= (p).x()))))

WPoint* CWorm::covers(QPoint field)
{
	WPoint prev;
	CPointListIterator it(*_points);
	prev = _tail;
	if (field == _tail) return &_tail;
	for (it.toFirst();it.current();++it) {
		if (it.current()->d() != DirTeleport && POINT_BETWEEN(field,prev,*it.current())) {
			return it.current();
		}
		prev = *it.current();
	}
	if POINT_BETWEEN(field,head(),prev) {
		return &_head;
	}

	return NULL;
}

#undef POINT_BETWEEN

bool CWorm::isActive()
{ 
	return _playground->runMode() == RM_RUNNING || _playground->runMode() == RM_PAUSED; 
}


void CWorm::redrawHead(QPoint prevHead)
{
	QRect rct(LPtoDP(head()),QSize(FDWIDTH,FDHEIGHT));
	QRect pRct(LPtoDP(prevHead),QSize(FDWIDTH,FDHEIGHT));
	rct = rct.unite(pRct);

	emit sigSpecialUpdate(DPtoLP(rct));
}

void CWorm::redrawTail(QPoint prevTail)
{
	QRect rct(LPtoDP(_tail),QSize(FDWIDTH,FDHEIGHT));
	QRect pRct(LPtoDP(prevTail),QSize(FDWIDTH,FDHEIGHT));
	rct = rct.unite(pRct);
	emit sigSpecialUpdate(DPtoLP(rct));
}

void CWorm::slotStart()
{
	_points->clear();
	WPoint npt(START_X,START_Y,WormUp);
	_head = npt;
	_tail = npt;
	_dirList->clear();
	_lastDir = DirUndef;
	_length = 0;
	_reqLength = 6;
}

void CWorm::slotNewDir(char d)
{
	if (playground()->runMode() == RM_RUNNING) addDir(d);
}

void CWorm::slotDie(char reason)
{
	// Nice animation maybe?
}

void CWorm::slotTimer(long c)
{
	if (!(c % TIMER_COUNT_WORM_GROW)) _reqLength++;
	if (c % TIMER_COUNT_WORM) return;

	QPoint pNext(head());
	QPoint pPrev(head());

	getDir();
	switch(dir()) {
	case WormUp: pNext += QPoint(0,-1); break;
	case WormDown: pNext += QPoint(0,1); break;
	case WormLeft: pNext += QPoint(-1,0); break;
	case WormRight: pNext += QPoint(1,0); break;
	default: ASSERT(false);
	}
	_length++;
	if (playground()->wormAccess(pNext.x(),pNext.y())) return;
	if (playground()->fieldState(pNext) == FieldCutter) {
		_reqLength = 6;
		
	}
	if (playground()->fieldState(pNext) == FieldTeleport) {
		playground()->maze()->onTeleport(pNext);
	} else {
		_head = pNext;
	}

	while (_length >= _reqLength) shorten();
	redrawHead(pPrev);
}

void CWorm::slotEatBall(char s)
{
	_reqLength += L_WORM_ADD_LEN;
}

void CWorm::draw(QPainter* pt)
{
	CPointListIterator it(*_points);
	it.toFirst();
	
	WPoint* current = &_tail;
	WPoint* last = NULL;
	int i;

	do {
		if (current->d() == DirTeleport || !last) drawAt(*current,playground(),LPtoDP(*current));
		else switch(current->d()) {
		case WormUp: 
			for(i = last->y(); i >= current->y(); i--) 
				drawAt(QPoint(current->x(),i),playground(),LPtoDP(QPoint(current->x(),i)));
			break;
		case WormDown:
			for(i = last->y(); i <= current->y(); i++) 
				drawAt(QPoint(current->x(),i),playground(),LPtoDP(QPoint(current->x(),i)));
			break;
		case WormLeft:
			for(i = last->x(); i >= current->x(); i--) 
				drawAt(QPoint(i,current->y()),playground(),LPtoDP(QPoint(i,current->y())));
			break;
		case WormRight:
			for(i = last->x(); i <= current->x(); i++) 
				drawAt(QPoint(i,current->y()),playground(),LPtoDP(QPoint(i,current->y())));
			break;
		}

		if (current == &_head) break; 
		last = current;
		current = it.current();
		if (current != NULL) ++it;
		else current = &_head;
		
	} while (true);
}

void CWorm::drawAt(QPoint field,QPaintDevice* dst,QPoint dstPoint)
{
	QPixmap* thePxm = NULL;

	if (field == head()) {
		switch(head().d()) {
		case WormUp: bitBlt(dst,dstPoint,_WormHeadUp); break;
		case WormDown: bitBlt(dst,dstPoint,_WormHeadDown); break;
		case WormLeft: bitBlt(dst,dstPoint,_WormHeadLeft); break;
		case WormRight: bitBlt(dst,dstPoint,_WormHeadRight); break;
		}
		return;
	}

	if (field == tail()) {
		switch(tail().d()) {
		case WormUp: bitBlt(dst,dstPoint,_WormTailDown); break;
		case WormDown: bitBlt(dst,dstPoint,_WormTailUp); break;
		case WormLeft: bitBlt(dst,dstPoint,_WormTailRight); break;
		case WormRight: bitBlt(dst,dstPoint,_WormTailLeft); break;
		}
		return;
	}


	WPoint* p = covers(field);
	if (!p) {
		return;
	}
	
	if (*p != field) {
		// field is not an edge
		switch (p->d()) {
		case WormUp:
		case WormDown:
			bitBlt(dst,dstPoint,_WormVrt);
			break;
		case WormRight:
		case WormLeft:
			bitBlt(dst,dstPoint,_WormHrz);
			break;
		}
		return;
	} else {
		_points->findRef(p);
		ASSERT(p);
		WPoint* next = _points->next(); // Required to determine the type of the next edge
		if (!next) next = &_head;

		unsigned int dirChange = (p->d() << 4) | next->d();

		switch (dirChange) {
		case 0x34: 
		case 0x21: thePxm = _WormEdgeLR; break;
			
		case 0x14:
		case 0x23: thePxm = _WormEdgeUR; break;
			
		case 0x32:
		case 0x41: thePxm = _WormEdgeLL; break;
			
		case 0x43:
		case 0x12: thePxm = _WormEdgeUL; break;
			
                // next direction is DirTeleport -> _WormTeleportXX for current field
		case 0x16: thePxm = _WormTeleportUp; break;
		case 0x26: thePxm = _WormTeleportRight; break;
		case 0x36: thePxm = _WormTeleportDown; break;
		case 0x46: thePxm = _WormTeleportLeft; break;

		// last direction is DirTeleport -> _WormTeleportExitXX for current field
		case 0x61: thePxm = _WormTeleportExitUp; break;
		case 0x62: thePxm = _WormTeleportExitRight; break;
		case 0x63: thePxm = _WormTeleportExitDown; break;
		case 0x64: thePxm = _WormTeleportExitLeft; break;
		}
	
		bitBlt(dst,dstPoint,thePxm);
		return;
	}
}


void CWorm::addDir(char dir)
{
	ASSERT(dir);
	if (dir == _lastDir) return;
	_lastDir = dir;
	char* d2 = new char;
	*d2 = dir;
	_dirList->enqueue(d2);
}

void CWorm::getDir()
{
	if (head() == WPoint(START_X,START_Y,WormUp)) return;
	char* d2 =_dirList->dequeue();
	if (!d2) return;
	if ((dir() - *d2) % 2) { // *d2 is orthogonal to _dir
		newPoint(head());
		_head.setDir(*d2);
	}
	delete d2;
}

void CWorm::shorten()
{
  	WPoint nextPt;
	//	if (!_points->isEmpty() && _tail == *_points->getFirst()) debug("Tail-Problem");

	if (_points->isEmpty()) nextPt = head();
	else nextPt = *_points->getFirst();

	WPoint prevTail = _tail;

	if (_tail.d() % 2) _tail.setY(_tail.y() > nextPt.y() ? _tail.y() - 1 : _tail.y() + 1);
	else _tail.setX(_tail.x() > nextPt.x() ? _tail.x() - 1 : _tail.x() + 1);
 	_length--;

	if (!_points->isEmpty() && _tail == *_points->getFirst()) {
		_points->removeFirst(); 
		if (!_points->isEmpty() && _points->getFirst()->d() == DirTeleport) {
			// Set _tail and remove as well
			_tail = *_points->getFirst();
			_points->removeFirst();
		}
		_tail.setDir(_points->isEmpty() ? _head.d() : _points->getFirst()->d());
	}

	redrawTail(prevTail);


}

