
#include "main.h"
#include "maze.h"

#include "playground.h"
#include "ball.h"

#include <kapp.h>
#include <qfile.h>

#include <malloc.h>
#include <string.h>
#include <stdlib.h>

// Constants passed to slotTimer for special updates
#define UDL_EXIT -1

CMaze::CMaze(CWormPlayground* pg)
{
	_playground = pg;
	connect(_playground,SIGNAL(sigStart()),this,SLOT(slotStart()));
	connect(_playground,SIGNAL(sigRestart()),this,SLOT(slotStart()));
	connect(_playground,SIGNAL(sigDie(char)),this,SLOT(slotTerminate(char)));
	connect(_playground,SIGNAL(sigTimer(long)),this,SLOT(slotTimer(long)));

	//	connect(_playground->worm(),SIGNAL(sigAccessField(int,int)),this,SLOT(slotAccessField(int,int)));
	connect(this,SIGNAL(sigEatBall(char)),this,SLOT(slotEatBall(char)));
	
	_data = (byte*) malloc(PGWIDTH * PGHEIGHT);
	if (!_data) fatal("Memory allocation failed.");
	_allocSpace = PGWIDTH * PGHEIGHT;

	_balls.setAutoDelete(true);

	clear();
	_active = false;
}

CMaze::~CMaze()
{
	free(_data);
}

/*inline*/ QPoint CMaze::LPtoDP(QPoint pt)
{ 
	return _playground->LPtoDP(pt); 
}

/*inline*/ QRect CMaze::LPtoDP(QRect rct)
{ 
	return _playground->LPtoDP(rct); 
}

/*inline*/ QPoint CMaze::DPtoLP(QPoint pt)
{ 
	return _playground->DPtoLP(pt); 
}

/*inline*/ QRect CMaze::DPtoLP(QRect rct)
{ 
	return _playground->DPtoLP(rct); 
}

void CMaze::clear()
{
	ASSERT(_allocSpace == PGWIDTH * PGHEIGHT);
	memset(_data,0,PGWIDTH * PGHEIGHT);
	
	_balls.clear();
}

char CMaze::accessible(QPoint field,char dir) // returns the same values as CWormPlayground::fieldState
{
	// dir may be undefined (255)
	if (field.x() <= 0 || field.x() >= (PGWIDTH - 1) || field.y() <= 0 || field.y() >= (PGHEIGHT - 1))
		{ return FieldBorder; }
	byte dvalue = _data [field.x() + field.y() * PGWIDTH];
	switch (dvalue) { 
	case DATA_WALL_1:
	case DATA_WALL_2:
	case DATA_WALL_3: return FieldWall;
	case DATA_SEPARATOR:
	case DATA_SEPARATOR_VISIBLE: return FieldSeparator;
	case DATA_CUTTER: return FieldCutter;
	case DATA_OW_RIGHT: 
		return (dir == WormRight ? FieldFreeOW : FieldOW);
	case DATA_OW_LEFT: 
		return (dir == WormLeft ? FieldFreeOW : FieldOW);
	case DATA_OW_UP: 
		return (dir == WormUp ? FieldFreeOW : FieldOW);
	case DATA_OW_DOWN: 
		return (dir == WormDown ? FieldFreeOW : FieldOW);
	case DATA_EXIT_1: return (_permExit ? FieldOpenExit : FieldExit);
	case DATA_EXIT_1_OPEN: fatal("AIA"); // This is obsolete, I hope
	default:
		if (dvalue >= DATA_TELEPORT_B && dvalue <= DATA_TELEPORT_C) return FieldTeleport;
	}
	QListIterator<CBall> ballIt(_balls);
	for (ballIt.toFirst();ballIt.current();++ballIt) 
		if (ballIt.current()->pos() == field) return FieldBall;
	return FieldEmpty;
}

QPoint CMaze::getCorrespondingExit(QPoint startPos)
{
	unsigned char dvalue = _data[startPos.y() * PGWIDTH + startPos.x()];
	ASSERT(dvalue >= DATA_TELEPORT_B && dvalue <= DATA_TELEPORT_C);

	int i,j;
	for (i = 0;i < PGHEIGHT;i++)
		for (j = 0;j < PGWIDTH;j++)
			if (_data[i * PGWIDTH + j] == dvalue + TELEPORT_EXIT_DIFF) 
				return QPoint(j,i);

	fatal("CMaze::getCorrespondingExit: exit not found"); // Fault of the level designer
	return QPoint(0,0);
}

void CMaze::redraw(QRect* rct)
{
	// !rct => update entire playground
	QRect dprct;
	if (rct) dprct = LPtoDP(*rct);
	else dprct.setRect(0,0,PGWIDTH*FDWIDTH,PGHEIGHT*FDHEIGHT);
	emit sigUpdate(dprct);
}

void CMaze::draw(QPainter* pt)
{
	int i,j;
	if (!_active) {
// 		fatal("CMaze::draw called when !_active");
// 		QPixmap pxm("kworm1.xpm");
// 		pt->drawPixmap(20,20,pxm);
		return;
	}
// 	QRect rct;
// 	QPen pn(red);
// 	for (i = 1;i <= PGWIDTH - 2;i++) pt->drawPixmap(LPtoDP(QPoint(i,0)),*_BorderHrz);
// 	for (i = 1;i <= PGWIDTH - 2;i++) pt->drawPixmap(LPtoDP(QPoint(i,PGHEIGHT - 1)),*_BorderHrz);
// 	for (i = 1;i <= PGHEIGHT - 2;i++) pt->drawPixmap(LPtoDP(QPoint(0,i)),*_BorderVrt);
// 	for (i = 1;i <= PGHEIGHT - 2;i++) pt->drawPixmap(LPtoDP(QPoint(PGWIDTH - 1,i)),*_BorderVrt);

//  	pt->drawPixmap(LPtoDP(QPoint(0,0)),*_BorderUL);
//  	pt->drawPixmap(LPtoDP(QPoint(PGWIDTH - 1,0)),*_BorderUR);
//  	pt->drawPixmap(LPtoDP(QPoint(0,PGHEIGHT - 1)),*_BorderLL);
//  	pt->drawPixmap(LPtoDP(QPoint(PGWIDTH - 1,PGHEIGHT - 1)),*_BorderLR);

// 	// Weiter
// 	pt->setPen(pn);
// 	for (i=0;i <= PGHEIGHT - 1;i++)
// 		for (j=0;j <= PGWIDTH - 1;j++) {
// 			QPixmap* thePxm;
// 			rct = LPtoDP(QRect(j,i,2,2));
// 			/* if (_data [j + i * PGWIDTH] == DATA_WALL) 
// 			   pt->drawPixmap(rct.topLeft(),*_Wall01);*/
// 			unsigned char dvalue = _data [j + i * PGWIDTH];
// 			switch (dvalue) {
// 			case DATA_NONE: thePxm = NULL; break;
// 			case DATA_WALL_1: thePxm = _Wall01; break;
//      			case DATA_WALL_2: thePxm = _Wall02; break;
// 			case DATA_WALL_3: thePxm = _Wall03; break;

// 			case DATA_OW_RIGHT: thePxm = _OW_Right; break;
// 			case DATA_OW_LEFT: thePxm = _OW_Left; break;
// 			case DATA_OW_UP: thePxm = _OW_Up; break;
// 			case DATA_OW_DOWN: thePxm = _OW_Down; break;

// 			case DATA_EXIT_1: thePxm = (_permExit ? _Exit01Open : _Exit01); break;

// 			default: 
// 				if (dvalue >= DATA_TELEPORT_B && dvalue <= DATA_TELEPORT_C) {
// 					thePxm = _Tele01; break;
// 				}
// 				if (dvalue >= DATA_TELEPORT_EX_B && dvalue <= DATA_TELEPORT_EX_C) {
// 					thePxm = _Tele01Open; break;
// 				}
// 			}
// 			if (thePxm) pt->drawPixmap(rct.topLeft(),*thePxm);
// 		}
	for (j = 0;j < PGHEIGHT;j++)
		for (i = 0;i < PGWIDTH;i++)
			drawAt(QPoint(i,j),playground(),LPtoDP(QPoint(i,j)));

	QListIterator<CBall> ballIt(_balls);
	for (ballIt.toFirst();ballIt.current();++ballIt)
		ballIt.current()->draw(pt);
}

void CMaze::drawAt(QPoint field,QPaintDevice* dst,QPoint dstPoint)
{
	if (field.x() == 0) {
		switch (field.y()) {
		case 0: bitBlt(dst,dstPoint,_BorderUL); break;
		case PGHEIGHT - 1: bitBlt(dst,dstPoint,_BorderLL); break;
		default: bitBlt(dst,dstPoint,_BorderVrt); break;
		}
		return;
	}
	if (field.x() == PGWIDTH - 1) {
		switch (field.y()) {
		case 0: bitBlt(dst,dstPoint,_BorderUR); break;
		case PGHEIGHT - 1: bitBlt(dst,dstPoint,_BorderLR); break;
		default: bitBlt(dst,dstPoint,_BorderVrt); break;
		}
		return;
	}
	if (field.y() == 0 || field.y() == PGHEIGHT - 1) {
		bitBlt(dst,dstPoint,_BorderHrz);
		return;
	}

	QPixmap* thePxm = NULL;
	unsigned char dvalue = _data [field.x() + field.y() * PGWIDTH];
	switch (dvalue) {
	case DATA_NONE: 
	case DATA_SEPARATOR: thePxm = NULL; break;
	case DATA_WALL_1: thePxm = _Wall01; break;
	case DATA_WALL_2: thePxm = _Wall02; break;
	case DATA_WALL_3: thePxm = _Wall03; break;
	case DATA_SEPARATOR_VISIBLE: thePxm = _Separator; break;
	case DATA_CUTTER: thePxm = _Cutter; break;
	case DATA_OW_RIGHT: thePxm = _OW_Right; break;
	case DATA_OW_LEFT: thePxm = _OW_Left; break;
	case DATA_OW_UP: thePxm = _OW_Up; break;
	case DATA_OW_DOWN: thePxm = _OW_Down; break;
	case DATA_EXIT_1: thePxm = (_permExit ? _Exit01Open : _Exit01); break;
	default: 
		if (dvalue >= DATA_TELEPORT_B && dvalue <= DATA_TELEPORT_C) {
			thePxm = _Tele01; break;
		}
		if (dvalue >= DATA_TELEPORT_EX_B && dvalue <= DATA_TELEPORT_EX_C) {
			thePxm = _Tele01Open; break;
		}
	}
	if (thePxm) bitBlt(dst,dstPoint,thePxm);
	
}
	
bool CMaze::load(QString group,int nr)
{
	bool status;
	char snr[50];
	QString appName = KApplication::getKApplication()->appName();
	QString datadir = KApplication::getKApplication()->kde_datadir();
	sprintf(snr,"%s/%s/%s%i.lev",(const char*) datadir,
		(const char*) appName,(const char*) group,nr);
  	QString filename(snr);
	QFile f(filename);
	status = f.open(IO_ReadOnly);
	if (!status) return false;
	status = (f.readBlock((char*)_data,1600) == 1600);
	if (!status) return false;
	_mzAdditionalBalls = (char) f.getch();
	_mzMinBalls = (char) f.getch();
	_mzPercMvBalls = (char) f.getch();
	_mzPercJmpBalls = (char) f.getch();
	f.close();
	return true;
}

void CMaze::addBall()
{
	CBall* bl;
	int ballType = GETRAND(100);
	if (ballType < _mzPercMvBalls) bl = new CMovingBall(this);
	else {
		if (ballType < _mzPercMvBalls + _mzPercJmpBalls) bl = new CJumpingBall(this,1);
		else bl = new CQuietBall(this);
	}
	QPoint pt;
	do {
		int ptx = rand() % PGWIDTH + 1;
		int pty = rand() % PGHEIGHT + 1;
		pt = QPoint(ptx,pty); 
	} while (playground()->fieldState(pt) != FieldEmpty);
	
	bl->init(&pt); // Should be able to find a good position itself
	_balls.append(bl);
// 	QRect blrct(pt,QSize(FDWIDTH,FDHEIGHT));
// 	redraw(&blrct);
}

void CMaze::slotStart()
{
	_active = true;
	_balls.clear();
	_eatenBalls = 0;
	_permExit = false;

	addBall();

// 	connect(_playground->worm(),SIGNAL(sigAccessField(int,int)),this,SLOT(slotAccessField(int,int)));
// 	connect(this,SIGNAL(sigEatBall(char)),this,SLOT(slotEatBall(char)));
}

bool CMaze::accessField(int x,int y)
{
	QListIterator<CBall> ballIt(_balls);
	for (ballIt.toFirst();ballIt.current();++ballIt) 
		if (ballIt.current()->pos() == QPoint(x,y)) {
			char sc = ballIt.current()->score();
			_balls.removeRef(ballIt.current());
			emit sigEatBall(sc);
			break;
		}
	unsigned char dvalue = _data[y * PGWIDTH + x];
	if (_permExit && dvalue == DATA_EXIT_1) {
		_playground->winningLevel();
		return true;
	}
        if (dvalue >= DATA_TELEPORT_B && dvalue <= DATA_TELEPORT_C) {
		// Moved elsewhere
// 		CWorm* w = playground()->worm();
// 		QPoint np(x,y);
// 		// Add _head as turnpoint         
// // 		w->newPoint(w->head()); // That's wrong, for _head is not yet changed
// 		w->newPoint(np,w->dir());
// 		// Add special point to indicate teleport
// 		QPoint ex(getCorrespondingExit(np));
// 		w->newPoint(ex,DirTeleport);
// 		debug("CMaze::slotAccessField: Teleport exit encountered at %i, %i",ex.x(),ex.y());
// 		w->setHead(WPoint(ex,w->dir()));
// // 		w->setTeleporting();
	}
		
	return false;
	
}

void CMaze::slotTimer(long c)
{
	// This seems to be empty, but might be a place for extensions
}

void CMaze::slotTimerBar()
{
	for (int i=1; i <= _mzAdditionalBalls;i++) {
		addBall();
	}


}

void CMaze::slotTerminate(char reason)
{
	_active = false;
}

void CMaze::slotEatBall(char score)
{
	if (!_balls.count()) { 
		_eatenBalls++;
		if (_eatenBalls < _mzMinBalls) addBall();
		else permitExit();
						       
	}
	
}

void CMaze::permitExit()
{
	_permExit = true;
	emit sigOpenExit();
	//	redraw();
	for (int i = 0;i <= 39;i++)
		for (int j = 0;j <= 39;j++)
			if (_data[i * PGWIDTH + j] == DATA_EXIT_1) drawAt(QPoint(j,i),playground(),
									  LPtoDP(QPoint(j,i)));
}

void CMaze::onTeleport(QPoint tp) // called before worm enters teleport
{
	CWorm* w = playground()->worm();
	// Add _head as turnpoint         
	w->newPoint(tp,w->dir());
	// Add special point to indicate teleport
	QPoint ex(getCorrespondingExit(tp));
	w->newPoint(ex,DirTeleport);
	w->setHead(WPoint(ex,w->dir()));
}
 
