/***************************************************************************
                          telnet.cpp
                      -------------------
    description          : Class for sending/receiving data via a telnet-connection
    begin                : Sun Jun 20 14:17:54 MEST 1999
    copyright            : (C) 1999 by Stephan Uhlmann
    email                : suhlmann@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ctelnet.h"

#include <kapp.h>
#include <qsocketnotifier.h>

#include <sys/types.h> /* for BSD compatibility */
#include <netinet/in.h>
#include <arpa/telnet.h>
#include <sys/socket.h>
/* modified 27.09.1999 11:33 for Sun Solaris 2.6 by dflinkmann@gmx.de */
#ifdef sun
#include <sys/filio.h>
#endif
/* modification for Sun Solaris 2.6 end */
#include <sys/ioctl.h>
#include <netdb.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>

#include "debug.h" // debug telnet commands



CTelnet::CTelnet()
{
	echo=true;
	tsocket=0;
	hostName="";
	hostIP=0;
	hostPort=0;
	buffer="";
	oldBuffer="";
	notifierRead=NULL;
}

CTelnet::~CTelnet()
{
	close();
	hostName="";
	hostIP=0;
	oldBuffer="";
}

/**
  * return: 0 ok, connected
  *        -1 host not found
  *        -2 cant create socket
  *        -3 connection refused
	*        -4 connection timeout
	*        -5 network unreachable
	*        -6 host unreachable (no route to host)
	*       -10 general connection failure
  */
int CTelnet::connect(QString hostname, unsigned long port, unsigned int timeout)
{
	KApplication* myApp=KApplication::getKApplication();
	struct in_addr hostip;
	struct hostent *host;
	struct sockaddr_in saddr;
	int sockid;
	int i,j;
	fd_set fdset;
	struct timeval tv;

	if (isConnected()) close(); // close existing connection first

	mccpState = mudcompress_new(); /* MCCP */

	// warning: this can block for a long time
	host = gethostbyname(hostname);
	if (!host) return -1; // host not found

	hostip=*(struct in_addr*)host->h_addr;

	// set Class vars
	hostName=hostname;
	hostIP=hostip.s_addr;
	hostPort=port;

	saddr.sin_family=AF_INET;
	saddr.sin_port=htons(port);
	saddr.sin_addr=hostip;

	// create socket (SOCK_STREAM for TCP, SOCK_DGRAM for UDP)
	sockid=socket(saddr.sin_family,SOCK_STREAM,0);
	if (sockid<0) return -2;  // cant create socket

	i=1;
	ioctl(sockid, FIONBIO, &i); // make non blocking


	// connect socket to ip
	// because socket was made nonblocking before, funtion will return immediately
	i=::connect(sockid,(struct sockaddr*)&saddr,sizeof(saddr));

	if (i != 0)
	{
		switch (errno)
		{
			case ECONNREFUSED: return -3;  // connection refused
			case ETIMEDOUT: return -4;     // connection timed out
			case ENETUNREACH: return -5;   // network unreachable
			case EHOSTUNREACH: return -6;   // no route to host
			case EINPROGRESS: break;  // if connection in progress then go ahaed and do the select stuff
			default: return -10;
		}
	}

	i=0;
	j=0;
	// call select until socket status changes or timeout occurs
	// to avoid blocking the application split the time and
	// call KApplication::processEvents() every so microseconds
	while ((i==0) && (j<(int)timeout*5))
	{
		// prepare filedescriptor set and timeval for select(2)
		// fdset and tv must be initialized every time because select(2) returns
		// them in an undefined state (see manpage)
		FD_ZERO(&fdset);
		FD_SET(sockid,&fdset);
		tv.tv_sec=0;
		tv.tv_usec=200000; // microseconds
		i=select(sockid+1, NULL, &fdset, NULL, &tv); // socket ok when writable
		j++;
		myApp->processEvents(50); // process events for x milliseconds
	}

	if (i == 0) return -4; // socket not writable -> timeout
	if (i == -1) return -10; // error in select -> give default error


	// error during connect?
	j=sizeof(i);
	getsockopt(sockid,SOL_SOCKET,SO_ERROR,&i,(socklen_t*)&j);

	if (i != 0)
	{
		switch (i)
		{
			case ECONNREFUSED: return -3;  // connection refused
			case ETIMEDOUT: return -4;     // connection timed out
			case ENETUNREACH: return -5;   // network unreachable
			case EHOSTUNREACH: return -6;   // no route to host
			default: return -10;
		}
	}


	notifierRead = new QSocketNotifier(sockid,QSocketNotifier::Read);
	QObject::connect(notifierRead, SIGNAL(activated(int)), this, SLOT(receiveData()));

	tsocket=sockid;

	return sockid;
}

void CTelnet::close()
{
	if (notifierRead!=NULL){
		QObject::disconnect(notifierRead, SIGNAL(activated(int)), this, SLOT(receiveData()));
		delete notifierRead;
		mudcompress_delete(mccpState); /* MCCP */
	}
	::shutdown(tsocket,2);
	tsocket=0;
	echo=true;
	buffer="";
	notifierRead=NULL;
	emit closedConnection();
}

int CTelnet::receiveData()
{
	int bytesReceived, bytesToWrite;
	char* tempbuf;
	int ansiOpenStart;

	tempbuf = new char[1024];
	bytesReceived = recv(tsocket, &tempbuf[0], 1024, 0);
	bytesToWrite=bytesReceived;

	if (bytesReceived==0) close();
	else {
		/* handle MCCP */
		mudcompress_receive(mccpState, &tempbuf[0], (unsigned int)bytesReceived);

		/* error? */
		if ( mudcompress_error(mccpState)) close();
		else {
			bytesToWrite = 0;
			while ( mudcompress_pending(mccpState) ) {
				bytesReceived = mudcompress_get( mccpState, &tempbuf[0], 1024 );

				tempbuf[bytesReceived]='\0';
				buffer.append(tempbuf);
				bytesToWrite+=processCommands();

				// prepend open ansi/vt100 escape sequences from last receive
				if (oldBuffer.isEmpty()==false)
				{
					buffer=oldBuffer+buffer;
					oldBuffer="";
				}

				// check for open ansi/vt100 escape sequences
				ansiOpenStart=buffer.findRev('\x1B',buffer.length());
				if (buffer.findRev('m',buffer.length()) < ansiOpenStart){
					oldBuffer=buffer.mid(ansiOpenStart,buffer.length()-ansiOpenStart);
					buffer=buffer.left(ansiOpenStart);
				}
			} 
			emit filledBuffer(buffer);  //send buffer to who ever wants it
			buffer="";     // empty Buffer
		}
	}
	delete[] tempbuf;
	return bytesToWrite;
}

int CTelnet::sendData(QString buf, int len) const
{
	const char *mccp_string=mudcompress_response(mccpState);
	if (mccp_string) send(tsocket, mccp_string, strlen(mccp_string), 0);

	return send(tsocket,buf, len, 0);
}

bool CTelnet::isEchoOn()
{
	return echo;
}

void CTelnet::setEcho(bool b)
{
	echo=b;
	emit changedEcho(b);
}

bool CTelnet::isInterpretAsCommand(unsigned char c) const
{
	return (c==IAC);
}

bool CTelnet::isCommand(unsigned char c) const
{
	switch (c) {
		case IAC	:
		case DONT	:
		case DO		:
		case WILL	:
		case WONT	:
		case SB		:
		case GA		:
		case EL		:
		case EC		:
		case AO		:
		case IP		:
		case BREAK	:
		case DM		:
		case NOP	:
		case SE		:
		case EOR	:
		case ABORT	:
		case SUSP	:
		case xEOF	: return true;break;

		default		: return false;
	}
}

bool CTelnet::isOption(unsigned char c) const
{
	switch (c) {
		case TELOPT_BINARY	:
		case TELOPT_ECHO	:
		case TELOPT_RCP		:
		case TELOPT_SGA		:
		case TELOPT_NAMS	:
		case TELOPT_STATUS	:
		case TELOPT_TM		:
		case TELOPT_RCTE	:
		case TELOPT_NAOL	:
		case TELOPT_NAOP	:
		case TELOPT_NAOCRD	:
		case TELOPT_NAOHTS	:
		case TELOPT_NAOHTD	:
		case TELOPT_NAOFFD	:
		case TELOPT_NAOVTS	:
		case TELOPT_NAOVTD	:
		case TELOPT_NAOLFD	:
		case TELOPT_XASCII	:
		case TELOPT_LOGOUT	:
		case TELOPT_BM		:
		case TELOPT_DET		:
		case TELOPT_SUPDUP	:
		case TELOPT_SUPDUPOUTPUT	:
		case TELOPT_SNDLOC	:
		case TELOPT_TTYPE	:
		case TELOPT_EOR		:
		case TELOPT_TUID	:
		case TELOPT_OUTMRK	:
		case TELOPT_TTYLOC	:
		case TELOPT_3270REGIME	:
		case TELOPT_X3PAD	:
		case TELOPT_NAWS	:
		case TELOPT_TSPEED	:
		case TELOPT_LFLOW	:
		case TELOPT_LINEMODE	:
		case TELOPT_XDISPLOC	:
		case TELOPT_OLD_ENVIRON	:
		case TELOPT_AUTHENTICATION	:
		case TELOPT_ENCRYPT	:
		case TELOPT_NEW_ENVIRON	: return true;break;

		default			: return false;
	}
}

QString CTelnet::convertCommand(unsigned char Command) const
{
	QString s="";
	switch (Command) {
		case IAC	: s="IAC"; break;
		case DONT	: s="DONT"; break;
		case DO		: s="DO"; break;
		case WILL	: s="WILL"; break;
		case WONT	: s="WONT"; break;
		case SB		: s="SB"; break;
		case GA		: s="GA"; break;
		case EL		: s="EL"; break;
		case EC		: s="EC"; break;
		case AYT	: s="AYT"; break;
		case AO		: s="AO"; break;
		case IP		: s="IP"; break;
		case BREAK	: s="BREAK"; break;
		case DM		: s="DM"; break;
		case NOP	: s="NOP"; break;
		case SE		: s="SE"; break;
		case EOR	: s="EOR"; break;
		case ABORT	: s="ABORT"; break;
		case SUSP	: s="SUSP"; break;
		case xEOF	: s="xEOF"; break;

		default		: s.setNum((unsigned int)Command);
	}

	return s+" ";
}

QString CTelnet::convertOption(unsigned char Option) const
{
	QString s;
	switch (Option) {
		case TELOPT_BINARY		: s="BINARY"; break;
		case TELOPT_ECHO		: s="ECHO"; break;
		case TELOPT_RCP			: s="RCP"; break;
		case TELOPT_SGA			: s="SGA"; break;
		case TELOPT_NAMS		: s="NAMS"; break;
		case TELOPT_STATUS		: s="STATUS"; break;
		case TELOPT_TM			: s="TM"; break;
		case TELOPT_RCTE		: s="RCTE"; break;
		case TELOPT_NAOL		: s="NAOL"; break;
		case TELOPT_NAOP		: s="NAOP"; break;
		case TELOPT_NAOCRD		: s="NAOCRD"; break;
		case TELOPT_NAOHTS		: s="NAOHTS"; break;
		case TELOPT_NAOHTD		: s="NAOHTD"; break;
		case TELOPT_NAOFFD		: s="NAOFFD"; break;
		case TELOPT_NAOVTS		: s="NAOVTS"; break;
		case TELOPT_NAOVTD		: s="NAOVTD"; break;
		case TELOPT_NAOLFD		: s="NAOLFD"; break;
		case TELOPT_XASCII		: s="XASCII"; break;
		case TELOPT_LOGOUT		: s="LOGOUT"; break;
		case TELOPT_BM			: s="BM"; break;
		case TELOPT_DET			: s="DET"; break;
		case TELOPT_SUPDUP		: s="SUPDUP"; break;
		case TELOPT_SUPDUPOUTPUT	: s="SUPDUPOUTPUT"; break;
		case TELOPT_SNDLOC		: s="SNDLOC"; break;
		case TELOPT_TTYPE		: s="TTYPE"; break;
		case TELOPT_EOR			: s="EOR"; break;
		case TELOPT_TUID		: s="TUID"; break;
		case TELOPT_OUTMRK		: s="OUTMRK"; break;
		case TELOPT_TTYLOC		: s="TTYLOC"; break;
		case TELOPT_3270REGIME		: s="3270REGIME"; break;
		case TELOPT_X3PAD		: s="X3PAD"; break;
		case TELOPT_NAWS		: s="NAWS"; break;
		case TELOPT_TSPEED		: s="TSPEED"; break;
		case TELOPT_LFLOW		: s="LFLOW"; break;
		case TELOPT_LINEMODE		: s="LINEMODE"; break;
		case TELOPT_XDISPLOC		: s="XDISPLOC"; break;
		case TELOPT_OLD_ENVIRON		: s="OLD_ENVIRON"; break;
		case TELOPT_AUTHENTICATION	: s="AUTHENTICATION"; break;
		case TELOPT_ENCRYPT		: s="ENCRYPT"; break;
		case TELOPT_NEW_ENVIRON		: s="NEW_ENVIRON"; break;

		default				: s.setNum((unsigned int)Option);
	}

	return s+" ";
}

// filters telnet commands and removes them from the buffer
// in debugmode inserts human readable commands/options to buffer
int CTelnet::processCommands()
{

	int i,di;
	bool comend;
	int size;
	QString humanReadable;
	unsigned char buf[64];

	size=buffer.size();

	// search for command sequence
	i=0;
	while (i<size){
		di=0;
		if (isInterpretAsCommand(buffer[i])) {
			humanReadable="";
			// convert Command
			humanReadable.append(convertCommand(buffer[i+di]));
			di++;
			comend=false;
			while ((!isInterpretAsCommand(buffer[i+di])) && (!comend)) {
				if (isCommand(buffer[i+di]))
					humanReadable.append(convertCommand(buffer[i+di]));
				else if (isOption(buffer[i+di]))
					humanReadable.append(convertOption(buffer[i+di]));
				else
					humanReadable.append(convertCommand(buffer[i+di]));
				if (isOption(buffer[i+di]))
					comend=true;
				di++;
				if (i+di>=size) break; // check range first
			}
			KDEBUG1(KDEBUG_INFO,kmud_telnet,"received telnet command: %s",humanReadable.data());// handle Command
			humanReadable="";
			
			// will echo -> do echo
			if (di==3 && (unsigned char)buffer[i]==IAC
			&& (unsigned char)buffer[i+1]==WILL
			&& (unsigned char)buffer[i+2]==TELOPT_ECHO){
				setEcho(false);
				buf[0]=IAC;buf[1]=DO;buf[2]=TELOPT_ECHO; sendData((char*)&buf, 3);
				humanReadable.append("IAC DO ECHO");
			} else
			
			// wont echo -> dont echo
			if (di==3 && (unsigned char)buffer[i]==IAC
			&& (unsigned char)buffer[i+1]==WONT
			&& (unsigned char)buffer[i+2]==TELOPT_ECHO){
				setEcho(true);
				buf[0]=IAC;buf[1]=DONT;buf[2]=TELOPT_ECHO;
				sendData((char*)&buf, 3);
				humanReadable.append("IAC DONT ECHO");
			} else
			
			// do echo -> will echo
			if (di==3 && (unsigned char)buffer[i]==IAC
			&& (unsigned char)buffer[i+1]==DO
			&& (unsigned char)buffer[i+2]==TELOPT_ECHO){
				setEcho(true);
				buf[0]=IAC;buf[1]=WONT;buf[2]=TELOPT_ECHO;
				sendData((char*)&buf, 3);
				humanReadable.append("IAC WONT ECHO");
			} else

			// don't echo -> won't echo
			if (di==3 && (unsigned char)buffer[i]==IAC
			&& (unsigned char)buffer[i+1]==DONT
			&& (unsigned char)buffer[i+2]==TELOPT_ECHO){
				setEcho(true);
				buf[0]=IAC;buf[1]=WONT;buf[2]=TELOPT_ECHO;
				sendData((char*)&buf, 3);
				humanReadable.append("IAC WONT ECHO");
			} else

			// refuse any IAC DO <option>
			if (di==3 && (unsigned char)buffer[i]==IAC
			&& (unsigned char)buffer[i+1]==DO
			&& isOption((unsigned char)buffer[i+2])){
				buf[0]=IAC;buf[1]=WONT;buf[2]=buffer[i+2];
				sendData((char*)&buf, 3);
				humanReadable.append("IAC WONT ");
				humanReadable.append(convertOption(buffer[i+2]));
			} else

			// refuse any IAC WILL <option>
			if (di==3 && (unsigned char)buffer[i]==IAC
			&& (unsigned char)buffer[i+1]==WILL
			&& isOption((unsigned char)buffer[i+2])){
				buf[0]=IAC;buf[1]=DONT;buf[2]=buffer[i+2];
				sendData((char*)&buf, 3);
				humanReadable.append("IAC DONT ");
				humanReadable.append(convertOption(buffer[i+2]));
			}

			KDEBUG1(KDEBUG_INFO,kmud_telnet,"reply telnet command: %s",humanReadable.data());

			buffer.replace(i,di,"");
			//i=i-di; // oops, buffer.size() is now smaller

			//i=i+di; //last 2 statements commented out by Tomer- "HUH?"
			size=buffer.size();
		} else i++; // no IAC -> skip byte
	}
	return buffer.size();
}


QString CTelnet::getHostName() const
{
	return hostName;
}

QString CTelnet::getHostIP() const
{

	unsigned long x = hostIP;
	QString s="";
	QString a="";

	a.setNum(x%256);
	s.append(a+".");
	x/=256;
	a.setNum(x%256);
	s.append(a+".");
	x/=256;
	a.setNum(x%256);
	s.append(a+".");
	x/=256;
	a.setNum(x%256);
	s.append(a);

	return s;
}

	
unsigned long CTelnet::getHostPort() const
{
	return hostPort;
}


bool CTelnet::isConnected() const
{
	return (tsocket!=0);
}
