/***************************************************************************
                          cmudlineedit.cpp
                        -------------------
    description          : Class implementing an input line
    begin                : Wed Sep 8 1999
    copyright            : (C) 1999 by Stephan Uhlmann
    email                : suhlmann@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmudlineedit.h"


CMudLineEdit::CMudLineEdit(KmudDoc* doc, QWidget * parent, const char * name)
 : QLineEdit(parent,name)
{
	document = doc;
	
	history.setAutoDelete(true);
	history.clear();
	setHistoryPosition(0);
	setMaxHistoryLines(1000);
	setKeepLastInput(false);
	setSelectAllOnFocus(false);
}

CMudLineEdit::~CMudLineEdit()
{
}


void CMudLineEdit::keyPressEvent(QKeyEvent* e)
{
	QString s=text();
	AliasEntry *alias;
	CCharacterProfile* profile;

	// cerr << e->key () << ' ' << e->ascii () << ' ' << e->state () << ' ' << e->type () << endl;

	profile = document->getCharacterProfile(document->getCurrentCharacterID());
	if (profile){
		profile->readAliases();
		// This is for macros -- Luke Ehresman (luke@usa.om.org)
		alias = profile->firstAlias();
		while (alias != NULL){
			if (alias->macro != 0){
				if ((e->key() == alias->macro) && !((e->state() & KeyButtonMask) ^ alias->modifiers)) {
					s = alias->definition;
					emit macroPressed(s);
					return;
				}
			}
			alias = profile->nextAlias();
		}
		// end macros
	}

	switch (e->key()){
		case Key_Up :
			if(document->getIntelligentHistory() && // is intelligent history-browsing turned on?
			getCompletionInput().isNull() &&  // no completion input yet
			s.length() > 0 &&                 // there is current input
			(unsigned int)getHistoryPosition() == history.count() &&
			s!=markedText() ) { // no completion if marked (to get keepLastInput to work)
				setCompletionInput(s);
			}
		case Key_Down :
		case Key_PageUp:
		case Key_PageDown:
			if (e->state()&ShiftButton)
				emit scrollKeyPressed(e);
			else
				historyScroll(e);
			break;
		case Key_Home:
		case Key_End:
			if (e->state()&ControlButton)
				emit scrollKeyPressed(e);
			else
				historyScroll(e);
			break;
		case Key_Enter :
		case Key_Return :
			// Add the input to the history
			addInputToHistory(s);

			// Send the SIGNAL for returnPressed (see kmudview)
			emit returnPressed();
			emit textSend (s);
			// clear the Inputline
			clear ();
                      
			// If we must keep the last input then we copy the
			// last history and select it all.
			if (keepLastInput && (s.isEmpty()==false)){
				setText(history.last());
				selectAll();
			}
			// When enter is pressed we flush the completion input
			setCompletionInput((const char*)NULL);
			
			break;
		default:
			QLineEdit::keyPressEvent(e);
			if( getCompletionInput().length() > 0 )
				setCompletionInput((const char*)NULL);


	}
}

void CMudLineEdit::focusInEvent(QFocusEvent* event)
{
	QLineEdit::focusInEvent(event);
	if (selectAllOnFocus && event->gotFocus())
	{
		selectAll();
	}
}


/** History consists of two implementations...
  * The first one is the easy one... If no input was given and the user
  * issues the arrow up key, he wanders through the history of all the
  * issued commands.
  * 
  * The second implementation is trickier as it tries to complete the given
  * input with former commands that matches the given input. It also
  * remembers the given input so you can go through the history of the given
  * input as be it the first history implementation. The given input is
  * stored in the QString completionInput and is reset only at pressing
  * the return key.
  */
void CMudLineEdit::historyUp()
{
	int historyDeleter = 1;
	QString currentPrompt=text();

	// if the Last line was kept for the user we have to act a little different
	if ((currentPrompt==history.last()) // if actual == last...
	&& (currentPrompt==markedText()) //  and it's selected...
	&& keepLastInput) {              // by keepLastInput...
		historyDeleter++;
	}

	if ( history.count()>0) {
	// If the completion input is set then we try to complete the input
	// with the CompletionInput pattern
		int pos=getHistoryPosition();
		
		if ( getCompletionInput().length() > 0 ) {
           
			int x = pos - historyDeleter;
			QString compareString;

			while(x>=0) {
				compareString = history.at(x);
				if( compareString.find(getCompletionInput() , 0 , FALSE ) == 0 ) {
					setHistoryPosition(x);
					setText(compareString);
					return;
				}
				x--;
			}
			// We do nothing if no match is found (or index 0 is reached).
			return;
		}
        
		if (pos>0)  // go up in the history list.
			setHistoryPosition(pos-historyDeleter);
		setText( history.at(historyPosition));
	}
}

void CMudLineEdit::historyDown(){
	QString currentPrompt=text();

	if (history.count()>0 && (unsigned int)getHistoryPosition() <= history.count()) {
	// If the completion input is set then we can see if we can go down
	// in the history list using completion, else we have normal history behaviour
		if( getCompletionInput().length() > 0) {
			int x = getHistoryPosition()+1;
			QString compareString;
			while((unsigned int)x<history.count()) {
				compareString = history.at(x);
				if( compareString.find(getCompletionInput() , 0 , FALSE ) == 0 ) {
					setHistoryPosition(x);
					setText(compareString);
					return;
				}
				x++;
			}
			// If no matches are found then insert the original completion input
			// and set the history to the history.count
			setText(getCompletionInput());
			setHistoryPosition(history.count());
			return;
        
		}
		// go down in the history list
		if( (unsigned int) getHistoryPosition() < (history.count()-1)){
			historyPosition++;
			setText(history.at(getHistoryPosition()));
		} else
			setText("");	// clear the input if we are at the end of history
	}
}

int CMudLineEdit::getMaxHistoryLines(){
	return maxHistoryLines;
}

void CMudLineEdit::setMaxHistoryLines(int lines)
{
	if (lines>0) maxHistoryLines=lines;
	else maxHistoryLines=1;
}


bool CMudLineEdit::getKeepLastInput()
{
	return keepLastInput;
}


void CMudLineEdit::setKeepLastInput(bool b)
{
	keepLastInput=b;
}

bool CMudLineEdit::getSelectAllOnFocus()
{
	return selectAllOnFocus;
}


void CMudLineEdit::setSelectAllOnFocus(bool b)
{
	selectAllOnFocus=b;
}

void CMudLineEdit::setHistoryPosition(int pos){
// We keep a history position to see where we are when walking through
// the history. The value may be equal to the length of the history in 
// which case we have not touched the history. (ie. history.at(pos) might
// give an index out of range!!!!!)
	historyPosition=pos;
}

int CMudLineEdit::getHistoryPosition(){
	return historyPosition;
}

int CMudLineEdit::setCompletionInput(QString input) {
	completionInput = input;
}

QString CMudLineEdit::getCompletionInput() {
	return completionInput;
}

void CMudLineEdit::addInputToHistory(QString input) {
// blocking passwords, empty strings and the same as the last inputs
// the input is added to the history.
	if ((echoMode() == QLineEdit::Normal)	// dont buffer passwords
	&& (input.isEmpty() == false)			// and empty strings
	&& (input != history.last() )){			// and same as the last
		// history overflow, remove the first entry
		while (history.count()>=(unsigned int)getMaxHistoryLines())
			history.removeFirst();
		history.append(input);
	}
	setHistoryPosition(history.count());
}

void CMudLineEdit::historyScroll(QKeyEvent * e){
	switch (e->key()){
		case Key_PageUp:
		case Key_PageDown:
		case Key_Home:
		case Key_End:
			QLineEdit::keyPressEvent(e);
			break;
		case Key_Up:
			historyUp();
			break;
		case Key_Down :
			historyDown();
			break;
    }
}

/** removes the QT-Selection-Bug */
void CMudLineEdit::focusOutEvent(QFocusEvent* event)
{
} // focusOutEvent

