/* go_board.cpp
 *
 * Pieter Eendebak <pte@ddsw.nl>
 */

#include "includes.h"

#include "go.h"
#include "go_board.moc"
#include "go_player.h"
#include "ktimer.h"


KGoBoard::KGoBoard( int x, int y )
	: QObject( 0, 0 ), white(0), black(0),
		pauze(false), p(0), ma(0), l(0)
{
	if ( x>=MIN_SIZE && x<=MAX_SIZE )
		xWidth = x ;
	else	xWidth = DEFAULT_X_SIZE ;

	if ( y>=MIN_SIZE && y<=MAX_SIZE )
		yWidth = y ;
	else	yWidth = DEFAULT_Y_SIZE ;

	white_timer = new KTimer() ;
	black_timer = new KTimer() ;

	newGame( xWidth, yWidth ) ;
}

KGoBoard::~KGoBoard()
{
	delete white_timer ;
	delete black_timer ;
}

void KGoBoard::initData( int xS, int yS )
{
	deleteArray( p, xWidth, yWidth ) ;
	deleteArray( ma, xWidth, yWidth ) ;
	deleteArray( l, xWidth, yWidth ) ;

	xWidth=xS ;
	yWidth=yS ;

	createArray( p, xWidth, yWidth ) ;
	createArray( ma, xWidth, yWidth ) ;
	createArray( l, xWidth, yWidth ) ;

	/* remove old stuff */
	removeWhitePlayer() ;
	removeBlackPlayer() ;
	resetTimers() ;
}

void KGoBoard::createArray( unsigned char** &p, int w, int h )
{
	#ifdef DEBUG
	// debug("createArray") ;
	#endif
	p = new unsigned char *[w] ;
	for( int i=0; i<w; i++ )
	{
		p[i] = new unsigned char [h] ;
/*		for( int j=0; j<h;j++)
			p[i][j]=EMPTY ;
*/
	}
}

void KGoBoard::deleteArray( unsigned char** &p, int w, int )
{
	if ( p!=0 )
	{
		for( int i=0; i<w; i++ )
			delete [] p[i] ;
		delete [] p ;
	}
	p=0 ;
}

void KGoBoard::newGame( int xS, int yS )
{
	/* create new */
	int i, j ;
	initData( xS, yS ) ;

	/* init board */
	for (i = 0; i < xWidth; i++)
        	for (j = 0; j < yWidth; j++)
        		p[i][j] = EMPTY;

/*	// use this for testing of counting scores
	for (i = 0; i < xWidth; i++)
        	for (j = 0; j <yWidth-1; j++)
        		p[i][j] = WHITE;
*/

	play = PLAYING ;
 	pass = 0;
	wh_cap_x = PASS ;
	bl_cap_x = PASS ;
	white_cap = 0 ; black_cap = 0 ;
	player_to_move = BLACK ;

	pauzeGame() ;

	emit boardSetup() ;
}

void KGoBoard::startGame() 
{
	resumeGame() ;
	pushPlayer() ;
}

void KGoBoard::resetTimers()
{
	if( white_timer!=0 )
		white_timer->stop() ;
	if( black_timer!=0 )
		black_timer->stop() ;
	pauze = false ;
}

void KGoBoard::pauzeGame() 
{
	pauze = true ;
	stopTimers() ;
	emit boardStateChange( PAUZED ) ;
}

void KGoBoard::resumeGame() 
{
	pauze = false ;
	resumeTimers() ;
	pushPlayer() ;
	emit boardStateChange( RESUMED ) ;
}

void KGoBoard::endGame()
{
	stopTimers() ;	
	play = ENDED ;
	emit boardStateChange( play ) ;
}

void KGoBoard::startCounting()
{
	play = COUNTING ;
	emit boardStateChange( play ) ;
}

int KGoBoard::load( QDataStream &d )
{
	#ifdef DEBUG
	//debug("KGoBoard-load()");
	#endif

	int i=0, j=0 ;
	char *str ;
	Q_INT32 time ;


	d >> str ;
	if ( strcmp(str, FILE_BOARD)!=0 )
	{
		delete str ;
		return ERROR ;
	}
	else	delete str ;
 	
	pauzeGame() ;
	d >> time ;
	white_timer->setTime( (long)time ) ;
	d >> time ;
	black_timer->setTime( (long)time ) ;


	int xS, yS ;

	d >> xS ;
	d >> yS ;

	initData( xS, yS ) ;

	/* the game board */
	for( i=0; i<xWidth; i++ )
	for( j=0; j<yWidth; j++ )
		d >> p[i][j] ;

	/* game state */
	d >> play;
	/* pass indicator */
	d >> pass;

        /* location of computer stone captured */
	d >> wh_cap_x >> wh_cap_y ;
        /* location of opponent stone captured */
	d >> bl_cap_x >> bl_cap_y ;

	d >> player_to_move ;

	resumeGame() ;
	emit boardSetup() ;
	return GOOD ;
}

int KGoBoard::save( QDataStream &d )
{
	#ifdef DEBUG
	debug("KGoBoard-save()") ;
	#endif

	int i=0, j=0 ;
	d << (char *)FILE_BOARD ;

	/* the game timers */
	d << (Q_INT32)white_timer->getTime() ;
	d << (Q_INT32)black_timer->getTime() ;

	d << xWidth ;
	d << yWidth ;

	/* the game board */
	for( i=0; i<xWidth; i++ )
	for( j=0; j<yWidth; j++ )
		d << p[i][j] ;

	/* don't save matrix for marking (ma and ml) */
	/* don't save matrix for liberty (l) */
	/* don't save current stone liberty (lib) */

	/* game state */
	d << play ;
	/* pass indicator */
	d << pass;

        /* location of computer stone captured */
	d << wh_cap_x << wh_cap_y ;
        /* location of opponent stone captured */
	d << bl_cap_x << bl_cap_y ;

	d << player_to_move ;

	return GOOD ;
}


bool KGoBoard::saveAbleGame()
{
	return white!=0 && black!=0 &&
		white->canBeSaved() && black->canBeSaved() ;
}
 
void KGoBoard::stopTimers()
{
	white_timer->freeze() ;
	black_timer->freeze() ;
}

void KGoBoard::resumeTimers()
{
	white_timer->resume() ;
	black_timer->resume() ;
}

int KGoBoard::getXWidth() const
{
	return xWidth ;
}

int KGoBoard::getYWidth() const
{
	return yWidth ;
}

void KGoBoard::set( int x, int y, int style, bool notify )
{
	if ( inRange( x, y ) )
		p[x][y]=style ;
	if ( notify )
		emit boardStateChange( BOARD_SET ) ;
} 

int KGoBoard::get( int x, int y ) const
{
	if( inRange( x, y ) )
		return p[x][y] ;
	return EMPTY ;
}

bool KGoBoard::inRange( int x, int y ) const
{
	return ( x>-1 && x<xWidth && y>-1 && y<yWidth ) ;
}

int KGoBoard::getWhiteScore()
{
	if ( play == COUNTING )
	{
		int w = 0 ;

		w += getWhiteFields() ;
		w -= whiteCaptured() ;
		return w ;
	}
	else	return -1 ;
}

int KGoBoard::getBlackScore()
{
	if ( play == COUNTING )
	{
		int w = 0 ;

		w += getBlackFields() ;
		w -= blackCaptured() ;
		return w ;
	}
	else	return -1 ;
}

int KGoBoard::getWhiteFields()
{
	int ret=0;

	for(int i=0;i<xWidth;i++)
	for(int j=0;j<yWidth;j++)
		if( p[i][j]==WHITE_ZONE )
			ret++ ;

	return ret ;		
}

int KGoBoard::getBlackFields()
{
	int ret=0;

	for(int i=0;i<xWidth;i++)
	for(int j=0;j<yWidth;j++)
		if( p[i][j]==BLACK_ZONE )
			ret++ ;

	return ret ;		
}

void KGoBoard::setWhitePlayer( KGoPlayer *p )
{
	removeWhitePlayer() ;

	white = p ;
	if( white!=0 )
	{
		connect( white, SIGNAL( doMove(int,int)),
			this, SLOT( whiteMove(int, int)) ) ;
		connect( this, SIGNAL( blackMoved( int, int ) ),
				white, SLOT( opponentMoved(int,int)) ) ;
		connect( this, SIGNAL( illegalMove(int, int)),
				white, SLOT( illegalMove(int, int)) ) ;
	}
}

void KGoBoard::setBlackPlayer( KGoPlayer *p )
{
	removeBlackPlayer() ;

	black = p ;

	if ( black!= 0 )
	{
		connect( black, SIGNAL( doMove(int,int)),
			this, SLOT( blackMove(int, int)) ) ;
		connect( this, SIGNAL( whiteMoved(int,int) ),
			black, SLOT( opponentMoved(int,int)) ) ;
		connect( this, SIGNAL( illegalMove(int, int)),
			black, SLOT( illegalMove(int, int)) ) ;
	}
}

int KGoBoard::getWhiteTime()
{
	return white_timer->getSeconds() ;
}

int KGoBoard::getBlackTime()
{
	return black_timer->getSeconds() ;
}

void KGoBoard::removeWhitePlayer()
{
	if ( white !=0 )
	{
		white->disconnect( this ) ;
	}
	white = 0 ;
}

void KGoBoard::removeBlackPlayer()
{
	if ( black !=0 )
	{
		black->disconnect( this ) ;
	}
	black = 0 ;
}

void KGoBoard::emit_boardStateChange(int s)
{
	emit boardStateChange( s ) ;
}

QString KGoBoard::getIllegalMoveReason( int i )
{
	QString p;

	switch( i )
	{
		case LEGAL:	p.append( i18n("legal move") ) ;
				break ;
 		case FIELD_NOT_EMPTY:
				p.append( i18n(
						"field not empty") ) ;
				break ;
 		case SUICIDE :	p.append( i18n("suicide") ) ;
				break ;
  		case KO:	p.append( i18n("ko") ) ;
				break ;
 		case OUTSIDE_BOARD:
				p.append( i18n("outside board")  ) ;
				break ;
		case OPPONENTS_TURN:
				p.append( i18n("not your turn")  ) ; 				break ;
  		default:	p.append( i18n("unknown") ) ;
				break ;
  	}
	return p ;
}

QString KGoBoard::translateMove(int x, int y)
{
	QString v;

	if ( x==PASS )
		v.sprintf( klocale->translate("pass") ) ;
	else if ( x<0 || y<0 )
	{
		v.sprintf( i18n("error: no such move possible") ) ;
	}
	else
	{
		v.sprintf("%c%d", x+65, y+1 ) ;
	}

	return v ;
} 

QString KGoBoard::translatePlayer( int p )
{
	QString v;

	switch( p )
	{
		case WHITE:
			v.sprintf( i18n("white") ) ;
			break ;
		case BLACK:
			v.sprintf( i18n("black") ) ;
			break ;
		default:
			v.sprintf( i18n("error: no such player") ) ;
			break ;
	}
	return v ;
} 

void KGoBoard::pushPlayer()
{
	if( playerToMove()==WHITE )
	{
		if( white!=0 )
			white->opponentMoved( PASS, PASS ) ;
	}
	else
	{
		if( black!=0 )
			black->opponentMoved( PASS, PASS ) ;
	}
}

void KGoBoard::player_moved( int x, int y, int player )
{
	#ifdef DEBUG
	//debug("player_moved") ;
	#endif

	if ( play!=PLAYING )
		return ;

	emit playerMoved( x, y, player ) ;

	if ( player == WHITE )
	{
		player_to_move = BLACK ;
		white_timer->freeze() ;
		black_timer->resume() ;
		emit whiteMoved( x, y ) ;
	}
	else	
	{
		player_to_move = WHITE ;
		black_timer->freeze() ;
		white_timer->resume() ;
		emit blackMoved( x, y ) ;
	}

}

void KGoBoard::printBoard()
{
	debug("Board field:\n") ;
	for( int n=0;n<xWidth;n++)
	for(int m=0;m<yWidth;m++)
	{
		printf("%d ", p[n][m] ) ;
		if ( m==yWidth-1 ) printf("\n");  
	}
}

void KGoBoard::makeMove( int x, int y, int player )
{
	#ifdef DEBUG
	//printf("Making move: %s - %d\n",
	//	translateMove(x,y)->data(), player ) ;
	#endif

	if ( x==-1 )
	{
		pass++ ;
	}
	else
	{
		pass=0;
		p[x][y] = player ;
	}

	examBoard( (player==WHITE?BLACK:WHITE) ) ;

	if ( pass<4 )
	{
		player_moved( x, y, player ) ;
		emit boardStateChange( PLAYING ) ;
	}
	else
	{
		endGame() ;
	}

}

void KGoBoard::rejectMove( int x, int y, int player, int r ) 
{
	#ifdef DEBUG
	printf( "illegal move for player %s:  %s\n",
			translatePlayer( player).data(),
			translateMove(x,y).data() );
	#endif
	emit illegalMove( r, player ) ;
}
 
void KGoBoard::tryMove( int x, int y, int player )
{
	if ( play!=PLAYING )
		return ;

	if ( player != playerToMove() )
	{
		rejectMove( x, y, player, OPPONENTS_TURN ) ;
		return ;
	}

	if ( x==PASS )
	{
		makeMove( PASS, PASS, player ) ;
	}
	else
	{
		pass=0 ;
	
		// check whether legal move...
		int r = legalMove( x, y, player ) ;
		if ( r == LEGAL )
			makeMove( x, y, player ) ;
		else
			rejectMove( x, y, player, r ) ;			
	}
}

void KGoBoard::whiteMove( int x, int y)
{
		tryMove( x, y, WHITE ) ;
}

void KGoBoard::blackMove( int x, int y )
{
		tryMove( x, y, BLACK ) ;
}

int KGoBoard::legalMove( int x, int y, int player )
{
	if( x==PASS )
		return LEGAL ;

	if ( inRange(x, y) )
	{
		if ( p[x][y] != EMPTY )
			return FIELD_NOT_EMPTY ;
		else if ( suicide( x, y, player ) )
			return SUICIDE ;
		else	return LEGAL ;
	}
	else	return  OUTSIDE_BOARD ;
}

bool KGoBoard::suicide( int x, int y, int player )
{
	if( x==PASS )
		return false ;

	int opponent = ( player==BLACK ? WHITE : BLACK ) ;

	/* check for suicide move at (x, y) */
	int m, n, k;

	/* check liberty of new move */
	int liberty = countLib(x, y, player );
	if ( liberty == 0)
	/* new move is suicide then check if kill my 
		pieces and Ko possibility*/
	{
		/* assume alive */
		p[x][y] = player ;

		/* check my pieces */
		eval( opponent );

		k = 0;

		for (n = 0; n < xWidth; n++)
		for (m = 0; m < yWidth; m++)
		{
			/* count pieces will be killed */
			if ((p[n][m] == opponent ) && l[n][m]==0)
					++k;
		}
		
		int player_cap_x, player_cap_y ;
		if ( player==WHITE )
 		{
			player_cap_x = wh_cap_x ; 
			player_cap_y = wh_cap_y ;
		}
		else
		{
			player_cap_x = bl_cap_x ; 
			player_cap_y = bl_cap_y ;
		}

		if	(	( k == 0 ) ||
				( k == 1 && ( (x == player_cap_x ) &&
					(y == player_cap_y ) )
			 	)
 			)
		/* either no effect on my pieces or an illegal Ko take back */
		{
	       		p[x][y] = EMPTY;   /* restore to open */
			return true;
			
		}
		else
		{	
			/* good move */
			// ??
	       		p[x][y] = EMPTY;   /* restore to open */
			return false ;
		}
      	}
	else
	{
		/* liberty>0 : good move */
		return false;
	}

}

int KGoBoard::countLib( int x, int y, int player )
{
	/* count liberty of piece at location i, j
   	with color 'player' and return value */

	/* liberty */
	lib=0 ;

	/* set all piece as unmarked */
	 for ( int i = 0; i < xWidth; i++)
	   for ( int j = 0; j < yWidth; j++)
	     ma[i][j] = UN_MARKED ;

	count( x, y, player ) ;

	return lib ;
}

void KGoBoard::count( int x, int y, int player )
{ 
	/* count liberty of piece at location i, j
   	with color 'player' and add value to 'lib' */

	/* set current piece as marked */
	ma[x][y] = MARK ;

	/* check West neighbor */
	if (x != 0)
   	{
   		if ((p[x - 1][y] == EMPTY) && ma[x - 1][y] == UN_MARKED)
   		{
			++lib;
			ma[x - 1][y] = MARK;
		}
		else
		if ((p[x - 1][y] == player) && ma[x - 1][y] == UN_MARKED ) 
			count(x - 1, y, player);
	}
	/* check East neighbor */
	if (x != xWidth-1)
	{
		if ((p[x + 1][y] == EMPTY) && ma[x + 1][y]==UN_MARKED )
		{
  			++lib;
 			ma[x + 1][y] = MARK;
 	    	}
		else
 		if ((p[x + 1][y] == player) && ma[x + 1][y] == UN_MARKED )
			count(x + 1, y, player);
	}
	/* check North neighbor */
	 if (y != 0 )
 	  {
 	   if ((p[x][y - 1] == EMPTY) && ma[x][y - 1] == UN_MARKED )
    	  {
 	      ++lib;
 	      ma[x][y - 1] = MARK;
	     }
 	   else
 	      if ((p[x][y - 1] == player ) && ma[x][y - 1] == UN_MARKED )
		  count(x, y - 1, player ) ;
 	 }
	/* check South neighbor */
	if (y != yWidth-1)
	{
		if ((p[x][y + 1] == EMPTY) &&
			ma[x][y + 1] == UN_MARKED )
		{
			++lib;
			ma[x][y + 1] = MARK;
		}
		else
 		{
			if ((p[x][y + 1] == player ) &&
				ma[x][y + 1] == UN_MARKED )
					count(x, y + 1, player ) ;
	 	}
	}

	/* end count */
}

void KGoBoard::eval( int player )
{
	int i, j;

	/* find liberty of each piece */
	for (i = 0; i < xWidth; i++)
	for (j = 0; j < yWidth; j++)
	{
		if ( p[i][j] == player )
		{
		 	int t = countLib(i, j, player );
		 	l[i][j] = t;
		}
		else	l[i][j] = 0 ;
	}

/*	// for debugging 
	for (i = 0; i < xWidth; i++)
	for (j = 0; j < yWidth; j++)
	{
		printf("%d ", l[i][j] ) ;
		if ( j==yWidth-1 ) printf("\n");
	}
*/
}

void KGoBoard::examBoard( int player)
{
	/* examine pieces, opponent has just made a move */
	
	int i, j, n;

	/* find liberty of each piece */
	eval( player );

	/* initialize piece captured */
	if ( player == WHITE )
	{
		wh_cap_x = -1;
		wh_cap_y = -1;
	}
	else
	{
		bl_cap_x = -1;
		bl_cap_y = -1;
	}

	n = 0; /* The number of captures this move for Ko purposes */

	/* remove all piece of zero liberty */
	for (i = 0; i < xWidth; i++)
	for (j = 0; j < yWidth; j++)
	{
		if ((p[i][j] == player) && (l[i][j] == 0))
		{
			p[i][j] = EMPTY;
			/* record piece captured */
			if ( player == WHITE )
			{
				wh_cap_x = i;
				wh_cap_y = j;
				++white_cap ;
			}
			else
			{
				bl_cap_x = i;
				bl_cap_y = j;
				++black_cap ;
			}
			++n;  /* increment number of captures on this move */
		}
	}

	/* reset to -1 if more than one stone captured since  no Ko possible */
	if ( player == WHITE && n > 1)
	{
		wh_cap_x = -1;   
		wh_cap_y = -1;
	}
	else if ( n > 1 )
	{
		bl_cap_x = -1;
		bl_cap_y = -1;
	}
	/* end examboard */
}

int KGoBoard::playerToMove()
{
	if ( play==PLAYING )
		return player_to_move ;
	else	return NO_PLAYER ;
}

void KGoBoard::setPlayerToMove( int c ) 
{
	player_to_move = c ;
}

int KGoBoard::whitePieces() const
{
	int ret=0 ;
	for( int i=0; i<xWidth; i++ )
	for( int j=0; j<yWidth; j++ )
		if ( p[i][j]==WHITE ) ret++ ;
	return ret ;
}

int KGoBoard::blackPieces() const
{
	int ret=0 ;
	for( int i=0; i<xWidth; i++ )
	for( int j=0; j<yWidth; j++ )
		if ( p[i][j]==BLACK ) ret++ ;
	return ret ;
}

int KGoBoard::whiteCaptured() const
{
	return white_cap ;
}	

int KGoBoard::blackCaptured() const
{
	return black_cap ;
}
