/***************************************************************************
                                 main.cpp
                             -------------------
    begin                : 01.06.1999
    copyright            : (C) 1999 by Ullrich Wagner
    email                : Ullrich.Wagner@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include<sys/wait.h>
#include <errno.h>
#include <time.h>

#include "kstring.h"
#include "dbwork.h"

#define BACKLOG		10

void serveClient(int actSocket, const char* clientName);
void getParam(const char* string, char* param1, char* param2, char* param3);
int koraSend(int actSocket, const KString& data);
char* getDateHour();
int getKOraSPort();
void sendLOGON(int actSocket, const char* clientName, char* buffer);
void sendLOGOFF(int actSocket, const char* clientName);
void sendEXECUTE_SINGLE(int actSocket, const char* clientName, char* buffer);
void sendEXECUTE_MULTI(int actSocket, const char* clientName, char* buffer);
void sendDESC(int actSocket, const char* clientName, char* buffer);
void sendLIST_SEQUENCES(int actSocket, const char* clientName);
void sendLIST_SYNONYMS(int actSocket, const char* clientName);
void sendLIST_TABLES(int actSocket, const char* clientName);
void sendLIST_PACKAGES(int actSocket, const char* clientName);
void sendDETAIL_SEQUENCE(int actSocket, const char* clientName, char* buffer);
void sendDETAIL_SYNONYM(int actSocket, const char* clientName, char* buffer);
void sendDETAIL_TABLE(int actSocket, const char* clientName, char* buffer);
void sendDETAIL_PACKAGE(int actSocket, const char* clientName, char* buffer);
void sendERROR_PACKAGE(int actSocket, const char* clientName, char* buffer);
void sendQUIT(int actSocket, const char* clientName);



bool	debug_mode = false;


int main()
{
	int			sockfd;
	int			new_fd;
	struct sockaddr_in	my_addr;
	struct sockaddr_in	their_addr;
	int			sin_size;
	char*			koraDebugVar;
	char			clientName[41];
	int			portNr;

	// get the environment for the debug
	koraDebugVar = getenv("KORA_DEBUG");

	// was the var found ?
	if (koraDebugVar != NULL)
	{
		// ok, the var is set. But is it set to 'true' ?
		if (!strcmp(koraDebugVar, "true"))
		{
			// ok, the debug mode is to set on
			debug_mode = true;
		}
	}

	// start message
	printf("KOra V0.3.2 - ORACLE SQL Connection server\nWritten by Ullrich Wagner\n");
	printf("This software is published under the GNU Public License\n");
	if (debug_mode)
	{
		printf("DEBUG MODE\n");
	}

	// create the socket for KOra
	sockfd = socket(AF_INET, SOCK_STREAM, 0);
	if (sockfd < 0)
	{
		fprintf(stderr, "Unable to create socket for KOra server\n");
		exit(-1);
	}

	// get the port of the server
	portNr = getKOraSPort();

	// configure the socket
	my_addr.sin_family = AF_INET;
	my_addr.sin_port = htons(portNr);
	my_addr.sin_addr.s_addr = INADDR_ANY;
	bzero(&(my_addr.sin_zero), 8);

	// bind the configuration to the socket
	if (bind(sockfd, (struct sockaddr*) &my_addr, sizeof(struct sockaddr)) < 0)
	{
		fprintf(stderr, "Unable to bind port to KOra socket\n");
		exit(-1);
	}

	// specify the maximum amount of users that can be connected to the server
	if (listen(sockfd, BACKLOG) < 0)
	{
		fprintf(stderr, "Unable to configure KOra socket\n");
		exit(-1);
	}

	printf("Running on port %d\n", portNr);

	// go into the server loop
	for(;;)
	{
		sin_size = sizeof(struct sockaddr_in);
		// wait for a new connection
		new_fd = accept(sockfd, (struct sockaddr*) &their_addr, (socklen_t*) &sin_size);

		// an error occured
		if (new_fd < 0)
		{
			fprintf(stderr, "Error while client tried to connect to KOra server. Stay online.\n");
			continue;
		}
		
		// set the clientName
		sprintf(clientName, "%s-%s", (char*) inet_ntoa(their_addr.sin_addr), getDateHour());

		// print statistic message
		printf("KOra Server got connection from %s\n", clientName);
		// fork a new process
		if (!fork())
		{
			// send message to client
			if (send(new_fd, "KOra 0.3.2 Server connected\n", 29, 0) < 0)
			{
				fprintf(stderr, "Unable to send data to client.\n");
				close(new_fd);
				exit(-1);
			}
			
			// go into the procedure that will do the work
			serveClient(new_fd, clientName);
			
			// close the socket
			close(new_fd);
			// terminate the process
			exit(0);
		}

		// close the socket (not needed)
		close(new_fd);

		// wait for the childs to terminate
		while (waitpid(-1, NULL, WNOHANG) > 0);
	}
}

void serveClient(int actSocket, const char* clientName)
{
	// define buffer for the data that is to receive
	char	buffer[1024];
	int	bytes_rec;
	KString	message;
	KString	sendMessage;

	// go into the serve loop
	for(;;)
	{
		// read the data (and wait for data)
		if ((bytes_rec = recv(actSocket, buffer, 1024, 0)) < 0)
		{
			if (debug_mode)
				fprintf(stderr, "Unable to receive data from client %s\n", clientName);
			continue;
		}
	
		// terminate the received string
		buffer[bytes_rec] = '\0';
		
		// if the last character of the received string is a newline, cut it away
		if (buffer[strlen(buffer)-1] == '\n')
		{
			buffer[strlen(buffer)-1] = '\0';
		}

		if (buffer[strlen(buffer) -1] == 0x0d)
		{
			buffer[strlen(buffer)-1] = '\0';
		}

		if (debug_mode)
		{
			if (!strncmp(buffer, "LOGON ", 6))
			{
				printf("Received command: <LOGON ...>\n");
			}
			else
			{
				printf("Received command: <%s>\n", buffer);
			}
		}

		// analyze the received string
//*****************************************************************************************
//*** LOGON TO ORACLE
//*****************************************************************************************
		if (!strncmp(buffer, "LOGON ", 6))
		{
			sendLOGON(actSocket, clientName, buffer);
		}
//******************************************************************************************
//*** LOGOFF FROM ORACLE
//******************************************************************************************
		else if (!strncmp(buffer, "LOGOFF", 6))
		{
			sendLOGOFF(actSocket, clientName);
		}
//******************************************************************************************
//*** EXECUTE SINGLE SQL COMMAND
//******************************************************************************************
		else if (!strncmp(buffer, "EXECUTE SINGLE ", 15))
		{
			sendEXECUTE_SINGLE(actSocket, clientName, buffer);
		}
//********************************************************************************************
//*** EXECUTE MULTIPLE SQL COMMAND
//********************************************************************************************
		else if (!strncmp(buffer, "EXECUTE MULTI ", 14))
		{
			sendEXECUTE_MULTI(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** DESCRIBE ORACLE TABLE
//*********************************************************************************************
		else if (!strncmp(buffer, "DESC ", 5))
		{
			sendDESC(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** LIST ALL SEQUENCES
//*********************************************************************************************
		else if (!strncmp(buffer, "LIST SEQUENCES", 14))
		{
			sendLIST_SEQUENCES(actSocket, clientName);
		}
//*********************************************************************************************
//*** LIST ALL SYNONYMS
//*********************************************************************************************
		else if (!strncmp(buffer, "LIST SYNONYMS", 13))
		{
			sendLIST_SYNONYMS(actSocket, clientName);
		}
//*********************************************************************************************
//*** LIST ALL TABLES
//*********************************************************************************************
		else if (!strncmp(buffer, "LIST TABLES", 11))
		{
			sendLIST_TABLES(actSocket, clientName);
		}
//*********************************************************************************************
//*** LIST ALL PACKAGES
//*********************************************************************************************
		else if (!strncmp(buffer, "LIST PACKAGES", 13))
		{
			sendLIST_PACKAGES(actSocket, clientName);
		}
//*********************************************************************************************
//*** GET DETAIL OF SEQUENCE
//*********************************************************************************************
		else if (!strncmp(buffer, "DETAIL SEQUENCE ", 16))
		{
			sendDETAIL_SEQUENCE(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** GET DETAIL OF SYNONYM
//*********************************************************************************************
		else if (!strncmp(buffer, "DETAIL SYNONYM ", 15))
		{
			sendDETAIL_SYNONYM(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** GET DETAIL OF TABLE
//*********************************************************************************************
		else if (!strncmp(buffer, "DETAIL TABLE ", 13))
		{
			sendDETAIL_TABLE(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** GET DETAIL OF PACKAGE
//*********************************************************************************************
		else if (!strncmp(buffer, "DETAIL PACKAGE ", 15))
		{
			sendDETAIL_PACKAGE(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** GET ERROR FOR A PACKAGE COMPILATION
//*********************************************************************************************
		else if (!strncmp(buffer, "ERROR PACKAGE ", 14))
		{
			sendERROR_PACKAGE(actSocket, clientName, buffer);
		}
//*********************************************************************************************
//*** QUIT CONNECTION TO KORA SERVER
//*********************************************************************************************
		else if (!strncmp(buffer, "QUIT", 4))
		{
			sendQUIT(actSocket, clientName);
			return;
		}
		else
		{
			// the given request wasn't understood
			fprintf(stderr, "The given request %s from %s wasn't understood. Disconnect from client\n", buffer, clientName);
			koraSend(actSocket, "ERROR\n");
			return;
		}

		// get the next command
	}
}

void getParam(const char* string, char* param1, char* param2, char* param3)
{
	int	nrParam = 1;
	int	count;
	char	transfer[2];
	int	errorTold = false;

	strcpy(param1, "");
	strcpy(param2, "");
	strcpy(param3, "");

	for(count = 0; count < (int) strlen(string); count++)
	{
		transfer[0] = string[count];
		transfer[1] = '\0';

		if (transfer[0] == ' ')
		{
			nrParam++;
			continue;
		}

		if (nrParam == 1)
		{
			strcat(param1, transfer);
		}
		else if (nrParam == 2)
		{
			strcat(param2, transfer);
		}
		else if (nrParam == 3)
		{
			strcat(param3, transfer);
		}
		else
		{
			if (!errorTold)
			{
				fprintf(stderr, "More params than posible. Overflow ignored.\n");
				errorTold = true;
			}
		}
	}
}








int koraSend(int actSocket, const KString& data)
{
	int	length;
	KString	command;
	char	strLength[21];
	int	rc = 0;
	int	count;
	KString	sendPart;
	int	wholeLength;

	// get the length of the data that is to transfer
	length = (int) data.length() + 18;
	sprintf(strLength, "%07d", length);

	// build the length info command
	command = "DATA LEN: ";
	command += strLength;
	command += "<";
	command += data;


	wholeLength = 0;

	if (debug_mode)
		printf("I want to send %d bytes.\n", (int) command.length());

	for(count = 0; count < (int) command.length();)
	{
		sendPart = command.mid(count, 1000);
		if (debug_mode)
		{
			printf("Sending bytes %d to %d of message\n", count, (count + (int) sendPart.length() - 1));
			printf("DATA:[%s]\n", (const char*) sendPart);
		}

		rc = send(actSocket, (const char*) sendPart, sendPart.length(), 0);


		if (rc == -1)
		{
			fprintf(stderr, "Data transmission error occured: ");
			if (errno == EBADF)
				fprintf(stderr, "An invalid descriptor was specified.\n");
			else if (errno == ENOTSOCK)
				fprintf(stderr, "The actual socket isn't valid (It's not a socket).\n");
			else if (errno == EFAULT)
				fprintf(stderr, "An invalid user space address was specified for a parameter.\n");
			else if (errno == EMSGSIZE)
				fprintf(stderr, "The socket requires that message be sent atomically, and the size of the message to be sent made this impossible.\n");
			else if (errno == EWOULDBLOCK)
				fprintf(stderr, "The socket is marked non-blocking and the requested operation would block.\n");
			else if (errno == ENOBUFS)
				fprintf(stderr, "The system was unable to allocate an internal buffer. The operation may succeed when buffers become available.\n");
			else
				fprintf(stderr, "The error reason is unknown. The code is %d\n", errno);
			
			return rc;
		}

		wholeLength += (int) sendPart.length();
		
		count += 1000;
	}

	if (debug_mode)
	{
		printf("THE CALCULATED LENGTH IS %d BYTES\n", length);
		printf("THE REALLY SENT LENGTH IS %d BYTES\n", wholeLength);
	}

	// deliver the return code to the caller
	return wholeLength;
}


char* getDateHour()
{
	time_t		actTime;
	struct tm*	timeStruct;
	static char	result[15];

	time(&actTime);
	timeStruct = gmtime(&actTime);

	sprintf(result, "%04d%02d%02d%02d%02d%02d", timeStruct->tm_year + 1900,
						    timeStruct->tm_mon + 1,
						    timeStruct->tm_mday,
						    timeStruct->tm_hour,
						    timeStruct->tm_min,
						    timeStruct->tm_sec);

	return result;	
}

int getKOraSPort()
{
	FILE*	handle;
        char*	homeDir;
	char	setFile[256];
	char	portLine[256];

	// get the name of the home directory
	homeDir = getenv("HOME");

	// is the home dir not found ?
	if (homeDir == NULL)
	{
		// so we must use the global settings from file /etc/koras
		handle = fopen("/etc/koras", "r");

		// is the file not opened ?
		if (handle == NULL)
		{
			// we don't have any file to open
			fprintf(stderr, "Unable to open settings file $HOME/.koras or /etc/koras. Aborting.\n");
			exit(-1);
		}

		// ok, the file is opened
	}
	else
	{
		// ok, we found the homedirectory. Build the name of the settings file
		sprintf(setFile, "%s/.koras", homeDir);

		// open the local settings file
		handle = fopen(setFile, "r");

		// couldn't the file be opened ?
		if (handle == NULL)
		{
			// We wasn't able to open a local file. So try to open the global file
			handle = fopen("/etc/koras", "r");
			
			// is the file not opened ?
			if (handle == NULL)
			{
				// we don't have any file to open
				fprintf(stderr, "Unable to open settings file $HOME/.koras or /etc/koras. Aborting.\n");
				exit(-1);
			}
		}
	}

	// if we are still alive we have opened a settings file. If possible the local .koras file. Otherwise the global
	// file /etc/koras

	// read the first line from file
	if (fgets(portLine, 256, handle) == NULL)
	{
		// we wasn't able to read parameter from file
		fprintf(stderr, "Unable to get port info from settings file.\n");
		exit(-1);
	}

	// correct the terminator of line if needed
	if (portLine[strlen(portLine)-1] == '\n')
	{
		portLine[strlen(portLine)-1] = '\0';
	}

	// is the leader of the found string the info for the port ?
	if (!strncmp(portLine, "PORT=", 5))
	{
		// ok, we got the port line.
		// close the file
		fclose(handle);
		// return the port as an integer
		return atoi(&(portLine[5]));
	}

	// we are at this point. This means that we wasn't able to read the port line.
	fprintf(stderr, "Unable to get port info from settings file.\n");
	
	// close the settings file
	fclose(handle);

	// exit
	exit(-1);

	return 0; // never reached
}

void sendLOGON(int actSocket, const char* clientName, char* buffer)
{
	char	param1[101];
	char	param2[101];
	char	param3[101];
	KString	message;
	KString	sendMessage;
	char*	command;

	int	numbytes;
	command = &(buffer[6]);
	getParam(command, param1, param2, param3);
	printf("Client %s is connecting to Oracle as %s@%s\n", clientName, param1, param3);
	if (DbWork::connectToOracle(param1, param2, param3, message))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		numbytes = koraSend(actSocket, "OK\n\n");
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendLOGOFF(int actSocket, const char* clientName)
{
	int numbytes;

	if (debug_mode)
		printf("Client %s is executing DbWork::releaseFromOracle()\n", clientName);
	else
		printf("Client %s releases from Oracle\n", clientName);
	DbWork::releaseFromOracle();
	numbytes = koraSend(actSocket, "OK\n");
	printf("Action from client %s successfully executed.\n", clientName);
}

void sendEXECUTE_SINGLE(int actSocket, const char* clientName, char* buffer)
{
	KString	message;
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	processedRows;
	KString	sqlResult;

	// get the real command
	command = &(buffer[15]);
	if (debug_mode)
		printf("The client %s is executing the simple DbWork::executeSql(%s, ...)\n", clientName, command);
	else
		printf("Client %s executes Sql command %s.\n", clientName, command);
	if (DbWork::executeSql(command, message, &processedRows, sqlResult, true))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += sqlResult;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendEXECUTE_MULTI(int actSocket, const char* clientName, char* buffer)
{
	KString	message;
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	processedRows;
	KString	sqlResult;

	// get the real command
	command = &(buffer[14]);
	if (debug_mode)
		printf("The client %s is executing the multiple DbWork::executeSql(%s, ...)\n", clientName, command);
	else
		printf("Client %s executes list of Sql commands: %s\n", clientName, command);
	if (DbWork::executeSql(command, message, &processedRows, sqlResult, false))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += sqlResult;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendDESC(int actSocket, const char* clientName, char* buffer)
{
	KString	message;
	KString	sendMessage;
	char*	command;

	int	numbytes;
	KString	sqlResult;

	// get the table name
	command = &(buffer[5]);

	if (debug_mode)
		printf("The client %s is executing DbWork::descTable(%s,...)\n", clientName, command);
	else
		printf("Client %s gets description of specified table.\n", clientName);
	if (DbWork::descTable(command, message, sqlResult))
	{
		sendMessage = "ERROR\n",
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += sqlResult;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendLIST_SEQUENCES(int actSocket, const char* clientName)
{
	KString	message;
	KString	sendMessage;

	int	numbytes;
	KString	seqList;

	if (debug_mode)
		printf("The client %s is executing DbWork::fillSequenceList(...)\n", clientName);
	else
		printf("Client %s retrieves list of sequences\n", clientName);
	if (DbWork::fillSequenceList(seqList, message))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += seqList;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendLIST_SYNONYMS(int actSocket, const char* clientName)
{
	KString	message;
	KString	sendMessage;

	int	numbytes;
	KString	synList;

	if (debug_mode)
		printf("The client %s is executing DbWork::fillSynonymList(...)\n", clientName);
	else
		printf("Client %s retrieves list of synonyms\n", clientName);
	if (DbWork::fillSynonymList(synList, message))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += synList;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendLIST_TABLES(int actSocket, const char* clientName)
{
	KString	message;
	KString	sendMessage;

	int	numbytes;
	KString	tabList;

	if (debug_mode)
		printf("The client %s is executing DbWork::fillTableList(...)\n", clientName);
	else
		printf("Client %s retrieves list of tables\n", clientName);
	if (DbWork::fillTableList(tabList, message))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += tabList;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendLIST_PACKAGES(int actSocket, const char* clientName)
{
	KString	message;
	KString	sendMessage;

	int	numbytes;
	KString	pacList;

	if (debug_mode)
		printf("The client %s is executing DbWork::fillPackageList(...)\n", clientName);
	else
		printf("Client %s retrieves list of packages\n", clientName);
	if (DbWork::fillPackageList(pacList, message))
	{
		sendMessage = "ERROR\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += pacList;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendDETAIL_SEQUENCE(int actSocket, const char* clientName, char* buffer)
{
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	index;
	KString	maxValue;
	KString	minValue;
	long	increment;
	KString	cycleFlag;
	char	transBuffer[10000];

	command = &(buffer[16]);
	if (debug_mode)
		printf("The client %s is executing DbWork::getSequenceData(%s,...)\n", clientName, command);
	else
		printf("Client %s retrieves sequence data\n", clientName);
	index = atoi(command);
	if (!DbWork::getSequenceData(index, maxValue, minValue, &increment, cycleFlag))
	{
		sendMessage = "ERROR\nUnable to get sequence detail\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += maxValue;
		sendMessage += "\n";
		sendMessage += minValue;
		sendMessage += "\n";
		sprintf(transBuffer, "%ld", increment);
		sendMessage += transBuffer;
		sendMessage += "\n";
		sendMessage += cycleFlag;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendDETAIL_SYNONYM(int actSocket, const char* clientName, char* buffer)
{
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	index;
	KString	tableOwner;
	KString	tableName;
	KString	dbLink;

	command = &(buffer[15]);
	if (debug_mode)
		printf("The client %s is executing DbWork::getSynonymData(%s,...)\n", clientName, command);
	else
		printf("Client %s retrieves synonym data\n", clientName);
	index = atoi(command);
	if (!DbWork::getSynonymData(index, tableOwner, tableName, dbLink))
	{
		sendMessage = "ERROR\nUnable to get synonym detail\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += tableOwner;
		sendMessage += "\n";
		sendMessage += tableName;
		sendMessage += "\n";
		sendMessage += dbLink;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendDETAIL_TABLE(int actSocket, const char* clientName, char* buffer)
{
	KString	message;
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	index;
	KString	tableData;

	command = &(buffer[13]);
	if (debug_mode)
		printf("The client %s is executing DbWork::getTableData(%s,...)\n", clientName, command);
	else
		printf("Client %s retrieves table data\n", clientName);
	index = atoi(command);
	if (!DbWork::getTableData(index, message, tableData))
	{
		sendMessage = "ERROR\nUnable to get table detail\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += tableData;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendDETAIL_PACKAGE(int actSocket, const char* clientName, char* buffer)
{
	KString	message;
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	index;
	KString	packageName;
	KString	packageDeclData;
	KString packageDeclDataRaw;
	KString	packageBodyData;
	KString	packageBodyDataRaw;

	command = &(buffer[15]);
	if (debug_mode)
		printf("The client %s is executing DbWork::getPackageData(%s,...)\n", clientName, command);
	else
		printf("Client %s retrieves package data\n", clientName);
	index = atoi(command);
	if (!DbWork::getPackageData(index, packageName, message, packageDeclData, packageDeclDataRaw,
				    packageBodyData, packageBodyDataRaw))
	{
		sendMessage = "ERROR\nUnable to get package detail\n";
		sendMessage += message;
		sendMessage += "\n";
		koraSend(actSocket, sendMessage);
		printf("Action from client %s failed.\n", clientName);
	}
	else
	{
		sendMessage = "OK\n";
		sendMessage += packageName;
		sendMessage += "\n";
		sendMessage += "DECL\n";
		sendMessage += packageDeclData;
		sendMessage += "\nDECL RAW\n";
		sendMessage += packageDeclDataRaw;
		sendMessage += "\nBODY\n";
		sendMessage += packageBodyData;
		sendMessage += "\nBODY RAW\n";
		sendMessage += packageBodyDataRaw;
		sendMessage += "\n";
		numbytes = koraSend(actSocket, sendMessage);
		printf("Action from client %s successfully executed.\n", clientName);
	}
}

void sendERROR_PACKAGE(int actSocket, const char* clientName, char* buffer)
{
	char	param1[101];
	char	param2[101];
	char	param3[101];
	KString	sendMessage;
	char*	command;

	int	numbytes;
	int	index;
	KString	result;

	command = &(buffer[14]);
	getParam(command, param1, param2, param3);
	if (debug_mode)
		printf("The client %s is executing DbWork::getPlSqlCompileError(%s,%s)\n", clientName, param2, param1);
	else
		printf("Client %s retrieves possibly found PL/SQL error message\n", clientName);
	index = atoi(param2);
	result = DbWork::getPlSqlCompileError(index, param1);
	sendMessage = "OK\n";
	sendMessage += result;
	sendMessage += "\n";
	numbytes = koraSend(actSocket, sendMessage);
	printf("Action from client %s successfully executed.\n", clientName);
}

void sendQUIT(int actSocket, const char* clientName)
{
	printf("Good bye %s\n", clientName);
	koraSend(actSocket, "BYE\n");
}

	

