/*
    File: smif.h
    Version: 0.3.1
    Date: 31-DEC-1998
    Copyright: Bernhard Kuhn <kuhn@lpr.e-technik.tu-muenchen.de>

    The Serial Monitor InterFace (SMIF) provides an OO-access
    to the procedural serial monitor utility "sermon-0.23" written
    by Peter Fox <fox@roestock.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/



#ifndef SMIF_H
#define SMIF_H



// include several C-standard header files and

#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>

// define the special IOCTL to support serialmon
#if defined __GLIBC__ && __GLIBC__ >= 2

#include <errno.h>
#include <sys/time.h>

#define TIOCSERGSTATS   0x54f3 /* Get interrupt statistics */

struct serial_stats_struct {
       int count;
       int rx;
       int tx;
       struct timeval ts;
};

#else /* libc5 */

#include <termios.h>
#define __KERNEL__
#include <linux/serial.h>       /* For the definition of serial_stats_struct */

#endif



// include Qt parent class definition and timer
#include <qobject.h>
#include <qdatetm.h>
#include <qstring.h>



// define Serial Monitor Interface (smif)

class smif : public QObject {

  Q_OBJECT

public:

  // constructor opens device
  smif(char* device=0,int interval=100,int tau=500);

  // destructor
  ~smif();

  // get line status
  bool getRX();
  bool getTX();
  bool getRTS();
  bool getCTS();
  bool getDTR();
  bool getDSR();
  bool getDCD();
  bool getRI();

  // determine i/o-transfer rates
  float getRXrate();
  float getTXrate();

  // determine i/o-transfer amounts
  float getRXcount();
  float getTXcount();

  // check input device and update interval
  const char* getDevice();
  int getInterval();
  int getTau();

protected:

  // periodic update
  void timerEvent(QTimerEvent*);

private:

  QString device;
  int interval;
  int tau;
  int fd;
  int status;
  bool rx,tx,rts,cts,dtr,dsr,dcd,ri;
  float rx_rate,tx_rate,rx_count,tx_count;
  float rxf_rate,txf_rate;
  struct serial_stats_struct info;
  QTime timer;
  int elapsed;
  bool tmp;
  float diff,rate;
  int timerID;
  
signals:

  // inform, when update occured
  void updated();

  // inform, when line states changed
  void RXstatusChanged(bool);
  void TXstatusChanged(bool);
  void RTSstatusChanged(bool);
  void CTSstatusChanged(bool);
  void DTRstatusChanged(bool);
  void DSRstatusChanged(bool);
  void DCDstatusChanged(bool);
  void RIstatusChanged(bool);

  // inform, when transfer rates changed
  void RXrateChanged(float);
  void TXrateChanged(float);

  // inform, when tranfer amount changed
  void RXcountChanged(float);
  void TXcountChanged(float);

public slots:

  // change input device
  void setDevice(const char*);

  // change update interval
  void setInterval(int);  

  // change tau
  void setTau(int);

};



#endif
