/*

    File: ksermon.h
    Version: 0.3.0
    Date: 31-DEC-1998
    Copyright: Bernhard Kuhn <kuhn@lpr.e-technik.tu-muenchen.de>

    ksermon is an system-utility to watch the activities on a given serial
    port. The current state of the TX,RX,CTS,RTS,DTR,DSR,DCD and RI lines are
    shown by some kind of LEDs. Furthermore, the current TX/RX transfer rates
    are visualisized using bars and within an osziloscope. Additionaly,
    the amount of received and transmitted data since programm start
    is displayed.
    
    This utility is set on top of the kernelpatch done by sermon-23 from
    Peter Fox (fox@roestock.demon.co.uk).
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/



// get class definitions

#include <qlayout.h> 
#include <qfont.h>

#include "fixedlabel.h"
#include "ksermon.h"
#include "ledtoolbar.h"

#include <kapp.h>
#include <klocale.h>


#include <iostream.h>


// mykapp is declared in main.cpp

extern KApplication* mykapp;



// constructor

ksermon::ksermon(char* name="") : KTopLevelWidget(name) {

 // create a serial monitor interface (smif)
  smon=new smif();

  // create Leds-Toolbar and add it to ksermon
  ledbar=new LedToolBar(smon,this);
  ledbarID = addToolBar(ledbar);

  // create Toolbar for Texts (TX/RX-count & online timer)
  // and add it to ksermon
  textbar=new TextToolBar(smon,this);
  textbarID = addToolBar(textbar);

  
  // ********** TX/RX-Bars and Oszi **********

  // create ksermon's Subframe
  QFrame* frm=new QFrame(this);

  // create V-Box to ensure 4pt space between tool-bars and frame
  QVBoxLayout* mainLayout = new QVBoxLayout(frm);
  mainLayout->addSpacing(4);

  // create H-Box for automatic placement of TX/RX-Bars and oszi
  QHBoxLayout* layout=new QHBoxLayout();
  mainLayout->addLayout(layout);

  // crate bar for TX-rate  
  txbar=new bar(frm,150,"TX-Bar",blue);
  layout->addWidget(txbar,1);
  layout->addSpacing(4);
  connect(smon,SIGNAL(TXrateChanged(float)),txbar,SLOT(setBar(float)));

  // crate bar for RX-rate  
  rxbar=new bar(frm,150,"RX-Bar",green);
  layout->addWidget(rxbar,1);
  layout->addSpacing(4);
  connect(smon,SIGNAL(RXrateChanged(float)),rxbar,SLOT(setBar(float)));

  // create osziloscope for TX/RX-rate  
  txrxoszi=new oszi(frm,150,"TX/RX-Oszi",blue,green);
  layout->addWidget(txrxoszi,6);
  connect(smon,SIGNAL(TXrateChanged(float)),txrxoszi,SLOT(setY1(float)));
  connect(smon,SIGNAL(RXrateChanged(float)),txrxoszi,SLOT(setY2(float)));
  connect(smon,SIGNAL(updated()),txrxoszi,SLOT(advance()));

  // ********** Menu-Bar **********

  // create "file"-menu
  QPopupMenu* file = new QPopupMenu();
  file->insertSeparator(-1);
  file->insertItem(klocale->translate("&Exit"),kapp,SLOT(quit()));

  // create "options"-menu
  options = new QPopupMenu();

  options->insertItem(klocale->translate("&Preferences"),
		      this,SLOT(settings()));
  options->insertSeparator(-1);

  hide_ledbar=false;
  ledbmnID = options->insertItem(klocale->translate("Hide &Led Bar"),
				 this,SLOT(toggleLedBar()));

  hide_textbar=false;
  textbmnID = options->insertItem(klocale->translate("Hide &Text Bar"),
				  this,SLOT(toggleTextBar()));
  
  options->insertSeparator(-1);
  options->insertItem(klocale->translate("Sa&ve Options"),
		      this,SLOT(writeSettings()));

  //create "help"-menu
  QPopupMenu *help = kapp->getHelpMenu(true,"KSermon 0.3.1\n\nby B. Kuhn");

  // create menubar and add it to ksermon
  menubar = new KMenuBar(this, "menubar");
  menubar->insertItem(klocale->translate("&File"),file);
  menubar->insertItem(klocale->translate("&Options"),options);
  menubar->insertItem(klocale->translate("&Help"),help);
  setMenu(menubar);


  // ********** Preferences **********

  // create the preference widget
  preferences = new pref();

  // attach preferences to tx/rx-bars and oszi
  connect(preferences,SIGNAL(rateChanged(float)),
	  txrxoszi,SLOT(setMaximum(float)));
  connect(preferences,SIGNAL(rateChanged(float)),
	  txbar,SLOT(setMaximum(float)));
  connect(preferences,SIGNAL(rateChanged(float)),
	  rxbar,SLOT(setMaximum(float)));
  connect(preferences,SIGNAL(deviceChanged(const char*)),
	  smon,SLOT(setDevice(const char*)));
  connect(preferences,SIGNAL(intervalChanged(int)),
	  smon,SLOT(setInterval(int)));
  connect(preferences,SIGNAL(tauChanged(int)),
	  smon,SLOT(setTau(int)));
  connect(preferences,SIGNAL(doSave()),
	  this,SLOT(writeSettings()));

  // inform preferences if ksermon changes settings
  connect(this,SIGNAL(deviceChanged(const char*)),
	  preferences,SLOT(deviceChangedIn(const char*)));
  connect(this,SIGNAL(maximumChanged(float)),
	  preferences,SLOT(rateChangedIn(float)));
  connect(this,SIGNAL(intervalChanged(int)),
	  preferences,SLOT(intervalChangedIn(int)));
  connect(this,SIGNAL(tauChanged(int)),
	  preferences,SLOT(tauChangedIn(int)));

 
  // ********** ksermon appearence **********

  // set Subframe for TopLevelWidget
  setView(frm);

  // window/widget title
  setCaption("KSermon 0.3.1");

  // read all settings (device,maximum,interval,apperance etc.)
  readSettings();

};



// toggle visibility of the led toolbar

void ksermon::toggleLedBar() {

  if(hide_ledbar) {
    hide_ledbar=FALSE;
    enableToolBar(KToolBar::Show,ledbarID);
    options->changeItem("Hide &Led Bar",ledbmnID);
  } 
  else {
    hide_ledbar=TRUE;
    enableToolBar(KToolBar::Hide,ledbarID);
    options->changeItem("Show &Led Bar",ledbmnID);
  }  

};



// toggle visibility of the text toolbar

void ksermon::toggleTextBar() {

  if(hide_textbar) {
    hide_textbar=FALSE;
    enableToolBar(KToolBar::Show,textbarID);
    options->changeItem(klocale->translate("Hide &Text Bar"),textbmnID);
  } 
  else {
    hide_textbar=TRUE;
    enableToolBar(KToolBar::Hide,textbarID);
    options->changeItem(klocale->translate("Show &Text Bar"),textbmnID);
  }  

};



// show the preference widget

void ksermon::settings() {
  preferences->show();
  preferences->raise();
};



// write current settings

void ksermon::writeSettings() {
  KConfig* config = mykapp->getConfig();


  // general settings

  config->setGroup( "General Settings" );

  config->writeEntry("Device",smon->getDevice());
  config->writeEntry("Maximum",(int)txrxoszi->getMaximum());
  config->writeEntry("Interval",smon->getInterval());
  config->writeEntry("Tau",smon->getTau());
  config->writeEntry("X",x());
  config->writeEntry("Y",y());
  config->writeEntry("Width",width());
  config->writeEntry("Height",height());


  // led toolbar

  config->setGroup("LedBar Apperance");

  config->writeEntry("Hidden",hide_ledbar ? "Yes" : "No");

  switch(ledbar->barPos()) {
  case KToolBar::Left:
    config->writeEntry("Position","Left");
    break;
  case KToolBar::Right:
    config->writeEntry("Position","Right");
    break;
  case KToolBar::Top:
    config->writeEntry("Position","Top");
    break;
  case KToolBar::Bottom:
    config->writeEntry("Position","Bottom");
    break;
  case KToolBar::Floating:
    config->writeEntry("Position","Floating");
    break;
  };

  config->writeEntry("X",ledbar->x());
  config->writeEntry("Y",ledbar->y());
  config->writeEntry("Width",ledbar->width());
  config->writeEntry("Height",ledbar->height());


  // text toolbar

  config->setGroup("TextBar Apperance");

  config->writeEntry("Hidden",hide_textbar ? "Yes" : "No");

  switch(textbar->barPos()) {
  case KToolBar::Left:
    config->writeEntry("Position","Left");
    break;
  case KToolBar::Right:
    config->writeEntry("Position","Right");
    break;
  case KToolBar::Top:
    config->writeEntry("Position","Top");
    break;
  case KToolBar::Bottom:
    config->writeEntry("Position","Bottom");
    break;
  case KToolBar::Floating:
    config->writeEntry("Position","Floating");
    break;
  };

  config->writeEntry("X",textbar->x());
  config->writeEntry("Y",textbar->y());
  config->writeEntry("Width",textbar->width());
  config->writeEntry("Height",textbar->height());

  // finalize writting
  config->sync();
};



// read current settings

void ksermon::readSettings() {

  KConfig* config = mykapp->getConfig();

  // general settings

  config->setGroup( "General Settings" );

  emit deviceChanged(config->readEntry("Device","/dev/ttyS0"));
  emit maximumChanged((float)config->readNumEntry("Maximum",200));
  emit intervalChanged(config->readNumEntry("Interval",100));
  emit tauChanged(config->readNumEntry("Tau",500));

  int appx=config->readNumEntry("X",-1);
  int appy=config->readNumEntry("Y",-1);
  int appwidth=config->readNumEntry("Width",232);
  int appheight=config->readNumEntry("Height",150);
  if(appx != -1) {
    setGeometry(appx,appy,appwidth,appheight);
  }
  else {
    resize(appwidth,appheight);
  };

  // led toolbar

  config->setGroup("LedBar Apperance");

  if(hide_ledbar!=strcmp(config->readEntry("Hidden","No"),"No")) {
    toggleLedBar();
  };

  const char* ledstr=config->readEntry("Position","Top");
  KToolBar::BarPosition ledpos=KToolBar::Top;
  if(strcmp(ledstr,"Left")==0) ledpos=KToolBar::Left;
  if(strcmp(ledstr,"Right")==0) ledpos=KToolBar::Right;
  if(strcmp(ledstr,"Top")==0) ledpos=KToolBar::Top;
  if(strcmp(ledstr,"Bottom")==0) ledpos=KToolBar::Bottom;
  if(strcmp(ledstr,"Floating")==0) ledpos=KToolBar::Floating;
  ledbar->setBarPos(ledpos);


  if(ledpos==KToolBar::Floating) {
    int ledx=config->readNumEntry("X");
    int ledy=config->readNumEntry("Y");
    int ledwidth=config->readNumEntry("Width");
    int ledheight=config->readNumEntry("Height");
    ledbar->setGeometry(ledx,ledy,ledwidth,ledheight);
    if(!hide_ledbar) ledbar->show();
  };


  // text toolbar

  config->setGroup("TextBar Apperance");

  if(hide_textbar!=strcmp(config->readEntry("Hidden","No"),"No")) {
    toggleTextBar();
  };
  
  const char* textstr=config->readEntry("Position","Top");
  KToolBar::BarPosition textpos=KToolBar::Top;
  if(strcmp(textstr,"Left")==0) textpos=KToolBar::Left;
  if(strcmp(textstr,"Right")==0) textpos=KToolBar::Right;
  if(strcmp(textstr,"Top")==0) textpos=KToolBar::Top;
  if(strcmp(textstr,"Bottom")==0) textpos=KToolBar::Bottom;
  if(strcmp(textstr,"Floating")==0) textpos=KToolBar::Floating;
  textbar->setBarPos(textpos);
 
  if(textpos==KToolBar::Floating) {
    int textx=config->readNumEntry("X");
    int texty=config->readNumEntry("Y");
    int textwidth=config->readNumEntry("Width");
    int textheight=config->readNumEntry("Height");
    textbar->setGeometry(textx,texty,textwidth,textheight);
    if(!hide_textbar) textbar->show();
  };

};












