/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [ansiterm.cpp]          ANSI/VT100 Terminal Emulation                      */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU Public Software Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

//FIXME: Ok, this code is to be about to be hacked to death.
//       We have to make proper modules, soon.
/*FIXME: Speedup
           General Technics
           - use a sort of "blinking cursor" together with "bulk mode"
             The cursor is deactivated once a bulk started
             All repaint-events in a bulk are delayed until the bulk ends
             A bulk either ends with a timeout (10ms?) or
             after a round of MAXBULK received tokens.
*/

#include <X11/Xlib.h>
#include "ansiterm.h"
#include <qpainter.h>
#include <qkeycode.h>

#include <stdio.h>
#include <unistd.h>

#include <assert.h>

#include "modem.h"

#include "ansiterm.moc"

static FILE* log = NULL;

#define ESC 27
#define MIN(a,b) ((a)<(b)?(a):(b))
#define MAX(a,b) ((a)>(b)?(a):(b))

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

#define BS_CLEARS TRUE

// Default Rendition
#define DEFAULT_BACK black
#define DEFAULT_FORE darkGray 

AnsiTerminal::AnsiTerminal( Modem* mo, QWidget *parent, const char *name )
    : QWidget( parent, name )
{
scan_buffer_init();
tmargin=0;
bmargin=lines-1;
bulk_rect = QRect(0,0,0,0);
bulk_cu_on = FALSE;
cu_on = FALSE;
QObject::connect(&bulk_timer, SIGNAL(timeout()), this, SLOT(show_bulk()) );
   this->mo = mo;
#if defined(_WS_X11_)
   // we only have a poor 8x16 raster font
   f.setFamily( "vga" );
   f.setRawMode( TRUE );
   if ( !f.exactMatch() )
     debug( "Sorry, could not find the X specific font 'vga'" );
#endif
    setFixedSize( columns*font_w, lines*font_h );
    reset();
    if ( parent ) parent->installEventFilter( this ); //FIXME: see below
    setBackgroundColor( DEFAULT_BACK );
  cu_fg = DEFAULT_FORE;
  cu_bg = DEFAULT_BACK;
  toggleCursor(TRUE);
  connect( mo, SIGNAL(data_in(int)), this, SLOT(rcv_byte(int)) );
  active = TRUE;
}

static const QColor color_table[2][8] =
// The following are almost IBM standard color codes, with some slight
// gamma correction for the dim colors to compensate for bright X screens.
// (thanks dosemu)
{
  { QColor(0x00,0x00,0x00),QColor(0x18,0x18,0xB2),
    QColor(0x18,0xB2,0x18),QColor(0x18,0xB2,0xB2),
    QColor(0xB2,0x18,0x18),QColor(0xB2,0x18,0xB2),
    QColor(0xB2,0x68,0x18),QColor(0xB2,0xB2,0xB2) },
  { QColor(0x68,0x68,0x68),QColor(0x54,0x54,0xFF),
    QColor(0x54,0xFF,0x54),QColor(0x54,0xFF,0xFF),
    QColor(0xFF,0x54,0x54),QColor(0xFF,0x54,0xFF),
    QColor(0xFF,0xFF,0x54),QColor(0xFF,0xFF,0xFF) }
};

/* The interpretation of the DOS color attributes
 *
 *   Bit: 0   Foreground blue
 *   Bit: 1   Foreground green
 *   Bit: 2   Foreground red
 *   Bit: 3   Foreground bold (intensity bit)
 *   Bit: 4   Background blue
 *   Bit: 5   Background green
 *   Bit: 6   Background red
 *   Bit: 7   blinking bit  (see below)
 * 
 * Foreground and Background bits have a different interpretation:
 * 
 *    Foreground   Background    Interpretation
 *      111          000           Normal white on black
 *      000          111           Reverse video (white on black)
 *      000          000           Invisible characters
 *      001          000           Underline
 *     anything else is invalid.
 */    

/* Decoding Ansi Colors

   ANSI and IBMPC colors are permuted
   
   Code ANSI(bgr) IBMPC(rgb)
   ---- --------- ----------
   0    Black     Black
   1    Red       Blue
   2    Green     Green
   3    Yellow    Cyan
   4    Blue      Red
   5    Magenta   Magenta
   6    Cyan      Yellow
   7    White     White
*/

int decode_ansi_color[8] = { 0,4,2,6, 1,5,3,7 };

// ------------------------------

void AnsiTerminal::clear()
{ 
  cu_fg = DEFAULT_FORE;
  cu_bg = DEFAULT_BACK;
  for(int x = 0; x < columns; x++)
  for(int y = 0; y < lines; y++)
  {
    image[x][y].c = ' ';
    image[x][y].f = cu_fg;
    image[x][y].b = cu_bg;
  }
  cu_x = 0;
  cu_y = 0;
/*Colortest
for (int x = 0; x < 8; x++)
for (int y = 0; y < 16; y++)
{
    image[3*x+0][y].c = ' ';
    image[3*x+1][y].c = '*';
    image[3*x+2][y].c = ' ';
    image[3*x+0][y].b = color_table[0][x];
    image[3*x+1][y].b = color_table[0][x];
    image[3*x+2][y].b = color_table[0][x];
    image[3*x+0][y].f = color_table[y/8][y%8];
    image[3*x+1][y].f = color_table[y/8][y%8];
    image[3*x+2][y].f = color_table[y/8][y%8];
  
}
*/
}

void AnsiTerminal::home()
{
  cu_x = 0;
  cu_y = 0;
}

void AnsiTerminal::reset()
{
  cu_fg   = DEFAULT_FORE;
  cu_bg   = DEFAULT_BACK;
  cu_bold = 0;
  scan_buffer_init();
  clear();
  home();
}

void AnsiTerminal::scroll_up()
{
printf("tmargin %d, bmargin %d\n",tmargin,bmargin);
  for(int x = 0; x < columns; x++)
  for(int y = tmargin; y < bmargin; y++)
  {
    image[x][y].c = image[x][y+1].c;
    image[x][y].f = image[x][y+1].f;
    image[x][y].b = image[x][y+1].b;
  }
  for(int x = 0; x < columns; x++)
  {
    image[x][bmargin].c = ' ';
    image[x][bmargin].f = cu_fg;
    image[x][bmargin].b = cu_bg;
  }
  repaint_range(0*8,0*16,80*8,25*16,FALSE); //FIXME: respect t/b-margin
}

void AnsiTerminal::scroll_down()
{
printf("tmargin %d, bmargin %d\n",tmargin,bmargin);
  for(int x = 0; x < columns; x++)
  for(int y = bmargin-1; y >= tmargin; y--)
  {
    image[x][y+1].c = image[x][y].c;
    image[x][y+1].f = image[x][y].f;
    image[x][y+1].b = image[x][y].b;
  }
  for(int x = 0; x < columns; x++)
  {
    image[x][tmargin].c = ' ';
    image[x][tmargin].f = cu_fg;
    image[x][tmargin].b = cu_bg;
  }
  repaint_range(0*8,0*16,80*8,25*16,FALSE);//FIXME: respect t/b-margin
}

void AnsiTerminal::BackSpace()
{
  cu_x = MAX(0,cu_x-1);
  if (BS_CLEARS) image[cu_x][cu_y].c = ' ';
}

void AnsiTerminal::Tabulate()
{
  ShowCharacter(' ');
  while(cu_x%8 != 0) ShowCharacter(' ');
}

void AnsiTerminal::NewLine()
{
  cu_x  = tmargin;
  cu_y += 1;
  if (cu_y > bmargin)
  {
    scroll_up();
    cu_y = bmargin;
  }  
}

void AnsiTerminal::Return()
{
  cu_x = 0;
}

void AnsiTerminal::ShowCharacter(unsigned char c)
{
  image[cu_x][cu_y].c = c;
  image[cu_x][cu_y].f = cu_fg;
  image[cu_x][cu_y].b = cu_bg;
  repaint_range(cu_x*8,cu_y*16,8,16,FALSE);
  cu_x += 1;
  if (cu_x >= columns)
  {
    cu_x = 0;
    cu_y += 1;
    if (cu_y >= lines)
    {
      scroll_up();
      cu_y = lines-1;
    }
  }
}

void AnsiTerminal::ReportTerminalType()
{
  mo->send_string("\033[?c"); // I'm a BBS Ansi Terminal
}

void AnsiTerminal::toggleCursor(bool on)
{
  
  if (cu_on != on)    // all these conditions are for speed-up
  {
    cu_on = on;
    if (bulk_cu_on)
    {
      QColor c = image[cu_x][cu_y].b;
      image[cu_x][cu_y].b = image[cu_x][cu_y].f;
      image[cu_x][cu_y].f = c;
      repaint_range(cu_x*8,cu_y*16,8,16,FALSE);
      bulk_cu_on = on;
    }
  }
}

void AnsiTerminal::repaint_range(int x, int y, int w, int h, bool b)
{
  if (!bulk_timer.isActive()) bulk_timer.start(50,TRUE);
  bulk_rect = bulk_rect.unite(QRect(x,y,w,h));
}

void AnsiTerminal::show_bulk()
{
  if (bulk_cu_on != cu_on && cu_on)
  { // eventually switch on
    QColor c = image[cu_x][cu_y].b;
    image[cu_x][cu_y].b = image[cu_x][cu_y].f;
    image[cu_x][cu_y].f = c;
    bulk_rect = bulk_rect.unite(QRect(cu_x*8,cu_y*16,8,16));
    bulk_cu_on = cu_on;
  }
  repaint(bulk_rect,FALSE);
  bulk_rect = QRect(0,0,0,0);
  if (bulk_timer.isActive()) bulk_timer.stop();
}

// ------------------------------

void AnsiTerminal::scan_buffer_add(int c)
{
  scan_buffer[scan_buffer_pos++] = c; 
}

void AnsiTerminal::scan_buffer_init()
{
  scan_buffer_state = 0;
  scan_buffer_pos = 0;
}

void AnsiTerminal::scan_buffer_report()
{ int i;
  if (scan_buffer_pos == 0 ||
      scan_buffer_pos == 1 && (scan_buffer[0] & 0xff) >= 32) return;
  printf("token: ");
  for (i = 0; i < scan_buffer_pos; i++)
  {
    if (scan_buffer[i] == '\\') 
      printf("\\\\");
    else
    if (scan_buffer[i] > 32) 
      printf("%c",scan_buffer[i]);
    else
      printf("\\%02x",scan_buffer[i]&0xff);
  }
  printf("\n");
}

void AnsiTerminal::ReportErrorToken()
{
  printf("undecodable control sequence: "); scan_buffer_report();
}

void AnsiTerminal::Interprete_1(unsigned char code)
// inteprete: code
{
  switch (code)
  {
  case 0x24    : break; // 'ZDLE' send by ZMODEM. Silently ignored.
  case '\b'    : BackSpace();            break;
  case '\t'    : Tabulate();             break;
  case '\n'    : NewLine();              break;
  case '\r'    : Return();               break;
  case 0x07    : XBell(qt_xdisplay(),0); break; //FIXME: Bell()
  default      : ReportErrorToken();     break;
  };
}

void AnsiTerminal::Interprete_2(unsigned char code)
// inteprete: ESC code
{
  switch (code)
  {
  case 'Z': ReportTerminalType(); break;	
  default : ReportErrorToken();   break;
  };
}

#define MAXARGS 15
void AnsiTerminal::Interprete_3(int len, unsigned char token[])
// interprete: ESC [ token
{ int argv[MAXARGS]; int argc; int i;
  argv[0]=0; argc=0;
  for (i = 0; i < len-1; i++)
  {
    switch (token[i])
    {
      case '0' : case '1' : case '2' : case '3' : case '4' :
      case '5' : case '6' : case '7' : case '8' : case '9' :
        argv[argc] = 10*(argv[argc]) + token[i] - '0';
        break;
      case ';' :
        argc = MIN(argc+1,MAXARGS-1);
        argv[argc] = 0;
        break;
      default: //FIXME: handle other cases as error
        ReportErrorToken(); return;
    }
  }
  switch(token[len-1])
  {
  case 'A' : // ESC [ Pn A               Cursor Up
    argv[0] = MAX(1,argv[0]);
    cu_y = MIN(lines-1,MAX(0,cu_y-argv[0]));
    break;
  case 'B' : // ESC [ Pn B               Cursor Down
    argv[0] = MAX(1,argv[0]);
    cu_y = MIN(lines-1,MAX(0,cu_y+argv[0]));
    break;
  case 'C' : // ESC [ Pn C               Cursor Right
    argv[0] = MAX(1,argv[0]);
    cu_x = MIN(columns-1,MAX(0,cu_x+argv[0]));
    break;
  case 'D' : // ESC [ Pn D               Cursor Left
    argv[0] = MAX(1,argv[0]);
    cu_x = MIN(columns-1,MAX(0,cu_x-argv[0]));
    break;
  case 'm' : // ESC [ Ps ; ... ; Ps m    Select Graphic Rendition
    for (i = 0; i <= argc; i++)
    { int arg = argv[i];
      if ( 30 <= arg && arg <= 37 )
        cu_fg = color_table[cu_bold][decode_ansi_color[arg-30]];
      else
      if ( 40 <= arg && arg <= 47 )
        cu_bg = color_table[0][decode_ansi_color[arg-40]];
      else
      switch (arg)
      {
        case 0  : /* default Rendition */ 
          cu_fg = DEFAULT_FORE;
          cu_bg = DEFAULT_BACK;
          cu_bold = 0;
          break;
        case 1  : /* bold */ 
          cu_bold = 1;
          break;
        case 10 : /* FIXME: unknown (linux console?) */ 
        case 11 : /* FIXME: unknown (linux console?) */ 
          break;
        default : printf("--------> attribute %d\n",arg); break;
      }
    }
    break;
  case 'f' : // ESC [ Py ; Px f    Direct Cursor Addressing
  case 'H' : // ESC [ Py ; Px H    Direct Cursor Addressing
    {
      int y; y = MAX(1,argv[0]);
      int x; x = argc<1?1:MAX(1,argv[1]);
      cu_x = MIN(columns-1,MAX(0,x-1));
      cu_y = MIN(lines-1,  MAX(0,y-1));
    }
    break;
  case 'K' : // ESC [ Pn K         Line Erasing
    switch (argv[0])
    {
      case 0: // ESC [ 0 K         Clear to end of line
        for (i = cu_x; i < columns; i++)
        {
          image[i][cu_y].c = ' ';
          image[i][cu_y].f = cu_fg;
          image[i][cu_y].b = cu_bg;
        }
        repaint_range(cu_x*8,cu_y*16,8*(columns-cu_x),16,FALSE);
        break;
      case 1: // ESC [ 1 K         Clear to begin of line
        for (i = 0; i <= cu_x; i++)
        {
          image[i][cu_y].c = ' ';
          image[i][cu_y].f = cu_fg;
          image[i][cu_y].b = cu_bg;
        }
        repaint_range(0,cu_y*16,8*cu_x,16,FALSE);
        break;
      case 2: // ESC [ 2 K         Clear entire      line
        for (i = 0; i < columns; i++)
        {
          image[i][cu_y].c = ' ';
          image[i][cu_y].f = cu_fg;
          image[i][cu_y].b = cu_bg;
        }
        repaint_range(0,cu_y*16,8*columns,16,FALSE);
        break;
    }
    break;
  case 'J' : // ESC [ Pn J         Screen Erasing
    switch (argv[0])
    {
      case 0: // ESC [ 0 J         Clear to end of screen
        for (int x = cu_x; x < columns; x++)
        {
          image[x][cu_y].c = ' ';
          image[x][cu_y].f = cu_fg;
          image[x][cu_y].b = cu_bg;
        }
        repaint_range(cu_x*8,cu_y*16,8*(columns-cu_x),16,FALSE);
        for (int y = cu_y+1; y < lines; y++)
        for (int x = 0; x < columns; x++)
        {
          image[x][y].c = ' ';
          image[x][y].f = cu_fg;
          image[x][y].b = cu_bg;
        }
        repaint_range(0,(cu_y+1)*16,8*columns,16*(lines-1-cu_y),FALSE);
        break;
      case 1: // ESC [ 1 K         Clear to begin of screen
        for (i = 0; i <= cu_x; i++)
        {
          image[i][cu_y].c = ' ';
          image[i][cu_y].f = cu_fg;
          image[i][cu_y].b = cu_bg;
        }
        repaint_range(0,cu_y*16,8*cu_x,16,FALSE);
        for (int y = 0; y < cu_y; y++)
        for (int x = 0; x < columns; x++)
        {
          image[x][y].c = ' ';
          image[x][y].f = cu_fg;
          image[x][y].b = cu_bg;
        }
        repaint_range(0,0,8*columns,16*(cu_y-1),FALSE);
        break;
      case 2: /* ESC [ 2 K      Clear entire screen (+ cu home)
        for (int y = 0; y < lines; y++)
        for (int x = 0; x < columns; x++)
        {
          image[x][y].c = ' ';
          image[x][y].f = cu_fg;
          image[x][y].b = cu_bg;
        }
        repaint_range(0,0,8*columns,16*lines,FALSE);
        cu_x = 0; cu_y = 0; /* includes 'home' */
        break;
      default:
        break;
    }
    break;
  case 'n' : // ESC [ Pn n          Report
    switch(argv[0])
    { char tmp[20];
    case 5:  // ESC [ 5 n   Status
      mo->send_string("\033[0n");
      break;
    case 6:  // ESC [ 6 n   Cursor Position
      sprintf(tmp,"\033[%d;%dR",cu_y+1,cu_x+1);
      mo->send_string(tmp);
    }
    break;
/*FIXME: not yet fool proof
    case 'r' : // ESC [ Pn ; Pn r
  printf("set region: %d %d\n",argv[0]-1,argv[1]-1);
      //FIXME: check for good values
      tmargin = argv[0]-1;
      bmargin = argv[1]-1;
      break;
    case 'L' : // ESC [ Pn L
  printf("insert line: %d\n",argv[0]);
      scroll_down();
      break;
    case 'M' : // ESC [ Pn M
  printf("insert line: %d\n",argv[0]);
      scroll_up();
      break;
*/
  default  : ReportErrorToken(); break;
  }
}

void AnsiTerminal::interprete(int len, unsigned char token[])
// interprete control code (or plain character)
{
//scan_buffer_report();
  if (len == 0) return;
toggleCursor(FALSE);
  if (len == 1 && token[0] >= 32)  ShowCharacter(token[0]); // type character
  if (len == 1 && token[0] <  32)  Interprete_1(token[0]);  // control code
  if (len == 2 && token[0] == ESC) Interprete_2(token[1]); // ESC code
  if (len >= 2 && token[0] != ESC) Interprete_3(len-1,token+1); // ESC[ ... code
  if (len >= 3)                    Interprete_3(len-2,token+2); // ESC[ ... code
toggleCursor(TRUE);
}

void AnsiTerminal::rcv_byte(int c)
// process application input to terminal
// this is a trivial scanner
{ 
  if (!active) return;
  c &= 0xff;
  scan_buffer_add(c);
  switch (scan_buffer_state)
  {
  case 0 : /* init */
    switch (c)
    {
    case 128+ESC : scan_buffer_state = 2; break; // ESC+128 == ESC [
    case ESC : scan_buffer_state = 1; break;
    default  : interprete(scan_buffer_pos,scan_buffer); scan_buffer_init();
               break;
    };
    break;
  case 1 : // ESC
    switch (c)
    {
    case '[' : scan_buffer_state = 2; break;
    default  : interprete(scan_buffer_pos,scan_buffer); scan_buffer_init();
               break;
    };
    break;
  case 2 : // ESC [
    if ((c & 0xf0) != 0x30)
    { interprete(scan_buffer_pos,scan_buffer); scan_buffer_init(); break; }
    break;
  default:
    assert(0);
  }
}

void AnsiTerminal::paintEvent( QPaintEvent* pe )
{
  QPainter paint;
  setUpdatesEnabled(FALSE);
  paint.begin( this );			// begin painting
  paint.setFont( f );
  paint.setBackgroundMode( OpaqueMode );
  int lux = pe->rect().left()   / 8;  //  8 == font_w
  int rlx = pe->rect().right()  / 8;  //  8 == font_w
  int luy = pe->rect().top()    / 16; // 16 == font_h
  int rly = pe->rect().bottom() / 16; // 16 == font_h
//printf("paintEvent: %d..%d, %d..%d (%d..%d, %d..%d)\n",lux,rlx,luy,rly,
//pe->rect().left(), pe->rect().right(), pe->rect().top(), pe->rect().bottom());
//FIXME: this can further be improved by using a scan line alg. for the colors
  for (int x = lux; x <= rlx; x++)
  for (int y = luy; y <= rly; y++)
  { char c[1];
    c[0] = image[x][y].c; 
    paint.setPen( image[x][y].f ); 
    paint.setBackgroundColor( image[x][y].b );
    paint.drawText(8*x,12+16*y, c,1);
  }
  paint.end();				// painting done
  setUpdatesEnabled(TRUE);
}

/* Structure to hold escape sequences. */
struct escseq {
  int code;
  char *linux_con;
  char *vt100_app;
  char *ansi;
};

/* Escape sequences for different terminal types. */
static struct escseq vt_keys[] = {
  { Key_F1,      "[[A", "[17~", "OP" },
  { Key_F2,      "[[B", "[18~", "OQ" },
  { Key_F3,      "[[C", "[19~", "OR" },
  { Key_F4,      "[[D", "[20~", "OS" },
  { Key_F5,      "[[E", "[21~", "OT" },
  { Key_F6,      "[17~", "[23~", "OU" },
  { Key_F7,      "[18~", "[24~", "OV" },
  { Key_F8,      "[19~", "[25~", "OW" },
  { Key_F9,      "[20~", "[26~", "OX" },
  { Key_F10,     "[21~", "[28~", "OY" },
  { Key_F11,     "[22~", "[29~", "OZ" },
  { Key_F12,     "[23~", "[31~", "OA" },
  { Key_Up,      "[A",   "OA",   "[A" },
  { Key_Left,    "[D",   "OD",   "[D" },
  { Key_Right,   "[C",   "OC",   "[C" },
  { Key_Down,    "[B",   "OB",   "[B" },
  { Key_Home,    "[1~",  "OP",   "[H" },
  { Key_End,     "[7~",  "OR",   "[Y" },
  { Key_Prior,   "[5~",  "OQ",   "[V" },
  { Key_Next,    "[6~",  "OS",   "[U" },
  { Key_Insert,  "[2~",  "[1~",  "[@"  },
  { Key_Delete,  "[3~", "[3~",  "\177" },
  { 0,            NULL,   NULL,   NULL }
};

//FIXME: an `eventFilter has been installed instead of a `keyPressEvent
//       due to a bug in `QT or the ignorance of the author to prevent
//       repaintevents being emitted to the screen whenever one leaves
//       or reenters the screen to/from another application.

bool AnsiTerminal::eventFilter( QObject *, QEvent *e )
{
  if (!active) return FALSE;                        // standard event processing
  if ( e->type() != Event_KeyPress )  return FALSE; // standard event processing

  // handle key press ////////////////////////////////

  QKeyEvent *ke = (QKeyEvent*)e;
  if (ke->ascii()>0)
  { unsigned char c[1]; 
    c[0] = ke->ascii(); mo->send_bytes((char*)c,1);
  }
  else
  { int key = ke->key();
    for(int i = 0; vt_keys[i].code != 0; i++)
    {
      if (vt_keys[i].code == key)
      { /* send esc seq */
        mo->send_byte(ESC);
        mo->send_string(vt_keys[i].linux_con);
        break;
      }
    }
  }
  return TRUE; // done event
}
