/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <stdio.h>

#include <qregexp.h>

#include "dmaport.h"
#include "interrupt.h"
#include "ioaddress.h"
#include "isapnpscanner.h"
#include "memaddress.h"
#include "resbaseconf.h"


ISAPnPScanner::ISAPnPScanner()
{
  _parser = new FileParser("/proc/isapnp");

  if (_parser->eof()) fprintf(stderr, "ALSA isapnp module not loaded\n");
}


ISAPnPScanner::~ISAPnPScanner()
{
  delete _parser;
}


/*
 * Public methods
 *****************/

Device  *ISAPnPScanner::firstDevice()
{
  if (_parser->reset()) return device();
  
  return 0L;
}


Device  *ISAPnPScanner::nextDevice()
{
  if (!_parser->eof()) return device();
  
  return 0L;
}


/*
 * Private methods
 ******************/
 
void  ISAPnPScanner::addActiveResource(const QString& line, ISAPnPDevice *isapnpdev)
{
  QString  pnpline = line.right(line.length()-6).simplifyWhiteSpace();
  QString  portstr, dmastr;
  int      pos;
  uint     iorange, port, irq, dma;
  ushort   bits;
    
  if (pnpline.left(4) == "port")
  {
    debug("    ADD ACTIVE RESOURCE: PORT");
    
    pnpline = pnpline.right(pnpline.length()-4).simplifyWhiteSpace();
    
    do
    {
      pos     = pnpline.find(',');	    
      portstr = (pos >= 0) ? pnpline.left(pos) : pnpline;
      
      sscanf(pnpline.data(), "%x", &port);
      
      iorange = ioRange(port);
      
      isapnpdev->addResource(new IOAddress(port, iorange));
      
      pnpline = (pos >= 0) ? pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace() : QString("");
    } while (pos >= 0);
  }
  else if (pnpline.left(3) == "IRQ")
  {
    debug("    ADD ACTIVE RESOURCE: IRQ");

    pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
    
    sscanf(pnpline.data(), "%u", &irq);
    
    isapnpdev->addResource(new Interrupt(irq));     
  }
  else if (pnpline.left(3) == "DMA")
  {
    debug("    ADD ACTIVE RESOURCE: DMA");

    pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
    bits    = 8;
    
    do
    {
      pos    = pnpline.find(',');	   
      dmastr = (pos >= 0) ? pnpline.left(pos) : pnpline;
      
      sscanf(pnpline.data(), "%u", &dma);
      
      isapnpdev->addResource(new DMAPort(dma, bits));
      
      bits   += 8;
      pnpline = (pos >= 0) ? pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace() : QString("");
    } while (pos >= 0);   
  } else debug("    ADD ACTIVE RESOURCE: UNKNOWN");
}


void  ISAPnPScanner::addBaseConfiguration(ISAPnPDevice *isapnpdev)
{
  QString             pnpline, irqstr, dmastr, numstr;
  QRegExp             commaslash("[,/]");
  ResourceBaseConfig  *baseconf = new ResourceBaseConfig();
  uint                fromport, toport, align, size, irq, dma, bits;
  int		      pos;
  QList<uint>         dmalist;
  
  debug("    ADD BASE CONFIGURATION"); 
  
  while ((pnpline = _parser->nextLine()).left(8) == "Priority");
  
  _parser->traceBack();

  for (pnpline = _parser->nextLine();
       (pnpline.left(4) == "Port") || (pnpline.left(3) == "IRQ") || (pnpline.left(3) == "DMA");
       pnpline = _parser->nextLine())
  {
    if (pnpline.left(4) == "Port")
    {
      pnpline = pnpline.right(pnpline.length()-4).simplifyWhiteSpace();
      
      sscanf(pnpline.data(), "%x-%x, align %x, size %x", &fromport, &toport, &align, &size);
      debug("      PORT 0x%04x-0x%04x ALIGN 0x%02x SIZE 0x%02x", fromport, toport, align, size);
      
      baseconf->addIOBaseConfig(new IOBaseConfig(fromport, toport, align, size));
    }
    else if (pnpline.left(3) == "IRQ")
    {
      pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
      irqstr  = "";				// only needed for debugging
      
      while ((pos = pnpline.find(commaslash)) >= 0)
      {      
        sscanf(pnpline.left(pos).data(), "%u", &irq);
	
        if (!irqstr.isEmpty()) irqstr += ", ";	// only needed for debugging
	irqstr += numstr.setNum(irq);	
	
	pnpline = pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace();
      
        baseconf->addInterrupt(new Interrupt(irq));
      }
      
      sscanf(pnpline.data(), "%u", &irq);
      baseconf->addInterrupt(new Interrupt(irq));
      
      if (!irqstr.isEmpty()) irqstr += ", ";	// only needed for debugging
      irqstr += numstr.setNum(irq);	
      debug("      IRQ %s", irqstr.data());
    }
    else if (pnpline.left(3) == "DMA")
    {
      pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
      dmastr  = "";				// only needed for debugging
      
      dmalist.setAutoDelete(true);
      dmalist.clear();
      
      while ((pos = pnpline.find(',')) >= 0)
      {      
        sscanf(pnpline.left(pos).data(), "%u", &dma);
	dmalist.append(new uint(dma));
	
        if (!dmastr.isEmpty()) dmastr += ", ";	// only needed for debugging
	dmastr += numstr.setNum(dma);	
	
	pnpline = pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace();
      }
            
      sscanf(pnpline.data(), "%u %u-bit", &dma, &bits);
      dmalist.append(new uint(dma));
      
      for (uint *dmaptr = dmalist.first(); dmaptr; dmaptr = dmalist.next())      
        baseconf->addDMAPort(new DMAPort(*dmaptr, bits));
      
      if (!dmastr.isEmpty()) dmastr += ", ";	// only needed for debugging
      dmastr += numstr.setNum(dma);	
      debug("      DMA %s\t%2u BITS", dmastr.data(), bits);
    }
  }
    
  isapnpdev->addBaseConfig(baseconf);  
  _parser->traceBack();
}


ISAPnPDevice  *ISAPnPScanner::device()
{
  QString             pnpline, logname;
  QString             devidstr;
  static QString      vendorid, devname;
  static uint         deviceid;
  char                mark = 39;
  uint                logid;
  int		      pos;
  
  while (!_parser->eof())
  {
    pnpline = _parser->nextLine();
    
    if (pnpline.left(6) == "Device")			// We have a physical device
    {
      if ((pos = pnpline.find(mark)) >= 0)
      {
        pnpline  = pnpline.right(pnpline.length()-pos-1); 	
	vendorid = pnpline.left(3); 
        devname  = pnpline.right(pnpline.length()-8);
	
	sscanf(pnpline.data()+3, "%x", &deviceid);
	
	devname.truncate(devname.find(mark)); 	
      }
      
      debug("PHYSICAL DEVICE <%s>", devname.data());      
    }
    else if (pnpline.left(14) == "Logical device")	// We have a logical device
    {
      if ((pos = pnpline.find(mark)) >= 0)
      {
        pnpline  = pnpline.right(pnpline.length()-pos-1);
	logname  = pnpline.right(pnpline.length()-8);	
	devidstr = pnpline.mid(3, 4);	
	
	sscanf(devidstr, "%x", &logid);
	
	logname.truncate(logname.find(mark)); 	
      }
    
      return logicalDevice(new ISAPnPDevice(vendorid, deviceid, devname, logid, logname));
    }
  }

  return 0L;
}


ISAPnPDevice  *ISAPnPScanner::logicalDevice(ISAPnPDevice *isapnpdev)
{
  QString  pnpline;

  debug("  LOGICAL DEVICE <%s>", isapnpdev->deviceName().data());

  for (pnpline = _parser->nextLine(); 
       (pnpline.left(10) == "Compatible") || (pnpline.left(9) == "Device is");
       pnpline = _parser->nextLine())
  {
    if (pnpline.left(16) == "Device is active") 
    {
      debug("  DEVICE IS ACTIVE");
      isapnpdev->setActive(true);
    }
  }

  _parser->traceBack();
  
  while ((pnpline = _parser->nextLine()).left(6) == "Active") addActiveResource(pnpline, isapnpdev);

  _parser->traceBack();
  
  while (_parser->nextLine().left(9) == "Resources")	// What does this tag mean?
  {    
    for (pnpline = _parser->nextLine();
         (pnpline.left(18) == "Priority preferred") || (pnpline.left(19) == "Alternate resources");
         pnpline = _parser->nextLine())
      addBaseConfiguration(isapnpdev);
    
    _parser->traceBack();
  }

  _parser->traceBack();
  
  return isapnpdev;
}


