/*
    This file is part of Konsole, an X terminal.
    Copyright 2005 by Maksim Orlovich <maksim@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301  USA.
*/

// Standard
#include <stdlib.h>
#include <iostream>
#include <iomanip>

// Qt
#include <QFile>
#include <QTextStream>
#include <QDebug>

using namespace std;

static quint32 charVal(QChar val)
{
    if (val == QLatin1Char(' ')) {
        return 0;
    } else {
        return 1;
    }
}

static quint32 readGlyphLine(QTextStream& input)
{
    QString line = input.readLine();
    while (line.length() < 5) {
        line += QLatin1Char(' ');
    }

    quint32 val =  charVal(line[0]) |
                   (charVal(line[1]) << 1)  |
                   (charVal(line[2]) << 2)  |
                   (charVal(line[3]) << 3)  |
                   (charVal(line[4]) << 4);
    return val;
}

static quint32 readGlyph(QTextStream& input)
{
    return readGlyphLine(input) |
           (readGlyphLine(input) << 5) |
           (readGlyphLine(input) << 10) |
           (readGlyphLine(input) << 15) |
           (readGlyphLine(input) << 20);
}

int main(int argc, char **argv)
{
    if (argc != 2) {
        qWarning("usage: fontembedder LineFont.src > LineFont.h");
        exit(1);
    }
    QFile inFile(QString::fromLocal8Bit(argv[1]));
    if (!inFile.open(QIODevice::ReadOnly)) {
        qWarning("Can not open %s", argv[1]);
        exit(1);
    }

    QTextStream input(&inFile);

    // Currently, for Konsole, the input glyphs file ranges from 0x2500
    // (9472) to x257f (9599) so this 128 will handle it.  However, if
    // more glyphs are added to the input file, this will be an issue.
    quint32 glyphStates[128];
    QMap<quint32, int> glyphMap;

    for (unsigned int & glyphState : glyphStates) {
        glyphState = 0; //nothing..
    }

    while (!input.atEnd()) {
        QString line = input.readLine();
        line = line.trimmed();
        if (line.isEmpty()) {
            continue; //Skip empty lines
        }
        if (line[0] == QLatin1Char('#')) {
            continue; //Skip comments
        }

        //Must be a glyph ID.
        int glyph = line.toInt(nullptr, 16);
        if ((glyph < 0x2500) || (glyph > 0x257f)) {
            qWarning("Invalid glyph number: %d aborting...", glyph);
            exit(1);
        } else {
            glyph = glyph - 0x2500;

            glyphStates[glyph] = readGlyph(input);
            // qWarning()<<glyph<<";"<<glyphStates[glyph];

            if (glyphMap.contains(glyphStates[glyph])) {
            // FIXME: get this qWarning working again
            //qWarning()<<"Code "<<glyph<<" and "<<glyphMap.value(glyphStates[glyph])<<"have the same glyph state"<<glyphStates[glyph];
            }
            glyphMap[glyphStates[glyph]] = glyph;
        }
    }

    //Output.
    cout << "// WARNING: Autogenerated by \"fontembedder " << argv[1] << "\".\n";
    cout << "// You probably do not want to hand-edit this!\n\n";
    cout << "static const quint32 LineChars[] = {\n";

    //Nicely formatted: 8 per line, 16 lines
    for (int line = 0; line < 128; line += 8) {
        cout << "\t";
        for (int col = line; col < line + 8; ++col) {
            cout << "0x" << hex << setw(8) << setfill('0') << glyphStates[col];
            if (col != 127) {
                cout << ", ";
            }
        }
        cout << "\n";
    }
    cout << "};\n";
    return 0;
}

