/***************************************************************************
 *   Copyright 2010 Artur Duque de Souza <asouza@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include <KDebug>
#include <KSycoca>
#include <KServiceTypeTrader>

#include <Plasma/DataContainer>

#include "shareengine.h"
#include "shareservice.h"


ShareEngine::ShareEngine(QObject *parent, const QVariantList &args)
    : Plasma::DataEngine(parent, args)
{
    Q_UNUSED(args);
}

void ShareEngine::init()
{
    connect(KSycoca::self(), SIGNAL(databaseChanged(QStringList)),
            this, SLOT(updatePlugins(QStringList)));
    updatePlugins(QStringList() << "services");
}

void ShareEngine::updatePlugins(const QStringList &changes)
{
    if (!changes.contains("services")) {
        return;
    }

    removeAllSources();

    QHash<QString, QStringList> mimetypes;
    const KService::List services = KServiceTypeTrader::self()->query("Plasma/ShareProvider");
    foreach (const KService::Ptr &service, services) {
        const QString pluginName =
            service->property("X-KDE-PluginInfo-Name", QVariant::String).toString();

        const QStringList pluginMimeTypes =
            service->property("X-KDE-PlasmaShareProvider-MimeType", QVariant::StringList).toStringList();

        const QString storageId = service->storageId();

        if (pluginName.isEmpty() || pluginMimeTypes.isEmpty()) {
            continue;
        }

        // create the list of providers
        Plasma::DataEngine::Data data;
        data.insert("Name", service->name());
        data.insert("Service Id", service->storageId());
        data.insert("Mimetypes", pluginMimeTypes);
        setData(pluginName, data);

        // create the list of providers by type
        foreach (const QString &pluginMimeType, pluginMimeTypes) {
            mimetypes[pluginMimeType].append(pluginName);
        }
    }


    QHashIterator<QString, QStringList> it(mimetypes);
    while (it.hasNext()) {
        it.next();
        setData("Mimetypes", it.key(), it.value());
    }
}

Plasma::Service *ShareEngine::serviceForSource(const QString &source)
{
    Plasma::DataContainer *data = containerForSource(source);

    if (!data) {
        return Plasma::DataEngine::serviceForSource(source);
    }

    if (source.compare("mimetype", Qt::CaseInsensitive) == 0) {
        return Plasma::DataEngine::serviceForSource(source);
    }

    const QString id = data->data().value("Service Id").toString();
    if (id.isEmpty()) {
        return Plasma::DataEngine::serviceForSource(source);
    }

    ShareService *service = new ShareService(this);
    service->setDestination(id);
    return service;
}

#include "shareengine.moc"

K_EXPORT_PLASMA_DATAENGINE(share, ShareEngine)
