/***************************************************************************
    Copyright 2009-2010 Frederik Gladhorn <gladhorn@kde.org>
    Copyright 2009-2010 Daniel Laidig <laidig@kde.org>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "writtenpracticewidget.h"
#include "ui_practice_widget_written.h"

#include "guifrontend.h"
#include "latexrenderer.h"

#include "prefs.h"

#include <kcolorscheme.h>

#include <QTimer>
#include <QDebug>

using namespace Practice;

WrittenPracticeWidget::WrittenPracticeWidget(GuiFrontend *frontend, QWidget *parent, bool isExampleSentenceMode)
    : AbstractModeWidget(frontend, parent), m_latexRenderer(0)
{
    m_ui = new Ui::WrittenPracticeWidget();
    m_ui->setupUi(this);
    m_ui->mixedSolutionLabel->setVisible(false);
    this->isExampleSentenceMode = isExampleSentenceMode;
    connect(m_ui->answerEdit, SIGNAL(returnPressed()), this, SLOT(continueClicked()));
    connect(frontend, SIGNAL(continueAction()), this, SIGNAL(stopAudio()));
    connect(frontend, SIGNAL(skipAction()), this, SIGNAL(stopAudio()));
}

void WrittenPracticeWidget::objectDestroyed(QObject *) {
    /**
      @page bug332596 bug 332596 Deleting a qstyled KLineEdit causes crash

     The bug was as follows.  When a WrittenPracticeWidget using a KLineEdit
     styled with a qStyleSheet that has focus is deleted this triggers a focus
     changing event, but the QStyleSheet returns a style that has already been
     deleted and causes a crash when the invalid style is applied.

     The obvious solution of responding to the destroyed() event doesn't work
     , because the destroyed() event happens after the crash.

     In order to fix this bug, I transistion focus away from the KLineEdit before
     the focus change preceding deletion.  The virtual function,
     WrittenPracticeWidget::objectDestroyed,  avoids having a
     practice mode switch statement as large as this comment at each delete event.

     Transitioning to helpLabel was chosen arbitrarily.

     @todo When frameworks/kde5 is implemented remove this code
     , if the bug in Qt (returning a pointer to a deleted stylesheet) is fixed.
     If this function is removed then the entire inheritance path can be removed.
     */

    m_ui->helpLabel->setFocus();
    delete m_ui;
    m_ui = 0;
}


void WrittenPracticeWidget::setQuestionFont(const QFont& font)
{
    m_ui->questionLabel->setFont(font);
}

void WrittenPracticeWidget::setSolutionFont(const QFont& font)
{
    m_ui->answerEdit->setFont(font);
    m_ui->solutionLabel->setFont(font);
    m_ui->mixedSolutionLabel->setFont(font);
}

void WrittenPracticeWidget::continueClicked()
{
    emit continueAction();
}

QVariant WrittenPracticeWidget::userInput()
{
    return QVariant(m_ui->answerEdit->text());
}

void WrittenPracticeWidget::setQuestion(const QVariant& question)
{
    m_ui->questionLabel->setMinimumSize(QSize(0, 0));
    if (LatexRenderer::isLatex(question.toString())) {
        if (!m_latexRenderer) {
            m_latexRenderer = new LatexRenderer(this);
            m_latexRenderer->setResultLabel(m_ui->questionLabel);
        }
        m_latexRenderer->renderLatex(question.toString());
    } else {
        m_ui->questionLabel->setText(question.toString());
    }
}

void WrittenPracticeWidget::showQuestion()
{
    m_ui->answerEdit->setEnabled(true);
    m_ui->answerEdit->clear();
    QTimer::singleShot(0, m_ui->answerEdit, SLOT(setFocus()));
    m_ui->answerEdit->setPalette(palette());
    m_ui->solutionLabel->setText(QString());
    m_ui->helpLabel->clear();

    foreach(QWidget * child, synonymWidgets) {
        m_ui->synonymList->removeWidget(child);
        delete child;
    }
    synonymWidgets.clear();

    if (isExampleSentenceMode == false) {
        m_ui->questionPronunciationLabel->setVisible(m_ui->questionPronunciationLabel->isEnabled());
        m_ui->questionSoundButton->setVisible(true); // TODO: Use Configuration's Sound Enable QCheckbox
    } else {
        m_ui->questionPronunciationLabel->setVisible(false);
        m_ui->questionSoundButton->setVisible(false);
    }
    m_ui->solutionPronunciationLabel->setVisible(false);
    m_ui->solutionSoundButton->setVisible(false);
}

void WrittenPracticeWidget::setSolution(const QVariant& solution)
{
    m_solution = solution.toString();
}

void WrittenPracticeWidget::showSolution()
{
    m_ui->solutionLabel->setText(m_solution);
    m_ui->answerEdit->setEnabled(false);
    if (m_feedbackState == AbstractFrontend::AnswerCorrect) {
        m_ui->answerEdit->setPalette(m_correctPalette);
    } else {
        m_ui->answerEdit->setPalette(m_wrongPalette);
    }
    m_ui->solutionLabel->setPalette(m_correctPalette);

    m_ui->solutionPronunciationLabel->setVisible(m_ui->solutionPronunciationLabel->isEnabled());
    m_ui->solutionSoundButton->setVisible(true); // TODO: Use Configuration's Sound Enable QCheckbox
}

void WrittenPracticeWidget::setSynonym(const QString &synonym)
{
    m_synonym = synonym;
}

void WrittenPracticeWidget::showSynonym()
{
    QLabel* synonym = new QLabel();
    QFont font;
    //TODO: use slightly smaller font for synonyms
    //font.setPointSize(9);
    synonym->setAlignment(Qt::AlignCenter);
    synonym->setFont(font);
    synonym->setText(i18n("Synonym: ") + m_synonym);
    synonymWidgets.append(synonym);
    m_ui->synonymList->addWidget(synonym);
    m_ui->answerEdit->clear();
}

void WrittenPracticeWidget::setHint(const QVariant& hint)
{
    m_ui->helpLabel->setText(hint.toString());
}

void WrittenPracticeWidget::setFeedback(const QVariant& feedback)
{
    m_ui->helpLabel->setText(feedback.toString());
}

void WrittenPracticeWidget::setFeedbackState(AbstractFrontend::ResultState feedbackState)
{
    m_feedbackState = feedbackState;
}

void WrittenPracticeWidget::setResultState(AbstractFrontend::ResultState resultState)
{
    m_resultState = resultState;
}

void WrittenPracticeWidget::setQuestionSound(const KUrl& soundUrl)
{
    m_ui->questionSoundButton->setSoundFile(soundUrl);
}

void WrittenPracticeWidget::setSolutionSound(const KUrl& soundUrl)
{
    m_ui->solutionSoundButton->setSoundFile(soundUrl);
}

void WrittenPracticeWidget::setSolutionPronunciation(const QString& pronunciationText)
{
    m_ui->solutionPronunciationLabel->setText('[' + pronunciationText + ']');
    m_ui->solutionPronunciationLabel->setEnabled(!pronunciationText.isNull());
}

void WrittenPracticeWidget::setQuestionPronunciation(const QString& pronunciationText)
{
    m_ui->questionPronunciationLabel->setText('[' + pronunciationText + ']');
    m_ui->questionPronunciationLabel->setEnabled(!pronunciationText.isNull());
}

#include "writtenpracticewidget.moc"
