/***************************************************************************
 *   Copyright © 2013 Aleix Pol Gonzalez <aleixpol@blue-systems.com>       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU General Public License as        *
 *   published by the Free Software Foundation; either version 2 of        *
 *   the License or (at your option) version 3 or any later version        *
 *   accepted by the membership of KDE e.V. (or its successor approved     *
 *   by the membership of KDE e.V.), which shall act as a proxy            *
 *   defined in Section 14 of version 3 of the license.                    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 ***************************************************************************/

#include "AppPackageKitResource.h"
#include <AppstreamQt/screenshot.h>
#include <AppstreamQt/image.h>
#include <KLocalizedString>
#include <KToolInvocation>
#include <QDebug>

AppPackageKitResource::AppPackageKitResource(const Appstream::Component& data, PackageKitBackend* parent)
    : PackageKitResource(data.packageNames().first(), QString(), parent)
    , m_appdata(data)
{
    Q_ASSERT(data.isValid());
}

QString AppPackageKitResource::name()
{
    return m_appdata.name();
}

QString AppPackageKitResource::longDescription()
{
    return m_appdata.description();
}

QString AppPackageKitResource::icon() const
{
    QString anIcon = m_appdata.icon();
    if (anIcon.isEmpty()) {
        QUrl iconUrl = m_appdata.iconUrl(QSize());
        if (iconUrl.isLocalFile())
            anIcon = iconUrl.toLocalFile();
    }
    return anIcon;
}

QString AppPackageKitResource::license()
{
    return m_appdata.projectLicense();
}

QStringList AppPackageKitResource::mimetypes() const
{
    return findProvides(Appstream::Provides::KindMimetype);
}

QStringList AppPackageKitResource::categories()
{
    return m_appdata.categories();
}

QString AppPackageKitResource::comment()
{
    return m_appdata.summary();
}

QUrl AppPackageKitResource::homepage()
{
    QList< QUrl > urls = m_appdata.urls(Appstream::Component::UrlKindHomepage);
    return urls.isEmpty() ? PackageKitResource::homepage() : urls.first();
}

bool AppPackageKitResource::isTechnical() const
{
    return false;
}

QStringList AppPackageKitResource::executables() const
{
    return findProvides(Appstream::Provides::KindBinary);
}

void AppPackageKitResource::invokeApplication() const
{
    QStringList exes = executables();
    if(!exes.isEmpty())
        KToolInvocation::startServiceByDesktopPath(exes.first());
}

QUrl screenshot(const Appstream::Component& comp, Appstream::Image::Kind kind)
{
    QUrl ret;
    for (const Appstream::Screenshot &s : comp.screenshots()) {
        for (const Appstream::Image &i : s.images()) {
            if (i.kind() == kind) {
                ret = i.url();
            }
        }
        if (s.isDefault() && !ret.isEmpty())
            break;
    }
    return ret;
}

QUrl AppPackageKitResource::screenshotUrl()
{
    QUrl url = screenshot(m_appdata, Appstream::Image::Plain);
    return url.isEmpty() ? PackageKitResource::screenshotUrl() : url;

}

QUrl AppPackageKitResource::thumbnailUrl()
{
    QUrl url = screenshot(m_appdata, Appstream::Image::Thumbnail);
    return url.isEmpty() ? PackageKitResource::screenshotUrl() : url;
}

bool AppPackageKitResource::canExecute() const
{
    return !executables().isEmpty();
}

QStringList AppPackageKitResource::findProvides(Appstream::Provides::Kind kind) const
{
    QStringList ret;
    for(Appstream::Provides p : m_appdata.provides())
        if (p.kind() == kind)
            ret += p.value();
    return ret;
}
