#include <qimage.h>
#include <qfileinfo.h>

#include <kurl.h>

#include "customevents.h"
#include "thumbnailmt.h"

#include "thumbnailcreator.h"

#define XMD_H
extern "C" {
#include <jpeglib.h>
#include <stdio.h>
}

ThumbnailCreator::ThumbnailCreator(QObject *_parent)
    : QThread()
{

    parent = _parent;

    mThreadExit = false;
    mForceRegeneration = false;
    mPaused = false;
    mRequestQueue.flush();
}

ThumbnailCreator::~ThumbnailCreator()
{
    mThreadExit = true;
    wait();
}

void ThumbnailCreator::requestThumbnail(const QString& fileName,
                                        const ThumbnailSize&
                                        thumbSize)
{
    mRequestQueue.enqueue(new ThumbnailRequest(fileName,
                                              thumbSize));
}

void ThumbnailCreator::stop()
{
    mRequestQueue.flush();
}

void ThumbnailCreator::pause(bool shouldPause)
{
    mutex.lock();
    mPaused = shouldPause;
    mutex.unlock();
}

void ThumbnailCreator::run()
{
    while (true) {

        if (mThreadExit) return;

        while (mRequestQueue.isEmpty()) {
            if (mThreadExit) return;
            msleep(200);
            showBusyStatus(false);
        }


        showBusyStatus(true);

        ThumbnailRequest* thumbRequest = mRequestQueue.dequeue();
        if (!thumbRequest) continue;


        QString fileName = thumbRequest->getFileName();
        ThumbnailSize thumbSize = thumbRequest->getThumbnailSize();


        KURL fileURL(fileName);

        QString thumbDirName;

        switch(thumbSize.getSize()) {
        case ThumbnailSize::Small:
            thumbDirName = getThumbSmallCacheDir();
            break;
        case ThumbnailSize::Medium:
            thumbDirName = getThumbMediumCacheDir();
            break;
        case ThumbnailSize::Large:
            thumbDirName = getThumbLargeCacheDir();
            break;
        default:
            continue;
        }


        QString thumbFileName(thumbDirName);
        thumbFileName += fileURL.fileName();


        QFileInfo thumbFileInfo(thumbFileName);
        QFileInfo fileInfo(fileName);

        ThumbnailMT thumbnail;

        if (thumbFileInfo.exists() &&
            thumbFileInfo.lastModified() >= fileInfo.lastModified() &&
            !isForceRegeneration()) {
            delete thumbRequest;
            thumbRequest = 0;
	    thumbnail.load(thumbFileName);
        }
        else {

            if (isPaused()) {
                mRequestQueue.enqueue(thumbRequest);
                thumbRequest = 0;
                msleep(200);
                continue;
            }
            else {
                delete thumbRequest;
                thumbRequest=0;
            }


            QImage image;

            if (isJPEG(fileName)) {
                if (!loadJPEG(image, thumbSize, fileName)) {
                    image.load(fileName);
                    image = image.smoothScale(thumbSize.getSizeAsNumber(),
                                              thumbSize.getSizeAsNumber(),
                                              QImage::ScaleMin);
                }
            }
            else {
                image.load(fileName);
                image = image.smoothScale(thumbSize.getSizeAsNumber(),
                                          thumbSize.getSizeAsNumber(),
                                          QImage::ScaleMin);
            }


            //image.save(thumbFileName,"JPEG");
	    thumbnail.load(image);
            thumbnail.dropShadow();
            thumbnail.save(thumbFileName);
        }

        postEvent(parent, new EventThumbnail(fileURL.directory(),
                                             fileURL.fileName(),
                                             thumbnail));

    }
}

void ThumbnailCreator::showBusyStatus(bool _busy)
{
    postEvent(parent, new EventBusy(_busy));
}

bool ThumbnailCreator::isJPEG(const QString& fileName)
{
    QString format=QImageIO::imageFormat(fileName);
    return format=="JPEG";
}

// Fudged Fast JPEG decoding code from GWENVIEW

bool ThumbnailCreator::loadJPEG(QImage& image,
                                const ThumbnailSize& thumbSize,
                                const QString& fileName)
{

    FILE* inputFile=fopen(fileName.data(), "rb");
    if(!inputFile) return false;

    struct jpeg_decompress_struct cinfo;
    struct jpeg_error_mgr jerr;
    cinfo.err = jpeg_std_error(&jerr);
    jpeg_create_decompress(&cinfo);
    jpeg_stdio_src(&cinfo, inputFile);
    jpeg_read_header(&cinfo, TRUE);

    int size = thumbSize.getSizeAsNumber();
    int imgSize = QMAX(cinfo.image_width, cinfo.image_height);

    int scale=1;
    while(size*scale*2<=imgSize) {
        scale*=2;
    }
    if(scale>8) scale=8;

    cinfo.scale_num=1;
    cinfo.scale_denom=scale;

    // Create QImage
    jpeg_start_decompress(&cinfo);

    switch(cinfo.output_components) {
    case 3:
    case 4:
        image.create( cinfo.output_width, cinfo.output_height, 32 );
        break;
    case 1: // B&W image
        image.create( cinfo.output_width, cinfo.output_height, 8, 256 );
        for (int i=0; i<256; i++)
            image.setColor(i, qRgb(i,i,i));
        break;
    default:
        return false;
    }

    uchar** lines = image.jumpTable();
    while (cinfo.output_scanline < cinfo.output_height)
        jpeg_read_scanlines(&cinfo, lines + cinfo.output_scanline,
                            cinfo.output_height);
    jpeg_finish_decompress(&cinfo);

    // Expand 24->32 bpp
    if ( cinfo.output_components == 3 ) {
        for (uint j=0; j<cinfo.output_height; j++) {
            uchar *in = image.scanLine(j) + cinfo.output_width*3;
            QRgb *out = (QRgb*)( image.scanLine(j) );

            for (uint i=cinfo.output_width; i--; ) {
                in-=3;
                out[i] = qRgb(in[0], in[1], in[2]);
            }
        }
    }

    int newMax = QMAX(cinfo.output_width, cinfo.output_height);
    int newx = size*cinfo.output_width / newMax;
    int newy = size*cinfo.output_height / newMax;

    image=image.smoothScale( newx, newy);

    jpeg_destroy_decompress(&cinfo);
    fclose(inputFile);

    return true;

}

void ThumbnailCreator::setThumbnailCacheDirs(const QString& smallCacheDir,
                                             const QString& mediumCacheDir,
                                             const QString& largeCacheDir)
{
    mutex.lock();
    mThumbSmallCacheDir  = smallCacheDir.latin1();
    mThumbMediumCacheDir = mediumCacheDir.latin1();
    mThumbLargeCacheDir  = largeCacheDir.latin1();
    mutex.unlock();
}

QString ThumbnailCreator::getThumbSmallCacheDir()
{
    QString cacheDir;
    mutex.lock();
    cacheDir = mThumbSmallCacheDir;
    mutex.unlock();
    return cacheDir;
}

QString ThumbnailCreator::getThumbMediumCacheDir()
{
    QString cacheDir;
    mutex.lock();
    cacheDir = mThumbMediumCacheDir;
    mutex.unlock();
    return cacheDir;
}

QString ThumbnailCreator::getThumbLargeCacheDir()
{
    QString cacheDir;
    mutex.lock();
    cacheDir = mThumbLargeCacheDir;
    mutex.unlock();
    return cacheDir;
}

void ThumbnailCreator::forceRegeneration(bool force)
{
    mutex.lock();
    mForceRegeneration = force;
    mutex.unlock();
}

bool ThumbnailCreator::isForceRegeneration()
{
    bool result;
    mutex.lock();
    result = mForceRegeneration;
    mutex.unlock();

    return result;

}

bool ThumbnailCreator::isPaused()
{
    bool result;
    mutex.lock();
    result = mPaused;
    mutex.unlock();

    return result;
}
