#include <qheader.h>
#include <qdragobject.h>
#include <qpoint.h>
#include <qpopupmenu.h>
#include <qtimer.h>
#include <qcursor.h>
#include <qstring.h>
#include <qstringlist.h>

#include <kfiletreebranch.h>
#include <kfiletreeviewitem.h>
#include <klocale.h>
#include <klineeditdlg.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kpropsdlg.h>
#include <kurl.h>
#include <kurldrag.h>

#include "cameradragobject.h"
#include "albumfoldertree.h"

AlbumFolderTree::AlbumFolderTree(QWidget* parent)
    : KFileTreeView(parent)
{
    addColumn(i18n("Album Folders"));
    setFullWidth(true);

    setDragEnabled(true);
    setDropVisualizer(false);
    setDropHighlighter(true);
    setAcceptDrops(true);

    connect(this, SIGNAL(selectionChanged(QListViewItem*)),
            this, SLOT(slot_selectionChanged(QListViewItem*)));
    connect(this, SIGNAL(contextMenu(KListView*,
                                     QListViewItem*,
                                     const QPoint&)),
            this,SLOT(slot_contextMenu(KListView*,
                                       QListViewItem*,
                                       const QPoint&)));

    mRootBranch = NULL;
    mDropTarget = NULL;
    mAutoOpenTimer = new QTimer(this);

    connect(mAutoOpenTimer, SIGNAL(timeout()),
            this, SLOT(slot_autoOpenDropTarget()));

    AUTO_OPEN_DELAY = 1000;

    kioJob = false;

}

AlbumFolderTree::~AlbumFolderTree()
{
    delete mAutoOpenTimer;
}

void AlbumFolderTree::setRootPath(const KURL& url)
{
    if (url.cmp(mRootURL,true))
        return;
    else if (mRootBranch)
        removeBranch(mRootBranch);

    mRootURL = url;
    mPrevURL = url;

    mRootBranch = addBranch(mRootURL,i18n("Albums"));
    setDirOnlyMode(mRootBranch, true);
    mRootBranch->setOpen();
    setSelected(mRootBranch->root(),true);

    connect(mRootBranch, SIGNAL(refreshItems(const KFileItemList&)),
            this, SLOT(slot_refreshItems(const KFileItemList&)));

}

void AlbumFolderTree::changeRootPath(const KURL& url)
{
    mRootURL = url;
    mPrevURL = url;

    if (mRootBranch)
        removeBranch(mRootBranch);

    mRootBranch = addBranch(mRootURL,i18n("Albums"));
    setDirOnlyMode(mRootBranch, true);
    mRootBranch->setOpen();
    setSelected(mRootBranch->root(),true);
}


void AlbumFolderTree::makeDir()
{

    if (!mRootBranch) return;

    if (!currentKFileTreeViewItem())
        setSelected(mRootBranch->root(),true);

    currentKFileTreeViewItem()->setOpen(true);

    bool ok;
    QString msg(i18n("Creating New Album in "));
    msg += "\"" + trimCurrentAlbumName() + "\"";
    msg += "\n";
    msg += i18n("Enter new Album Name :");
    QString newDir =
        KLineEditDlg::getText(msg,"",&ok,this);
    if (!ok) return;

    kioJob = true;

    KURL newURL(currentURL());
    newURL.addPath(newDir);
    KIO::Job* job = KIO::mkdir(newURL);

    connect(job, SIGNAL(result(KIO::Job*)),
            this, SLOT(slot_onMkDir(KIO::Job*)));
}

void AlbumFolderTree::slot_onMkDir(KIO::Job* job)
{
    if (job->error()) {
        job->showErrorDialog(this);
        return;
    }

    if (!currentKFileTreeViewItem()) return;
    currentKFileTreeViewItem()->setOpen(true);
}


void AlbumFolderTree::rmDir()
{
    if (! currentKFileTreeViewItem () ) return;

    if (currentKFileTreeViewItem() == mRootBranch->root())
        return;

    QString msg(i18n("About to Delete this Album: "));
    msg += "\"" + trimCurrentAlbumName() + "\"";
    msg += i18n(" and all its contents : \n");
    msg += i18n("Are you sure ?");


    if (KMessageBox::warningYesNo(this, msg) ==
        KMessageBox::Yes) {

        QListViewItem* parentItem =
            currentKFileTreeViewItem()->parent();

        KIO::Job* job = KIO::del(currentKFileTreeViewItem()->url());
        connect(job, SIGNAL(result(KIO::Job*)),
                this, SLOT(slot_onRmDir(KIO::Job*)));

        setSelected(parentItem,true);

    }


}

void AlbumFolderTree::slot_onRmDir(KIO::Job* job)
{
    if (job->error()) {
        job->showErrorDialog(this);
        return;
    }

}

void AlbumFolderTree::renameDir()
{
    if (!mRootBranch) return;
    if (!currentKFileTreeViewItem ()) return;

    if (currentKFileTreeViewItem() == mRootBranch->root())
        return;

    bool ok;
    QString msg(i18n("Rename "));
    msg += "\"" + trimCurrentAlbumName() + "\"";
    msg += i18n(" to :");
    QString newName = KLineEditDlg::getText(msg,
                                            currentKFileTreeViewItem()->url().fileName(),
                                            &ok,this);
    if (!ok) return;

    KURL newURL = currentURL().upURL();
    newURL.addPath(newName);

    KURL oldURL = currentURL();


    KIO::Job* job = KIO::rename(oldURL, newURL, false);
    connect(job, SIGNAL(result(KIO::Job*)),
            this, SLOT(slot_onRenameDir(KIO::Job*)));

}

void AlbumFolderTree::slot_onRenameDir(KIO::Job* job)
{
    if (job->error()) {
        job->showErrorDialog(this);
        return;
    }
}


void AlbumFolderTree::showPropertiedDir()
{
    (void) new KPropertiesDialog(currentURL());
}

void AlbumFolderTree::slot_refreshItems(const KFileItemList&
                                        itemList)
{
    KFileItemListIterator iter(itemList);
    KFileItem* fileItem;

    while((fileItem=iter.current()) != 0) {
        ++iter;
        KFileTreeViewItem* treeItem =
            static_cast<KFileTreeViewItem *>(fileItem->extraData(mRootBranch));
        if (treeItem) {
            treeItem->setText(0,fileItem->url().fileName());
        }
    }
}

void AlbumFolderTree::slot_selectionChanged(QListViewItem* it)
{
    if (!it) return;

    KFileTreeViewItem* item
        = static_cast<KFileTreeViewItem*>(it);

    if (!item) return;

    if (currentURL() != mPrevURL) {
        mPrevURL = currentURL();
        emit signal_currentURLChanged(currentURL());
    }

}

void AlbumFolderTree::slot_contextMenu(KListView* listView,
                                       QListViewItem* item,
                                       const QPoint& pos)
{
    if (!listView || !item) return;

    KFileTreeViewItem* treeItem
        = static_cast<KFileTreeViewItem*>(item);
    if (!treeItem) return;

    QPopupMenu* popMenu = new QPopupMenu(this);
    popMenu->insertItem(SmallIcon("folder_new"),
                        i18n("New Album"), 10);
    popMenu->insertSeparator();
    if (treeItem != mRootBranch->root()) {
        popMenu->insertItem(i18n("Rename Album"), 11);
        popMenu->insertItem(SmallIcon("editdelete"),
                            i18n("Delete Album"),
                            12);
        popMenu->insertSeparator();
    }
    popMenu->insertItem(i18n("Properties"),
                           13);

    int id = popMenu->exec(pos);

    switch(id) {
    case 10: {
        makeDir();
        break;
    }
    case 11: {
        renameDir();
        break;
    }
    case 12: {
        rmDir();
        break;
    }
    case 13: {
        (void)new KPropertiesDialog(currentURL());
        break;
    }
    default:
        break;
    }

    delete popMenu;

}

QString AlbumFolderTree::trimCurrentAlbumName()
{
    QString albumName = currentURL().path();
    albumName.remove(0,mRootURL.path().length());
    if (albumName.isEmpty())
        albumName = i18n("Main Album");
    return albumName;
}

void AlbumFolderTree::slotNewTreeViewItems(KFileTreeBranch* branch,
                                         const KFileTreeViewItemList& itemList)
{
    KFileTreeView::slotNewTreeViewItems(branch, itemList);

    if (kioJob) {
        kioJob = false;

        KFileTreeViewItem *item = itemList.getFirst();
        if (item) {
            setSelected(item,true);
        }
    }
}

// -- Most of the DND code has been fudged from GWENVIEW ----

void AlbumFolderTree::contentsDragMoveEvent(QDragMoveEvent* event) {

    if (!QUriDrag::canDecode(event) &&
        !CameraDragObject::canDecode(event)) {
        event->ignore();
        return;
    }

    // Get a pointer to the new drop item
    QPoint point(0,event->pos().y());
    KFileTreeViewItem* newDropTarget =
        static_cast<KFileTreeViewItem*>(
            itemAt(contentsToViewport(point)) );
    if (!newDropTarget) {
        event->ignore();
        mAutoOpenTimer->stop();
        return;
    }

    event->accept();
    if (newDropTarget==mDropTarget) return;

    if (mDropTarget)
        stopAnimation(mDropTarget);

    // Restart auto open timer if we are over a new item
    mAutoOpenTimer->stop();
    mDropTarget=newDropTarget;
    startAnimation(mDropTarget);

    mAutoOpenTimer->start(AUTO_OPEN_DELAY,true);

}


void AlbumFolderTree::contentsDragLeaveEvent(QDragLeaveEvent*) {

    mAutoOpenTimer->stop();
    if (mDropTarget) {
        stopAnimation(mDropTarget);
        mDropTarget=NULL;
    }
}


void AlbumFolderTree::contentsDropEvent(QDropEvent* event)
{
    mAutoOpenTimer->stop();
    if (!mDropTarget) return;

    stopAnimation(mDropTarget);

    KURL destURL = mDropTarget->url();

    if (QUriDrag::canDecode(event)) {

        KURL::List srcURLs;
        KURLDrag::decode(event,srcURLs);

        QPopupMenu popMenu(this);
        popMenu.insertItem( i18n("&Copy"), 10 );
        popMenu.insertItem( i18n("&Move"), 11 );

        popMenu.setMouseTracking(true);
        int id = popMenu.exec(QCursor::pos());
        switch(id) {
        case 10: {
            KIO::copy(srcURLs,destURL,true);
            break;
        }
        case 11: {
            KIO::move(srcURLs,destURL,true);
            break;
        }
        default:
            break;
        }

    }
    else if (CameraDragObject::canDecode(event)) {



        QStringList nameList;
        QStringList folderList;

        CameraDragObject::decode(event, nameList, folderList);

        QPopupMenu popMenu(this);
        popMenu.insertItem( i18n("&Download"), 10 );
        popMenu.setMouseTracking(true);

        int id = popMenu.exec(QCursor::pos());
        switch(id) {
        case 10: {
            emit signal_requestCameraDownload(nameList,
                                              folderList,
                                              destURL.path());
            break;
        }
        default:
            break;
        }

    }

    mDropTarget=0;

}

void AlbumFolderTree::slot_autoOpenDropTarget()
{
    if (mDropTarget) {
        if (!mDropTarget->isOpen()) {
            mDropTarget->setOpen(true);
        }
    }
}

