/***************************************************************************
                          frameitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qptrlist.h>
#include <klocale.h>
#include <ksimpleconfig.h>
#include "frameitem.h"
#include "kgraph.h"
#include "utils.h"
#ifndef KPL_CLASSES_ONLY
#include <qlistview.h>
#include "framedlg.h"
#endif

FrameItem::FrameItem() : logx(false), logy(false), xBottom(true), xTop(false),
 yLeft(true), yRight(false), collapsed(false), ndigx(-1), ndigy(-1), mticx(5),
 mticy(2), gridmode(KplGraph::AxesWithLabels), iex(0), iey(0), colFrame(0),
 colGrid(0), xf(21.0), yf(29.7), w(15.0), h(10.0), xmi(0.0), xma(15.0),
 ymi(0.0), yma(10.0), xtic(5.0), ytic(2.0), xOff(0.0), yOff(0.0)
{
  x0 = 4.0;
  y0 = 3.0;
}

FrameItem::FrameItem(KplNamespace::AutoStruct* aut) :
 logx(aut->autoLogX), logy(aut->autoLogY), xBottom(true), xTop(false),
 yLeft(true), yRight(false), collapsed(false), ndigx(-1), ndigy(-1), mticx(5),
 mticy(2), gridmode(aut->autoGrid), iex(0), iey(0), colFrame(aut->colFrame),
 colGrid(aut->colGrid), xf(aut->xf), yf(aut->yf), w(aut->xlAuto),
 h(aut->ylAuto), xmi(0.0), xma(aut->xlAuto), ymi(0.0), yma(aut->ylAuto),
 xtic(5.0), ytic(2.0), xOff(0.0), yOff(0.0), sx(aut->sxAuto),
 sy(aut->syAuto), sh(aut->shAuto)
{
  x0 = aut->x0Auto;
  y0 = aut->y0Auto;
  relsiz = aut->relSize;
}

FrameItem::FrameItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut) :
 KplItem(plo)
{
  logx = plo->readBoolEntry("logx");
  logy = plo->readBoolEntry("logy");
  xTop = plo->readBoolEntry("xTop");
  xBottom = plo->readBoolEntry("xBottom", !xTop);
  yRight = plo->readBoolEntry("yRight");
  yLeft = plo->readBoolEntry("yLeft", !yRight);
  collapsed = plo->readBoolEntry("collapsed");
  ndigx = plo->readNumEntry("ndigx", -1);
  ndigy = plo->readNumEntry("ndigy", -1);
  mticx = plo->readNumEntry("mticx", 5);
  mticy = plo->readNumEntry("mticy", 2);
  gridmode = plo->readNumEntry("gridmode", aut->autoGrid);
  iex = plo->readNumEntry("iex");
  iey = plo->readNumEntry("iey");
  QString s = plo->readEntry("colframe", "");
  colFrame = s.isEmpty() ? aut->colFrame : KplGraph::rgbQt(s);
  s = plo->readEntry("colgrid", "");
  colGrid = s.isEmpty() ? aut->colGrid : KplGraph::rgbQt(s);
  xf = plo->readDoubleNumEntry("xf", aut->xf);
  yf = plo->readDoubleNumEntry("yf", aut->yf);
  x0 = plo->readDoubleNumEntry("xw0", aut->x0Auto);
  w = plo->readDoubleNumEntry("xw", aut->xlAuto);
  y0 = plo->readDoubleNumEntry("yw0", aut->y0Auto);
  h = plo->readDoubleNumEntry("yw", aut->ylAuto);
  xmi = plo->readDoubleNumEntry("xmi");
  xma = plo->readDoubleNumEntry("xma", xmi + 15.0);
  ymi = plo->readDoubleNumEntry("ymi");
  yma = plo->readDoubleNumEntry("yma", ymi + 10.0);
  xtic = plo->readDoubleNumEntry("xtic", 5.0);
  ytic = plo->readDoubleNumEntry("ytic", 2.0);
  xOff = plo->readDoubleNumEntry("xOff", 0.0);
  yOff = plo->readDoubleNumEntry("yOff", 0.0);
  sx = plo->readEntry("sx", "");
  sy = plo->readEntry("sy", "");
  sh = plo->readEntry("sh", "");
}

FrameItem::FrameItem(bool act, bool lgx, bool lgy, int ndx, int ndy,
                     int mtx, int mty, int gm, int ex, int ey,
                     const QString& col_f, const QString& col_g,
                     double x_0, double wx, double y_0, double hy,
                     double xmin, double xmax, double ymin,
                     double ymax, double xt, double yt, double relSize,
                     const QString& sX, const QString& sY, const QString& sH,
                     KplNamespace::AutoStruct* aut, double x_off,
                     double y_off, bool xB, bool xT, bool yL, bool yR,
                     bool coll) :
 logx(lgx), logy(lgy), xBottom(xB), xTop(xT), yLeft(yL), yRight(yR),
 collapsed(coll),  ndigx(ndx), ndigy(ndy), mticx(mtx), mticy(mty),
 gridmode(gm), iex(ex), iey(ey), xf(aut->xf), yf(aut->yf), w(wx), h(hy),
 xmi(xmin), xma(xmax), ymi(ymin), yma(ymax), xtic(xt), ytic(yt), xOff(x_off),
 yOff(y_off), sx(sX), sy(sY), sh(sH)
{
  active = act;
  x0 = x_0;
  y0 = y_0;
  relsiz = relSize;
  colFrame = KplGraph::rgbQt(col_f);
  colGrid = KplGraph::rgbQt(col_g);
}

FrameItem::~FrameItem()
{
}

KplItem::ItemTypes FrameItem::iType() const
{
  return Frame;
}

void FrameItem::draw(KplGraph* g)
{
  if (!g->type())
    bRect.setRect(0, 0, 0, 0);
  if (w && h && active) {
    g->format(xf, yf);
    g->setRelSize(relsiz);
    g->Window(x0, w, y0, h);
    if (!g->type()) {
      KGraph* gr = (KGraph*) g;
      bRect.setCoords(gr->minx, gr->maxy, gr->maxx, gr->miny);
    }
    g->scale(xmi, xma, ymi, yma, logx, logy);
    g->setDig(ndigx, ndigy);
    g->setColFrame(colFrame);
    g->setColGrid(colGrid);
    g->raster(xtic, ytic, mticx, mticy, gridmode, xOff, yOff, xBottom, xTop,
              yLeft, yRight);
    if (xBottom)
      g->letX(sx, iex, false);
    if (xTop)
      g->letX(sx, iex, true);
    if (yLeft)
      g->letY(sy, iey, false);
    if (yRight)
      g->letY(sy, iey, true);
    g->letH(sh, xTop);
  }
}

#ifndef KPL_CLASSES_ONLY
void FrameItem::writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                         KplDoc* m) const
{
  plo->writeEntry("Type", "FRAMEITEM");
  KplItem::writePlo(plo, url, abs, m);
  if (logx)
    plo->writeEntry("logx", logx);
  if (logy)
    plo->writeEntry("logy", logy);
  if (!xBottom)
    plo->writeEntry("xBottom", xBottom);
  if (xTop)
    plo->writeEntry("xTop", xTop);
  if (!yLeft)
    plo->writeEntry("yLeft", yLeft);
  if (yRight)
    plo->writeEntry("yRight", yRight);
  if (collapsed)
    plo->writeEntry("collapsed", collapsed);
  plo->writeEntry("ndigx", ndigx);
  plo->writeEntry("ndigy", ndigy);
  plo->writeEntry("mticx", mticx);
  plo->writeEntry("mticy", mticy);
  plo->writeEntry("gridmode", gridmode);
  plo->writeEntry("iex", iex);
  plo->writeEntry("iey", iey);
  plo->writeEntry("colframe", QString::number(KplGraph::rgbQt1(colFrame), 16));
  plo->writeEntry("colgrid", QString::number(KplGraph::rgbQt1(colGrid), 16));
  plo->writeEntry("xf", xf);
  plo->writeEntry("yf", yf);
  plo->writeEntry("xw0", x0);
  plo->writeEntry("xw", w);
  plo->writeEntry("yw0", y0);
  plo->writeEntry("yw", h);
  plo->writeEntry("xmi", xmi);
  plo->writeEntry("xma", xma);
  plo->writeEntry("ymi", ymi);
  plo->writeEntry("yma", yma);
  plo->writeEntry("xtic", xtic);
  plo->writeEntry("ytic", ytic);
  if (xOff)
    plo->writeEntry("xOff", xOff);
  if (yOff)
    plo->writeEntry("yOff", yOff);
  plo->writeEntry("sx", sx);
  plo->writeEntry("sy", sy);
  plo->writeEntry("sh", sh);
}

void FrameItem::setText(QListViewItem* it, bool*, bool*) const
{
  it->setText(1, i18n("Frame"));
  QString s1, s2;
  it->setText(2, s1.sprintf("x = %g cm, y = %g cm, ", x0, y0) + i18n("size")
              + s2.sprintf(" %g cm x %g cm", w, h));
}

int FrameItem::editItem(QWidget* parent, KplDoc* m, int i)
{
  FrameDlg dlg(parent, m, i);
  return dlg.exec();
}
#endif

KplItem* FrameItem::copy() const
{
  return new FrameItem(*this);
}

bool FrameItem::calcAutoScale(bool autoNorm, int item,
                              QPtrList<KplItem>* items, double* fx, double* fy)
{ int iexmin, ieymin;
  if (autoNorm) {
    double fxt, fyt;
    int iext, ieyt;
    iexmin = ieymin = 999;
    for (KplItem* itm = items->at(item + 1); itm; itm = items->next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      if (itm->isActive()) {
        iext = iexmin;
        ieyt = ieymin;
        itm->expoItem(&iext, &ieyt, &fxt, &fyt);
        if (iext < iexmin) {
          iexmin = iext;
          *fx = fxt;
        }
        if (ieyt < ieymin) {
          ieymin = ieyt;
          *fy = fyt;
        }
      }
    }
  } else {
    iexmin = ieymin = 0;
    *fx = *fy = 1.0;
  }
  bool success = (iexmin != 999) && (ieymin != 999);
  if (success) {
    iex = iexmin;
    iey = ieymin;
    double xmin = 1.0e300;
    double ymin = xmin;
    double xmax = -xmin;
    double ymax = xmax;
    for (KplItem *itm = items->at(item + 1); itm; itm = items->next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      if (itm->isActive()) {
        double xmint = xmin;
        double ymint = ymin;
        double xmaxt = xmax;
        double ymaxt = ymax;
        itm->minMax(&xmint, &xmaxt, &ymint, &ymaxt);
        xmin = QMIN(xmin, xmint);
        xmax = QMAX(xmax, xmaxt);
        ymin = QMIN(ymin, ymint);
        ymax = QMAX(ymax, ymaxt);
      }
    }
    autoScale(xmin, xmax, *fx, ymin, ymax, *fy);
  }
  return success;
}

void FrameItem::normalize(int item, QPtrList<KplItem>* items, double fx,
                          double fy)
{
  for (KplItem* itm = items->at(item + 1); itm; itm = items->next()) {
    if (itm->iType() == KplItem::Frame)
      break;
    itm->normalize(fx, fy);
  }
}

void FrameItem::autoScale(double xmin, double xmax, double fx, double ymin,
                          double ymax, double fy)
{
  Utils::autoSc(&xmi, &xma, &xtic, &mticx, &ndigx, xmin, xmax, fx, 0.2, logx);
  Utils::autoSc(&ymi, &yma, &ytic, &mticy, &ndigy, ymin, ymax, fy,
                0.2 * w / h, logy);
}

bool FrameItem::autoScale(bool autoNorm, QPtrList<KplItem>* items)
{
  int item = items->find(this);
  if (item == -1)
    return false;
  double fx, fy;
  if (autoNorm)
    logx = logy = false;
  bool success = calcAutoScale(autoNorm, item, items, &fx, &fy);
  if (success)
    normalize(item, items, fx, fy);
  return success;
}
