/* -*- c++ -*-
*
* setupwizard.cpp
*
* Copyright (C) 2004 Petter Stokke <ummo@hellokitty.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.
*
*/

#include "setupwizard.h"

#include "fileselector.h"
#include "hostmanager.h"

#include <kdebug.h>
#include <klocale.h>
#include <kactivelabel.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kpixmapeffect.h>
#include <kpixmap.h>
#include <kconfig.h>
#include <klineedit.h>
#include <qpainter.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>

class KMLWizardGraphic : public QWidget
{
public:
    KMLWizardGraphic( QWidget* parent = 0, const char* name = 0 )
            : QWidget( parent, name )
    {
        setBackgroundMode( Qt::NoBackground );
        m_logo = KGlobal::iconLoader() ->loadIcon( "kmldonkey", KIcon::Desktop, 128 );
        setFixedWidth( m_logo.width() );
    }

protected:
    virtual void paintEvent( QPaintEvent* )
    {
        QPainter wp( this );
        QRect r( rect() );
        if ( r.size() != m_buffer.size() ) {
            m_buffer.resize( r.size() );
            KPixmapEffect::gradient( m_buffer, QColor( "#fff" ), QColor( "#888" ), KPixmapEffect::VerticalGradient );
            QPainter p( &m_buffer );
            p.drawPixmap( 0, 0, m_logo );
        }
        wp.drawPixmap( 0, 0, m_buffer );
    };

private:
    KPixmap m_buffer, m_logo;
};

class KMLWizardPage : public QWidget
{
public:
    KMLWizardPage( QWidget* parent = 0, const char* name = 0 )
            : QWidget( parent, name )
    {
        l = new QHBoxLayout( this );
        l->setSpacing( 8 );
        l->addWidget( new KMLWizardGraphic( this ) );
    }

    void setContent( QWidget* content )
    {
        QVBoxLayout * l2 = new QVBoxLayout( this );
        l2->addStretch();
        l2->addWidget( content );
        l2->addStretch();
        l->addLayout( l2 );
    }

private:
    QHBoxLayout* l;
};

SetupWizard::SetupWizard( QWidget *parent, const char *name )
        : KWizard( parent, name )
{
    setCaption( i18n( "KMLDonkey Setup Wizard" ) );

    introPage = new KMLWizardPage( this );
    introPage->setContent( new KActiveLabel( i18n( "<h1>KMLDonkey Setup Wizard</h1>"
                           "<p>Making KMLDonkey work can be a daunting task for the uninitiated. "
                           "This wizard will guide you through the basic steps needed to "
                           "get you up and running.</p>"
                           "<p>If you understand how to set up and administer an MLDonkey core "
                           "and want to configure KMLDonkey yourself, you can safely skip this "
                           "wizard; otherwise, press the \"Next\" button below to get started.</p>"
                           "<p><em>Read the text in the wizard carefully; everything in it is "
                           "important.</em></p>"
                           "<p><b>WARNING:</b> Running this wizard will <em>erase</em> any "
                           "previous core connections you may have configured. "
                           "It will not affect any previous MLDonkey cores you have installed.</p>" ), introPage ) );
    addPage( introPage, i18n( "KMLDonkey Setup Wizard" ) );

    mldonkeyDownloadPage = new KMLWizardPage( this );
    mldonkeyDownloadPage->setContent( new KActiveLabel( i18n( "<h1>Download MLDonkey</h1>"
                                      "<p>The first thing you need to do is obtain an MLDonkey core. "
                                      "This is the actual file sharing tool for which KMLDonkey acts as a frontend.</p>"
                                      "<p>Visit one of the web sites listed below (just click on the links), "
                                      "and download a binary package suitable for your system; "
                                      "alternatively, if your system's package manager provides an MLDonkey "
                                      "package, you can install that instead, and note where the \"mlnet\" binary "
                                      "is installed for later (usually in <em>/usr/bin</em>).</p>"
                                      "<ul>%1</ul>"
                                      "<p>Extract the package somewhere convenient, and press \"Next\". "
                                      "Do not worry about configuring or starting what you downloaded; "
                                      "that will be done later.</p>" ).
                                      arg( "<li><a href=\"http://download.berlios.de/pub/mldonkey/spiralvoice/\">Spiralvoice</a></li>"
                                           "<li><a href=\"http://mldonkey.dyndns.info/\">Knocker</a></li>"
                                           "<li><a href=\"http://kmldonkey.org/node/view/68\">KMLDonkey.org</a></li>" ),
                                      mldonkeyDownloadPage ) );
    addPage( mldonkeyDownloadPage, i18n( "Download MLDonkey" ) );

    workingDirPage = new KMLWizardPage( this );
    QVBox* workingDirContainer = new QVBox( workingDirPage );
    workingDirContainer->setSpacing( 8 );
    ( void ) new KActiveLabel( i18n( "<h1>Set Up a Work Folder</h1>"
                                     "<p>You need a work folder for your MLDonkey core. "
                                     "This is where it stores its configuration data, temporary files "
                                     "and finished downloads. You should place it on a disk partition "
                                     "with a lot of free space.</p>"
                                     "<p>NOTE: You can change the locations where it stores temporary files "
                                     "and finished downloads later, if you like.</p>"
                                     "<p>Select your work folder here:</p>" ), workingDirContainer );
    workingDirSelector = new FileSelector( i18n( "Select Your Work Folder" ), true, workingDirContainer );
    workingDirPage->setContent( workingDirContainer );
    addPage( workingDirPage, i18n( "Set Up a Work Folder" ) );
    connect( workingDirSelector, SIGNAL( fileSelected( const QString& ) ), this, SLOT( slotWorkingDirSelected() ) );

    coreSelectionPage = new KMLWizardPage( this );
    QVBox* coreSelectionContainer = new QVBox( coreSelectionPage );
    coreSelectionContainer->setSpacing( 8 );
    ( void ) new KActiveLabel( i18n( "<h1>Locate Your Core</h1>"
                                     "<p>Now, select the full path to your MLDonkey core, which you "
                                     "(or your distribution) downloaded earlier, below. "
                                     "If the archive you downloaded only contained one file, then this "
                                     "is the core, otherwise look for the file called \"mlnet\".</p>" ), coreSelectionContainer );
    coreSelector = new FileSelector( i18n( "Locate Your MLDonkey Core" ), false, coreSelectionContainer );
    coreSelectionPage->setContent( coreSelectionContainer );
    addPage( coreSelectionPage, i18n( "Locate Your Core" ) );
    connect( coreSelector, SIGNAL( fileSelected( const QString& ) ), this, SLOT( slotCoreSelected() ) );

    startupModePage = new KMLWizardPage( this );
    QVBox* startupModeContainer = new QVBox( startupModePage );
    startupModeContainer->setSpacing( 8 );
    ( void ) new KActiveLabel( i18n( "<h1>Launcher Settings</h1>"
                                     "<p>Now, you need to decide when you want the core to be running. "
                                     "The easiest way is to start the core whenever you launch KMLDonkey, "
                                     "and shut it down when you close the application. You can also elect to "
                                     "have it running continuously when you are logged into your KDE session, "
                                     "regardless of whether KMLDonkey is running.</p>" ), startupModeContainer );
    startupModeSelector = new QButtonGroup( 1, Horizontal, startupModeContainer );
    ( new QRadioButton( i18n( "Start the core when KMLDonkey starts" ), startupModeSelector ) ) ->setChecked( true );
    ( void ) new QRadioButton( i18n( "Start the core when you log into KDE" ), startupModeSelector );
    startupModePage->setContent( startupModeContainer );
    addPage( startupModePage, i18n( "Launcher Settings" ) );

    namePage = new KMLWizardPage( this );
    QVBox* nameContainer = new QVBox( namePage );
    nameContainer->setSpacing( 8 );
    ( void ) new KActiveLabel( i18n( "<h1>Name Your Core</h1>"
            "<p>The last thing you need to do is assign a name to your core. "
            "This is for your own use only; the default \"MLDonkey\" is "
            "quite adequate, but you can choose anything you like.</p>" ), nameContainer );
    nameEntry = new KLineEdit(nameContainer);
    nameEntry->setText(i18n("default name for wizard-created core connection", "MLDonkey"));
    nameEntry->selectAll();
    namePage->setContent( nameContainer );
    addPage( namePage, i18n( "Name Your Core" ) );
    connect(nameEntry, SIGNAL(textChanged(const QString& )), this, SLOT(slotNameChanged(const QString&)));

    endPage = new KMLWizardPage( this );
    endPage->setContent( new KActiveLabel( i18n( "<h1>Configuration Complete</h1>"
                                           "<p>Enough information has now been obtained to set up a working "
                                           "MLDonkey configuration. "
                                           "Press \"Finish\" below to get things started.</p>"
                                           "<p>Remember, your finished downloads go into the <em>incoming</em> "
                                           "folder under your work folder.</p>"
                                           "<p>Enjoy KMLDonkey!</p>" ), endPage ) );
    addPage( endPage, i18n( "Configuration Complete" ) );

    setNextEnabled( coreSelectionPage, false );
    setNextEnabled( workingDirPage, false );
    setFinishEnabled( endPage, true );
}

SetupWizard::~SetupWizard()
{}

void SetupWizard::slotCoreSelected()
{
    setNextEnabled( coreSelectionPage, coreSelector->isExecutable() );
}

void SetupWizard::slotWorkingDirSelected()
{
    setNextEnabled( workingDirPage, workingDirSelector->isDir() );
}

void SetupWizard::slotNameChanged(const QString& t)
{
    setNextEnabled(namePage, !t.isEmpty());
}

void SetupWizard::accept()
{
    KConfig * config = new KConfig( "mldonkeyrc", false, false );

    QStringList list = config->groupList();
    QStringList::iterator git;
    for ( git = list.begin(); git != list.end(); ++git )
        config->deleteGroup( *git );

    config->setGroup( nameEntry->text() );
    config->writeEntry( "DonkeyHost", "localhost" );
    config->writeEntry( "DonkeyGuiPort", 4001 );
    config->writeEntry( "DonkeyHTTPPort", 4080 );
    config->writeEntry( "DonkeyUsername", "admin" );
    config->writeEntry( "DonkeyPassword", "" );
    config->writeEntry( "HostMode", HostInterface::Managed );
    config->writePathEntry( "BinaryPath", coreSelector->fileName() );
    config->writePathEntry( "RootPath", workingDirSelector->fileName() );
    config->writeEntry( "StartupMode", (startupModeSelector->selectedId() == 0) ? HostInterface::AtKMLDonkeyStart : HostInterface::AtKDEStart );
    config->writeEntry( "Default", true );
    if ( config->hasKey( "LocalHost" ) )
        config->deleteEntry( "LocalHost" );

    config->sync();
    delete config;

    KWizard::accept();
}


#include "setupwizard.moc"
