/* -*- c++ -*-
 *
 * consolepage.cpp
 *
 * Copyright (C) 2003 Petter Stokke <ummo@hellokitty.com>
 * Copyright (C) 2006 Christian Muehlhaeuser <chris@chris.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <qstylesheet.h>
#include <qregexp.h>

#include <klocale.h>
#include <ktextbrowser.h>
#include <klineedit.h>
#include <kaction.h>
#include <kdebug.h>

#include <donkeyprotocol.h>

#include "version.h"
#include "kmldonkey.h"
#include "disclaimer.h"
#include "consolepage.h"
#include "consolepage.moc"

ConsolePage::ConsolePage(QWidget* parent)
    : QVBox(parent, "consolePage")
    , KMLDonkeyPage()
{
    alreadyClear = true;

    consoleView = new KTextBrowser(this, "consoleView");
    consoleView->setFocusPolicy(QTextBrowser::ClickFocus);
    consoleView->setTextFormat(QTextBrowser::RichText);
    consoleView->setLinkUnderline(true);

    consoleView->append(i18n("<p><b>KMLDonkey %1</b>, Copyright 2003-2006 Petter Stokke, Sebastian Sauer and Christian Muehlhaeuser</p>").arg(KMLDONKEY_VERSION));
    consoleView->append(i18n("<p>This program is free software; you can redistribute it and/or modify "
			     "it under the terms of the GNU General Public License as published by "
			     "the Free Software Foundation; either version 2 of the License, or "
			     "(at your option) any later version.</p>"
			     "<p>This program is distributed in the hope that it will be useful, "
			     "but WITHOUT ANY WARRANTY; without even the implied warranty of "
			     "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
			     "GNU General Public License for more details.</p>"
			     "<p>You should have received a copy of the GNU General Public License "
			     "along with this program; if not, write to the Free Software "
			     "Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301  USA</p>"));
    consoleView->append("<hr>");
    consoleView->append(DisclaimerDialog::disclaimerText());
    consoleView->append("<hr>");

    consoleEntry = new KLineEdit(this, "consoleEntry");
    connect(consoleEntry, SIGNAL(returnPressed(const QString&)),
	    consoleEntry->completionObject(), SLOT(addItem(const QString&)));
    connect(consoleEntry, SIGNAL(returnPressed(const QString&)),
            this, SLOT(sendConsoleMessage(const QString&)));

    connect(KMLDonkey::App->donkey, SIGNAL(consoleMessage(const QString&)), this, SLOT(consoleMessage(const QString&)));
}

void ConsolePage::clear()
{
    if (!alreadyClear) {
	consoleView->append("<hr>");
	alreadyClear = true;
    }
}

void ConsolePage::applyPreferences(KMLDonkeyPreferences*)
{
    consoleView->setFont(KMLDonkey::App->consoleFont);
}

void ConsolePage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("Activate Console Page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_console");
}

void ConsolePage::consoleMessage(const QString& msg)
{
    alreadyClear = false;

    QString txt = QStyleSheet::escape(msg);

    if (txt.startsWith("\n+--"))
        txt = "<b>" + txt + "</b>";
    else if (txt.startsWith("CORRUPTION DETECTED"))
        txt = "<p><b>" + txt + "</b></p>";
    else {
        QRegExp rxurl("[^\">](((https|http|ftp|news)://|(?!://)www\\.)([\\w\\.\\-\\_/~?=&%;:@+$,!*'\\(\\)\\|\\^\\[\\]\\`\\{\\}]+))");
        rxurl.setCaseSensitive(false);
        int pos = 0;
        QString url, replaceurl;
        while ((pos = rxurl.search(txt, pos)) >= 0) {
            url = ( (rxurl.cap(2).startsWith("www")) ? "http://" : QString::null ) + rxurl.cap(2) + rxurl.cap(4);
            replaceurl = "<a href=\"" + url + "\">" + rxurl.cap(1) + "</a>";
            txt.replace(pos + 1, rxurl.cap(1).length(), replaceurl);
            pos += replaceurl.length();
        }
    }

    txt.replace(QRegExp("\n(?!$)"), "<br>");

    if(txtBefore != txt) {
        consoleView->append(txt);
        txtBefore = txt;
    }
}

void ConsolePage::sendConsoleMessage(const QString& txt)
{
    alreadyClear = false;

    QString foo = "<p><b>" + txt + "</b></p>";
    consoleView->append(foo);
    consoleEntry->clear();
    foo = txt;
    KMLDonkey::App->donkey->sendConsoleMessage(foo);
}

void ConsolePage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

