/*
 * This file is part of Krita
 *
 * Copyright (c) 2006 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "kis_red_eye_removal_tool.h"

#include <kpushbutton.h>

#include <kis_button_release_event.h>
#include <kis_canvas_painter.h>
#include <kis_canvas_subject.h>
#include <kis_image.h>
#include <kis_iterators_pixel.h>
#include <kis_paint_device.h>
#include <kis_selection.h>
#include <kis_transaction.h>
#include <kis_undo_adapter.h>

#include "wdgredeyeremoval.h"

QRect KisRedEyeRemovalTool::RedEyeRegion::area() const
{
    if(m_uptodate)
        return m_area;

    QValueList< QPoint>::const_iterator it = m_points.begin();
    m_area = QRect( (*it).x(), (*it).y(), 0,0);
    ++it;
    for (; it != m_points.end(); ++it )
    {
        if((*it).x() > m_area.right())
        {
            m_area.setRight((*it).x());
        }
        if((*it).x() < m_area.left())
        {
            m_area.setLeft((*it).x());
        }
        if((*it).y() > m_area.bottom())
        {
            m_area.setBottom((*it).y());
        }
        if((*it).y() < m_area.top())
        {
            m_area.setTop((*it).y());
        }
    }
    m_uptodate = true;
    return m_area;
}

KisRedEyeRemovalTool::RegionStatus* KisRedEyeRemovalTool::RedEyeRegion::mask() const
{
    if(m_mask && m_uptodate)
        return m_mask;
    if(m_mask)
        delete[] m_mask;
    // Initialize which pixel is a member
    QRect r = area();
    int xstart = r.topLeft().x();
    int ystart = r.topLeft().y();
    m_mask = new RegionStatus[r.width() * r.height()];
    RegionStatus* statusIt1 = m_mask; // For some reasons memset is sucking, so we have to initialized by hand
    for(int k = 0; k < r.height() * r.width(); k++, statusIt1++)
    {
        *statusIt1 = NONMEMBER;
    }
    for(QValueList< QPoint>::const_iterator it = m_points.begin(); it != m_points.end(); ++it )
    {
        m_mask[ ((*it).x()-xstart) + ((*it).y()-ystart) * r.width()] = MEMBER;
    }
    return m_mask;
}

bool KisRedEyeRemovalTool::RedEyeRegion::fillRec(uint x, uint y, QRect& r) const
{
    RegionStatus* status = m_mask + x + y * r.width();
    switch(*status)
    {
        case BORDERCONNECTED:
            return true;
        case MEMBER:
        case CANDIDAT:
            return false;
        case NONMEMBER:
        {
            *status = CANDIDAT;
            bool a = fillRec(x-1, y, r); // We _do_ want the four call to be made
            bool b = fillRec(x, y-1, r);
            bool c = fillRec(x+1, y, r);
            bool d = fillRec(x, y+1, r);
            return ( a || b || c || d );
        }
    }
    return false; // never called but it shut up gcc
}

KisRedEyeRemovalTool::KisRedEyeRemovalTool() : KisToolNonPaint(i18n("Red-Eye Removal Tool"))
{
    setName("tool_red_eye_removal");

}

KisRedEyeRemovalTool::~KisRedEyeRemovalTool()
{
    
}

void KisRedEyeRemovalTool::setup(KActionCollection *collection)
{
    m_action = static_cast<KRadioAction *>(collection -> action(name()));

    if (m_action == 0) {
        m_action = new KRadioAction(i18n("&Red eye removal"),
                                    "tool_red_eye_removal",
                                    KShortcut(),
                                    this,
                                    SLOT(activate()),
                                    collection,
                                    name());
        Q_CHECK_PTR(m_action);
        m_action -> setExclusiveGroup("tools");
        m_action -> setToolTip(i18n("Remove red eyes"));
        m_ownAction = true;
    }
}
void KisRedEyeRemovalTool::update (KisCanvasSubject *subject)
{
    KisToolNonPaint::update (subject);
    if (m_subject)
        m_currentImage = m_subject->currentImg ();
}

void KisRedEyeRemovalTool::fixAllRegions()
{
    paint();
    for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); ++it )
    {
        correctRegion( *it );
    }
    m_regions.clear();
}
void KisRedEyeRemovalTool::unmarkAllRegions()
{
    paint();
    m_regions.clear();
}


// This function partly rely on an algorithm describes in AUTOMATED RED-EYE DETECTION AND CORRECTION IN DIGITAL PHOTOGRAPHS by Lei Zhang, Yanfeng Sun, Mingjing Li, Hongjiang Zhang
void KisRedEyeRemovalTool::activate()
{
    KisToolNonPaint::activate();
    
    if (!m_subject || !m_currentImage)
        return;
    KisPaintDeviceSP device = m_currentImage->activeDevice ();;
    QRect rect;
    if(device->hasSelection())
    {
        rect = device->selection()->selectedExactRect();
    } else {
        rect = device->exactBounds();
    }
    KisColorSpace* cs = device->colorSpace();
    QColor c;
    kdDebug() << "Step 1 : find and mark red pixels" << endl;
    Classification* classifications = new Classification[ rect.width() * rect.height() ];
    Classification* itClass = classifications;
    for(int y = rect.y(); y < rect.height(); y++)
    {
        KisHLineIteratorPixel itPixel = device->createHLineIterator(rect.x(), y, rect.width(), false);
        for(int x = rect.x(); x < rect.width(); x++)
        {
            *itClass = UNDEFINEDPIXEL;
            cs->toQColor( itPixel.oldRawData(), &c);
            double rgb = 1.0 / ((double)c.red() + (double)c.green() + (double)c.blue());
            if( c.red() > 50 && c.red() * rgb > 0.4 && c.green() * rgb < 0.31 && c.blue() * rgb < 0.36)
            {
                *itClass = REDPIXEL;
            } else if( c.green() * rgb > 0.4 || c.blue() * rgb > 0.45)
            {
                *itClass = NONSKINPIXEL;
            }
            ++itClass;
            ++itPixel;
        }
    }

#define BRIGHTNESS(bouh)   (bouh.red() * 0.212656 + bouh.green() * 0.715158 + bouh.blue() * 0.072186)
    kdDebug() << "Step 2 : find and mark highlight pixels" << endl;
    for(int y = rect.y() + 2; y < rect.height() - 3; y++)
    {
        for(int x = rect.x() + 2; x < rect.width() - 3; x++)
        {
            if( *(classifications + x + y * rect.width()) == REDPIXEL)
            {
                itClass = classifications + (x-2) + (y -2) * rect.width();
                uint numRed = 0;
                uint sumNeigh = 0;
                uint sumShadow = 0;
                KisHLineIteratorPixel itPixel = device->createHLineIterator(x - 3, y - 3, 6, false);
                for(uint i = 0; i < 6; i++)
                {
                    for(uint j = 0; j < 6; j++)
                    {
                        if(*itClass == REDPIXEL || *itClass == HIGHLIGHTPIXEL)
                        {
                            numRed++;
                        }
                        if( j > 1 && j < 4 && i > 1 && i < 4)
                        {
                            cs->toQColor( itPixel.oldRawData(), &c);
                            sumNeigh += (uint)BRIGHTNESS(c);
                        } else if( ( (j == 0 || j == 5 ) && ( i != 0 && i != 5 ) ) || ( (i == 0 || i == 5 ) && ( j != 0 && j != 5 ) ) )
                        {
                            cs->toQColor( itPixel.oldRawData(), &c);
                            sumShadow += (uint)BRIGHTNESS(c);
                        }
                        ++itClass;
                        ++itPixel;
                    }
                    itPixel.nextRow();
                }
                if(numRed > 10 && (4*sumNeigh + sumShadow) > (16*140) )
                {
//                     kdDebug() << "Highlight at " << x << " " << y << endl;
                    *(classifications + x + y * rect.width()) = HIGHLIGHTPIXEL;
                }
            }
            ++itClass;
        }
    }
#undef BRIGHTNESS
    // TODO: merge the growing, with all the steps of elimination of invalid regions
    kdDebug() << "Step 3 : growing regions" << endl;
    m_regions.clear();
    itClass = classifications;
    for(int y = rect.y(); y < rect.height(); y++)
    {
        for(int x = rect.x(); x < rect.width(); x++)
        {
            if(*itClass == HIGHLIGHTPIXEL)
            {
                RedEyeRegion region;
                *itClass = REGIONMEMBER;
                region.addPoint(QPoint(x,y));
                regionGrowing( region, classifications, x, y, HIGHLIGHTPIXEL, rect );
                if(region.size() > 1) // don't add unary regions, they are just noise
                    m_regions.push_back(region);
            }
            ++itClass;
        }
    }
    kdDebug() << "Found " << m_regions.size() << " regions" << endl;
#if 1
    kdDebug() << "Step 4 : remove invalid regions" << endl;
    for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); )
    {
        // Compute the averages values
        double avePctR = 0.;
        double avePctG = 0.;
        double aveR = 0.;
        for (QValueList< QPoint>::iterator itPoints = (*it).begin(); itPoints != (*it).end(); ++itPoints )
        {
            cs->toQColor( device->pixel( (*itPoints).x(), (*itPoints).y()), &c);
            double sum = 1/(double)(c.red() + c.green() + c.blue());
            avePctR += c.red() * sum;
            avePctG += c.green() * sum;
            aveR += c.red();
        }
        avePctR /= (*it).size();
        avePctG /= (*it).size();
        aveR /= (*it).size();
        (*it).avePctR = avePctR;
        (*it).avePctG = avePctG;
        (*it).aveR = aveR;
        // Look in the neighbourghood
        QRect area = (*it).area();
        area.addCoords(-12,-12, 12, 12);
        area.intersect( rect );
        KisRectIteratorPixel itPixel = device->createRectIterator(area.x(), area.y(), area.width(), area.height(), false);
        uint count = 0;
        while(!itPixel.isDone())
        {
            cs->toQColor( itPixel.oldRawData(), &c);
            double sum = 1/(double)(c.red() + c.green() + c.blue());
            if( QABS(c.red() * sum - avePctR) < 0.03 && QABS(c.green() * sum - avePctG) < 0.03 &&  QABS(c.red() - aveR) < 20 )
            {
                ++count;
            }
            ++itPixel;
        }
        if(count > 8 + (*it).size())
        { // Too much similar pixels, remove the region
            kdDebug() << "eliminated count = " << count << " for area = "<< area << " of size =" << (*it).size() << endl;
            it = m_regions.erase(it);
        } else {
            kdDebug() << "keeped count = " << count << " for area = "<< area << " of size =" << (*it).size() << endl;
            ++it;
        }
    }
    kdDebug() << "After first elimination, there are " << m_regions.size() << " regions" << endl;
#endif

#if 0
    kdDebug() << "Step 5 : remove misshaped regions" << endl;
    for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); )
    {
        QRect area = (*it).area();
        if(area.width() > 1.1 * area.height())
        {
            kdDebug() << "eliminated area = "<< area << " of shape =" << area << endl;
            it = m_regions.erase(it);
        } else {
            ++it;
        }
    }
#endif

#if 1
    kdDebug() << "Step 6 : remove regions which are surrounded by non-skin pixels" << endl;
    for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); )
    {
        QRect area = (*it).area();
        uint size = QMAX(area.width(), area.height())/4;
        area.addCoords(-size,-size, size, size);
        area.intersect( rect );
        itClass = classifications + area.x() + area.y() * rect.width();
        uint count = 0;
        for(int y = area.y(); y < area.height(); y++)
        {
            for(int x = area.x(); x < area.width(); x++, itClass++)
            {
                if(*itClass == NONSKINPIXEL)
                {
                    count ++;
                }
            }
            itClass += rect.width() - area.width();
        }
        if(count > size)
        {
            kdDebug() << "eliminated area = "<< area << " with " << count << " non-skins pixels around for a size of " << size << endl;
            it = m_regions.erase(it);
        } else {
            ++it;
        }
    }
#endif

#if 1
    kdDebug() << "Step 7 : last but not least, expands regions with the surrounding red pixels" << endl;
    for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); ++it )
    {
    // Compute the averages values
        for (QValueList< QPoint>::iterator itPoints = (*it).begin(); itPoints != (*it).end(); ++itPoints )
        {
            uint x = (*itPoints).x();
            uint y = (*itPoints).y();
            regionGrowingSimilar( *it, classifications, x, y, REDPIXEL, device, rect );
        }
    }
#endif

#if 1
    kdDebug() << "Step 8 : remove misshaped regions (again)" << endl;
    for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); )
    {
        QRect area = (*it).area();
        if(area.width() > 1.1 * area.height() || area.height() > 2.0 * area.width() )
        {
            kdDebug() << "eliminated area = "<< area << " of shape =" << area << endl;
            it = m_regions.erase(it);
        } else {
            ++it;
        }
    }
#endif

    delete[] classifications;
    paint();
}

void KisRedEyeRemovalTool::regionGrowing( RedEyeRegion& region, Classification* classifications, int x, int y, Classification classifier, QRect rect )
{ // WARNING: this function was originaly designed to be recursive, but but, it can leads to crash, that's explain the flat design and the use of the list
#if 1
    QValueList<QPoint> list;
    list.push_back( QPoint(x,y ));
    while(! list.empty())
    {
        QPoint p = list.front();
        list.pop_front();
        for(int u = -1; u <= 1; u++)
        {
            int xu = p.x() + u;
            for(int v = -1; v <= 1; v++)
            {
                int yv = p.y() + v;
                if(rect.contains(xu, yv))
                {
                    Classification* classif = classifications + xu + yv * rect.width();
                    if( *classif == classifier )
                    {
                        *classif = REGIONMEMBER;
                        region.addPoint(QPoint(xu,yv));
                        list.push_back( QPoint(xu,yv ));
                    }
                }
            }
        }
    }
#endif
#if 0
    for(int u = -1; u <= 1; u++)
    {
        int xu = x+u;
        for(int v = -1; v <= 1; v++)
        {
            int yv = y+v;
            if(rect.contains(xu, yv))
            {
                Classification* classif = classifications + xu + yv * rect.width();
                if( *classif == classifier )
                {
                    *classif = REGIONMEMBER;
                    regionGrowing( region, classifications, xu, yv, classifier, rect );
                    region.addPoint(QPoint(xu,yv));
                }
            }
        }
    }
#endif
}

void KisRedEyeRemovalTool::regionGrowingSimilar( RedEyeRegion& region, Classification* classifications, int x, int y, Classification classifier, KisPaintDeviceSP device, QRect rect )
{
    for(int u = -1; u <= 1; u++)
    {
        int xu = x+u;
        for(int v = -1; v <= 1; v++)
        {
            int yv = y+v;
            Classification* classif = classifications + xu + yv * rect.width();
            if(rect.contains(xu, yv))
            {
//                 const Q_UINT8* pixel = device->pixel(xu, yv);
//                 double sum = 1/(double)(pixel[0] + pixel[1] + pixel[2]);
//                 if((*classif == classifier)) kdDebug( ) << sum << " "<< (region.avePctR - pixel[2] * sum) << " "  << (region.avePctG - pixel[1] * sum ) << " " <<  (pixel[2] - region.aveR) << " " << (int)pixel[0] << " " << (int)pixel[1] << " " << (int)pixel[2] << " " << endl;
                if( *classif == classifier /*&& QABS(pixel[2] * sum - region.avePctR) < 0.03 && QABS(pixel[1] * sum - region.avePctG) < 0.03 &&  QABS(pixel[2] - region.aveR) < 20*/ )
                {
                    *classif = REGIONMEMBER;
                    regionGrowingSimilar( region, classifications, xu, yv, classifier, device, rect );
                    region.addPoint(QPoint(xu,yv));
                }
            }
        }
    }
}

void KisRedEyeRemovalTool::paint()
{
    if (m_subject) {
        KisCanvasController *controller = m_subject->canvasController();
        KisCanvas *canvas = controller->kiscanvas();
        KisCanvasPainter gc(canvas);
        QRect rc;

        paint(gc, rc);
    }
}


void KisRedEyeRemovalTool::paint(KisCanvasPainter& gc)
{
    paint(gc, QRect());
}

void KisRedEyeRemovalTool::paint(KisCanvasPainter& gc, const QRect& /*rc*/)
{
    if(m_regions.size() != 0)
    {
        RasterOp op = gc.rasterOp();
        QPen old = gc.pen();
        for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); ++it )
        {
            KisCanvasController *controller = m_subject->canvasController();
            QRect area = (*it).area();
            QPen pen(QColor((uint)(255*(random()/(double)RAND_MAX)),(uint)(255*(random()/(double)RAND_MAX)),(uint)(255*(random()/(double)RAND_MAX))),1,Qt::SolidLine);
            pen.setWidth(1);
//             QPen pen(Qt::SolidLine);
//             pen.setWidth(1);
            gc.setRasterOp(Qt::NotROP);
            gc.setPen(pen);
            
            QPoint start = controller->windowToView(area.topLeft());
            QPoint end = controller->windowToView(area.bottomRight());
            gc.drawRect( QRect(start,end) );

/*            for (QValueList< QPoint>::iterator itPoints = (*it).begin(); itPoints != (*it).end(); ++itPoints )
            {
                gc.drawPoint( controller->windowToView(*itPoints) );
            }*/
            RegionStatus* statusIt1 = (*it).mask();
            for(int j = 0; j < area.height(); j++)
            {
                for(int i = 0; i < area.width(); i++, statusIt1++)
                {
                    if(*statusIt1 == MEMBER)
                    {
                        gc.drawPoint( controller->windowToView(QPoint( i, j ) + area.topLeft() ) );
//                         gc.drawRect( QRect(controller->windowToView(QPoint( i, j ) + area.topLeft() ), controller->windowToView(QPoint( i+1, j+1 ) + area.topLeft() ) - QPoint(1,1) ) );
                    }
                }
            }
            
            // Restore old values

        }
        gc.setRasterOp(op);
        gc.setPen(old);
    }
}

void KisRedEyeRemovalTool::buttonRelease(KisButtonReleaseEvent *e)
{
    if (m_subject && m_subject->currentImg() && e->button() == LeftButton) {
        // look for the red eye that has been clicked on, and correct it
        for (QValueList< RedEyeRegion >::iterator it = m_regions.begin(); it != m_regions.end(); ++it )
        {
            QRect area = (*it).area();
            if( area.contains( e->pos().floorQPoint() ) )
            {
                paint();
                correctRegion( *it );
                paint();
                m_regions.remove( it );
                return;
            }
        }
        // didn't click on a red-eye region, if the clicked pixel is red, then fix the region connected to this pixel
        KisPaintDeviceSP device = m_currentImage->activeDevice ();
        QRect rect;
        if(device->hasSelection())
        {
            rect = device->selection()->selectedExactRect();
        } else {
            rect = device->exactBounds();
        }
        KisColorSpace* cs = device->colorSpace();
        QPoint clicked = e->pos().floorQPoint();
        QColor c;
        cs->toQColor( device->pixel(clicked.x(), clicked.y()), &c );
        double sum = 1/((double)c.red() + (double)c.green() + (double)c.blue());
        if( c.red() > 50 && c.red() * sum > 0.4  && c.green() * sum < 0.31 && c.blue() * sum < 0.36)
        {
            paint();
            // Region growing
            QValueList<QPoint> list;
            list.push_back( clicked);
            RedEyeRegion region;
            region.addPoint(clicked);
            while(! list.empty())
            {
                QPoint p = list.front();
                list.pop_front();
                for(int u = -1; u <= 1; u++)
                {
                    int xu = p.x() + u;
                    for(int v = -1; v <= 1; v++)
                    {
                        int yv = p.y() + v;
                        if(rect.contains(xu, yv) && !region.contains(QPoint(xu,yv)))
                        {
                            cs->toQColor( device->pixel(xu, yv), &c );
                            sum = 1/((double)c.red() + (double)c.green() + (double)c.blue());
                            if( c.red() > 50 && c.red() * sum > 0.4  && c.green() * sum < 0.31 && c.blue() * sum < 0.36)
                            {
                                region.addPoint(QPoint(xu,yv));
                                list.push_back( QPoint(xu,yv ));
                            }
                        }
                    }
                }
            }
            m_regions.push_back( region);
            paint();
        }

    }
}

#define AdjustBrightness(c, factor) ((Q_UINT8)CLAMP((0.4 * (factor) * 255 + (c)),0,255))
#define AdjustContrast(c, factor, meanL) ((Q_UINT8)CLAMP(((c) + (double)((c) - (meanL)) * (factor)),0,255))

void KisRedEyeRemovalTool::correctRegion( RedEyeRegion& region)
{
#if 1
    KisPaintDeviceSP device = m_currentImage->activeDevice ();
    KisTransaction* transaction = new KisTransaction(i18n("Remove a red-eye"), device);
    // Create a desaturate adjustement
    KisColorSpace* cs = device->colorSpace();
    KisColorAdjustment* desatAdj = cs->createDesaturateAdjustment();
    // Apply desaturate adjustement
    QRect rect = region.area();
    KisHLineIteratorPixel itPixel = device->createHLineIterator(rect.x(), rect.y(), rect.width(), true);
    RegionStatus* statusIt1 = region.mask();
    for(int y = 0; y < rect.height(); ++y)
    {
        for(int x = 0; x < rect.width(); ++x, ++statusIt1, ++itPixel)
        {
            if(*statusIt1 == MEMBER)
            {
                cs->applyAdjustment( itPixel.oldRawData(), itPixel.rawData(), desatAdj, 1 );
            }
        }
        itPixel.nextRow();
    }
    device->setDirty(rect);
    m_currentImage->undoAdapter()->addCommand(transaction);
#endif
}

QWidget* KisRedEyeRemovalTool::createOptionWidget(QWidget* parent)
{
    m_optionsWidget = new WdgRedEyeRemoval(parent);

    connect(m_optionsWidget->btnFixAll, SIGNAL(clicked()), SLOT(fixAllRegions()));
    connect(m_optionsWidget->btnUnmarkAllRegions, SIGNAL(clicked()), SLOT(unmarkAllRegions()));
//     return 0;
    return m_optionsWidget;
}

QWidget* KisRedEyeRemovalTool::optionWidget()
{
//     return 0;
    return m_optionsWidget;
}


#undef AdjustBrightness
#undef AdjustContrast

#include "kis_red_eye_removal_tool.moc"
