/* This file is part of the KDE project
   Copyright (C) 2005 Martin Ellis <m.a.ellis@ncl.ac.uk>
 
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "mdbmigrate.h"

// @todo Move string2Identifer to kexidb.
// #include <core/kexi.h>
/* This header file provides string2Identifier.  However the header isn't
  (and shouldn't be) installed.  Instead we need to move string2Identifer
  to KexiDB */

#include <qstring.h>
#include <qcstring.h>
#include <qregexp.h>
#include <qfile.h>
#include <qvariant.h>
#include <qdatetime.h>
#include <qvaluelist.h>
#include <kdebug.h>

#include <string.h>

using namespace KexiMigration;

/* This is the implementation for the MDB file import routines. */
KEXIMIGRATE_DRIVER_INFO( MDBMigrate, mdb );

/* ************************************************************************** */
//! Constructors
MDBMigrate::MDBMigrate()
{
  initBackend();
}

MDBMigrate::MDBMigrate(QObject *parent, const char *name,
                       const QStringList &args) :
	KexiMigrate(parent, name, args)
{
  initBackend();
}

/* ************************************************************************** */
//! Destructor
MDBMigrate::~MDBMigrate() {
  releaseBackend();
}

/* ************************************************************************** */
void MDBMigrate::initBackend() {
  mdb_init();
}
void MDBMigrate::releaseBackend() {
  mdb_exit();
}

/* ************************************************************************** */
/*! Connect to the db backend */
bool MDBMigrate::drv_connect() {
  kdDebug() << "mdb_open:" << endl;
  KexiDB::ConnectionData *m_data = m_externalData;

  // mdb_open takes a char*, not const char*, hence this nonsense.
  char *filename = qstrdup(m_data->fileName().local8Bit());
  mdb = mdb_open (filename, MDB_NOFLAGS);
  delete [] filename;

  if (!mdb) {
    kdDebug() << "mdb_open failed." << endl;
    return false;
  }
  return true;
}


/*! Disconnect from the db backend */
bool MDBMigrate::drv_disconnect()
{ 
  mdb_close(mdb);
  return true;
}

//! Get the table definition for a given table name
/*! Look up the table definition for the given table.  This only returns a ptr
    to the MdbTableDef - it doesn't load e.g. the column data.
    Remember to mdb_free_tabledef the table definition when it's finished
    with.
    \return the table definition, or null if no matching table was found */
MdbTableDef* MDBMigrate::getTableDef(const QString& tableName)
{
  MdbTableDef *tableDef = 0;

  // Look through each entry in the catalogue ...
  for (unsigned int i = 0; i < mdb->num_catalog; i++) {
    MdbCatalogEntry* dbObject = 
        (MdbCatalogEntry*)(g_ptr_array_index (mdb->catalog, i));

    // ... for a table with the given name
    if (dbObject->object_type == MDB_TABLE) {
      QString dbObjectName = QString::fromLocal8Bit(dbObject->object_name);
      if (dbObjectName.lower() == tableName.lower()) {
        tableDef = mdb_read_table(dbObject);
        break;
      }
    }
  }

  return tableDef;
}

/* ************************************************************************** */
/*! Get the types and properties for each column. */
bool MDBMigrate::drv_readTableSchema(const QString table)
{
//! @todo add option for this?
//! @todo move Kexi::string2Identifier to kexidb (it's in kexicore now)
//  const QString tableID( Kexi::string2Identifier(table).lower() );
  // Er, Kexi can't read in the db table unless we .lower()
  const QString tableID(table.lower());
  m_table = new KexiDB::TableSchema(tableID);
  m_table->setCaption(table);

  //! Get the column meta-data
  MdbTableDef *tableDef = getTableDef(table);
  if(!tableDef) {
    kdDebug() << "MDBMigrate::drv_getTableDef: couldn't find table "
              << table << endl;
    return false;
  }
  mdb_read_columns(tableDef);
  kdDebug() << "MDBMigrate::drv_readTableSchema: #cols = "
            << tableDef->num_cols << endl;

  /*! Convert column data to Kexi TableSchema
     Nice mix of terminology here, MDBTools has columns, Kexi has fields. */
  MdbColumn *col;
  for (unsigned int i = 0; i < tableDef->num_cols; i++) {
    col = (MdbColumn*) g_ptr_array_index(tableDef->columns, i);

    // Field name: who know what kind of encoding?
    QString fldName = QString::fromLocal8Bit(col->name);
    kdDebug() << "MDBMigrate::drv_readTableSchema: got column "
        << fldName << "\"" << col->name << endl;

//! @todo Use const QString fldID( Kexi::string2Identifier(fldName).lower() );
//  Needs string2Identifier to be installed in KexiDB devel files.
//  For now, at least change spaces to '_'
    QString fldID(fldName.lower().replace(" ", "_"));

    // Field type
    KexiDB::Field *fld = 
        new KexiDB::Field(fldID, type(col->col_type));

    kdDebug() << "MDBMigrate::drv_readTableSchema: size "
        << col->col_size << " type " << type(col->col_type) <<endl;
    fld->setCaption(fldName);
    m_table->addField(fld);
  }

  getPrimaryKey(m_table, tableDef);
  //! Free the column meta-data - as soon as it doesn't seg fault.
  //mdb_free_tabledef(tableDef);
  return true;
}


/*! Get a list of tables and put into the supplied string list */
bool MDBMigrate::drv_tableNames(QStringList& tableNames)
{
  // Try to read the catalogue of database objects
  if (!mdb_read_catalog (mdb, MDB_ANY)) {
    kdDebug() << "MDBMigrate::drv_tableNames: couldn't read catalogue" << endl;
    return false;
  }

  // Add non-system tables to the list
  for (unsigned int i = 0; i < mdb->num_catalog; i++) {
    MdbCatalogEntry* dbObject = 
        (MdbCatalogEntry*)(g_ptr_array_index (mdb->catalog, i));
    if (dbObject->object_type == MDB_TABLE) {
      QString dbObjectName = QString::fromLocal8Bit(dbObject->object_name);
      if (!dbObjectName.startsWith("MSys")) {
        kdDebug() << "MDBMigrate::drv_tableNames: " << dbObjectName << endl;
        tableNames << dbObjectName;
      }
    }
  }
  return true;
}


QVariant MDBMigrate::toDateTime(const char* data) {
  // Need to convert "DD/MM/YY HH:MM:SS" to "YYYY/MM/DDTHH:MM:SS"
  QRegExp rx = QRegExp("^([0-9]{1,2})/([0-9]{1,2})/([0-9]{1,4}) "
                       "([0-9]{1,2}):([0-9]{1,2}):([0-9]{1,2})");
  QString date = QString(data);
  QStringList values = QStringList();
  rx.search(date, 0);
  int matchedLen = rx.matchedLength();

  if (matchedLen != -1) {
    QString year = QString("%1").arg(rx.cap(3).toInt(),4);
    QString isoDate = year      + "/" + rx.cap(2) + "/" + rx.cap(1) + " " +
                      rx.cap(4) + "/" + rx.cap(5) + "/" + rx.cap(6);
    return QVariant(QDateTime::fromString(isoDate, Qt::ISODate));
  } else {
    kdDebug() << "MDBMigrate::toDateTime: Didn't understand " << data << endl;
    return QVariant();
  }
}

QVariant MDBMigrate::toQVariant(const char* data, unsigned int len, int type) {
  if(len == 0) {
    // Null ptr => null value
    return QVariant();
  } else if (type == MDB_SDATETIME ) {
    return toDateTime(data);
  } else {
    return QVariant(QString::fromUtf8(data, len));
  }
}

/*! Copy MDB table to KexiDB database */
bool MDBMigrate::drv_copyTable(const QString& srcTable,
                               KexiDB::TableSchema* dstTable) {
  QString kdLoc = "MDBMigrate::drv_copyTable: ";
  MdbTableDef *tableDef = getTableDef(srcTable);
  if(!tableDef) {
    kdDebug() << kdLoc << srcTable << endl;
    return false;
  }

  char *columnData[256];
  int columnDataLength[256];

  //! Bind + allocate the DB columns to columnData and columnDataLength arrays
  mdb_read_columns(tableDef); // mdb_bind_column dies without this
  for (unsigned int i = 0; i < tableDef->num_cols; i++) {
    columnData[i] = (char*) g_malloc(MDB_BIND_SIZE);
    // Columns are numbered from 1
    // and why aren´t these unsigned ints?
    mdb_bind_column(tableDef, i + 1, columnData[i], &columnDataLength[i]);
  }

  //! Copy each row into vals
  mdb_rewind_table(tableDef);
  kdDebug() << kdLoc << "Fetching " << tableDef->num_rows << " rows" << endl;

  KexiDB::Transaction transaction(m_kexiDB->beginTransaction());
  KexiDB::TransactionGuard tg(transaction);
  if (!transaction.isNull()) {
   while(mdb_fetch_row(tableDef)) {
      QValueList<QVariant> vals = QValueList<QVariant>();

      kdDebug() << kdLoc << "Copying " << tableDef->num_cols << " cols" << endl;
      for (unsigned int i = 0; i < tableDef->num_cols; i++) {
        MdbColumn *col = (MdbColumn*) g_ptr_array_index(tableDef->columns, i);

        if (col->col_type == MDB_OLE && col->cur_value_len) {
          kdDebug() << kdLoc << "OLE!! @ " << i << endl;
          mdb_ole_read(mdb, col, columnData[i], MDB_BIND_SIZE);
        }

        //! @todo: How to import binary data?
        QVariant var = toQVariant(columnData[i], columnDataLength[i],
                                col->col_type);
        kdDebug() << kdLoc << " col=" << i 
                << " data=" << columnData[i] 
                << " qvar=" << var.toString() << endl;
        vals << var;
      }
      m_kexiDB->insertRecord(*dstTable, vals);
      progressDoneRow();
    }
  }
  tg.commit();
  //! Deallocate (unbinding) the DB columns arrays and column meta-data
  for (unsigned int i = 0; i < tableDef->num_cols; i++) {
    g_free(columnData[i]);
  }

  // When memory leaks are better than seg. faults...
  //mdb_free_tabledef(tableDef);

  return true;
}


//! Convert a MySQL type to a KexiDB type, prompting user if necessary.
KexiDB::Field::Type MDBMigrate::type(int type)
{
  // Field type
	KexiDB::Field::Type kexiType = KexiDB::Field::InvalidType;

	switch(type)
	{
    case MDB_BOOL:
      kexiType = KexiDB::Field::Boolean;
      break;
    case MDB_BYTE:
      kexiType = KexiDB::Field::Byte;
      break;
    case MDB_INT:
      kexiType = KexiDB::Field::Integer;
      break;
    case MDB_LONGINT:
      kexiType = KexiDB::Field::BigInteger;
      break;
    case MDB_MONEY:
      kexiType = KexiDB::Field::LongText; //userType();
      break;
    case MDB_FLOAT:
      kexiType = KexiDB::Field::Float;
      break;
    case MDB_DOUBLE:
      kexiType = KexiDB::Field::Double;
      break;
    case MDB_SDATETIME:
      kexiType = KexiDB::Field::DateTime;
      break;
    case MDB_TEXT:
      kexiType = KexiDB::Field::LongText;
      break;
    case MDB_OLE:
      kexiType = KexiDB::Field::BLOB;
      break;
    case MDB_MEMO:
      kexiType = KexiDB::Field::LongText;
      break;
    case MDB_REPID:
      // ?
    case MDB_NUMERIC:
      // ?
		default:
		  kexiType = KexiDB::Field::InvalidType;
	}

	//Ask the user what to do with this field if we don't know what it is.
	if (kexiType == KexiDB::Field::InvalidType) {
		return KexiDB::Field::LongText; //userType();
	}
	return kexiType;
}


bool MDBMigrate::getPrimaryKey(KexiDB::TableSchema* /*table*/, MdbTableDef* tableDef) {
  QString kdLoc = "MDBMigrate::getPrimaryKey: ";
  MdbIndex *idx;

  if (!tableDef) {
    return false;
  }
  mdb_read_columns(tableDef);
  mdb_read_indices(tableDef);

  //! Find the PK index in the MDB file
  bool foundIdx = false;
  for (unsigned int i = 0; i < tableDef->num_idxs; i++) {
    idx = (MdbIndex*) g_ptr_array_index (tableDef->indices, i);
    QString fldName = QString::fromLocal8Bit(idx->name);
    kdDebug() << kdLoc << "got index "
        << fldName << "\"" << idx->name << endl;
    if (!strcmp(idx->name, "PrimaryKey")) {
      idx = (MdbIndex*) g_ptr_array_index (tableDef->indices, i);
      foundIdx = true;
      break;
    }
  }

  if(!foundIdx) {
    mdb_free_indices(tableDef->indices);
    return false;
  }

  //! @todo: MDB index order (asc, desc)

  kdDebug() << kdLoc << "num_keys " << idx->num_keys << endl;

  //! Create the KexiDB IndexSchema ...
  QByteArray key_col_num(idx->num_keys);

  // MDBTools counts columns from 1 - subtract 1 where necessary
  KexiDB::IndexSchema* p_idx = new KexiDB::IndexSchema(m_table);
  for (unsigned int i = 0; i < idx->num_keys; i++) {
    key_col_num[i] = idx->key_col_num[i];
    kdDebug() << kdLoc << "key " << i + 1 
              << " col " << key_col_num[i]
              << m_table->field(idx->key_col_num[i-1])->name()
              << endl;
    p_idx->addField(m_table->field(idx->key_col_num[i-1]));
  }

  kdDebug() << kdLoc << p_idx->debugString() << endl;

  //! ... and add it to the table definition
  // but only if the PK has only one field, so far :o(
  if(idx->num_keys == 1) {
    m_table->field(idx->key_col_num[0] - 1)->setPrimaryKey(true);
  } else {
    //! @todo: How to add a composite PK to a TableSchema?
    //m_table->setPrimaryKey(p_idx);
  }

  mdb_free_indices(tableDef->indices);
  return true;
}

bool MDBMigrate::drv_getTableSize(const QString& table, Q_ULLONG& size) {
  //! Get the column meta-data
  MdbTableDef *tableDef = getTableDef(table);
  if(!tableDef) {
    kdDebug() << "MDBMigrate::drv_getTableDef: couldn't find table "
              << table << endl;
    return false;
  }
  size = (Q_ULLONG)(tableDef->num_rows);
  mdb_free_tabledef(tableDef);
  return true;
}


#include "mdbmigrate.moc"
