/***************************************************************************
                          arraydlg.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfileinfo.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qlayout.h>
#include <kapp.h>
#include <klocale.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <kseparator.h>
#include <kcolorbtn.h>
#include <kmessagebox.h>
#include "arraydlg.h"
#include "kpldoc.h"
#include "symboldlg.h"
#include "arrayitem.h"
#include "kpldoubleedit.h"
#include "kplspinbox.h"

ArrayDlg::ArrayDlg(QWidget* parent, KplDoc* model, ArrayItem* ad0) :
 QDialog(parent, 0, true), m(model), ad(ad0), adt(0)
{
  resize(400, 230);
  setCaption(i18n("Array"));
  QVBoxLayout* vbox = new QVBoxLayout(this, 11, 6);
  QGridLayout* grid = new QGridLayout(vbox, 3, 6, 6);
  QPushButton* b = new QPushButton(i18n("File"), this);
  grid->addWidget(b, 0, 0);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  QHBoxLayout* hbox = new QHBoxLayout(6);
  hbox->addWidget(fileName = new QLabel(ad->url.isLocalFile() ?
                                        ad->url.path() : ad->url.url(), this));
  fileName->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  fileName->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::Minimum));
  hbox->addWidget(bReload = new QPushButton(this));
  bReload->setPixmap(BarIcon("reload"));
  bReload->setDisabled(fileName->text().isEmpty());
  connect(bReload, SIGNAL(clicked()), SLOT(slotReload()));
  grid->addMultiCellLayout(hbox, 0, 0, 1, 5);
  grid->addWidget(new QLabel(i18n("x column"), this), 1, 0);
  grid->addWidget(ix = new KplSpinBox(0, 0, 1, this), 1, 1);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 1, 3);
  grid->addWidget(new QLabel(i18n("y column"), this), 1, 4);
  grid->addWidget(iy = new KplSpinBox(0, 0, 1, this), 1, 5);
  grid->addWidget(new QLabel(i18n("Error column"), this), 2, 0);
  grid->addWidget(ie = new KplSpinBox(0, 0, 1, this), 2, 1);
  ie->setEnabled(ad->errbars);
  grid->addMultiCellWidget(err = new QCheckBox(i18n("Error bar"), this),
                           2, 2, 2, 5);
  err->setChecked(ad->errbars);
  connect(err, SIGNAL(toggled(bool)), ie, SLOT(setEnabled(bool)));
  grid->addWidget(new QLabel(i18n("Start index"), this), 3, 0);
  grid->addWidget(iStart = new KplSpinBox(0, 0, 1, this), 3, 1);
  connect(iStart, SIGNAL(valueChanged(int)), SLOT(slotStartChanged(int)));
  grid->addWidget(new QLabel(i18n("Number of points"), this), 3, 4);
  grid->addWidget(n = new KplSpinBox(0, 0, 1, this), 3, 5);
  connect(n, SIGNAL(valueChanged(int)), SLOT(slotNChanged(int)));
  grid->addWidget(new QLabel(i18n("x normalization"), this), 4, 0);
  grid->addWidget(efx = new KplDoubleEdit(ad->fx, 0.0, 0.0, this, "%.6lg"),
                  4, 1);
  grid->addWidget(new QLabel(i18n("y normalization"), this), 4, 4);
  grid->addWidget(efy = new KplDoubleEdit(ad->fy, 0.0, 0.0, this, "%.6lg"),
                  4, 5);
  grid->addWidget(new QLabel(i18n("Symbol"), this), 5, 0);
  grid->addWidget(symb = new KplSpinBox(-17, 9, 1, this), 5, 1);
  symb->setValue(ad->symb);
  SymbolButton *bSymb = new SymbolButton(this, symb);
  bSymb->setFixedWidth(50);
  grid->addWidget(bSymb, 5, 2);
  grid->addWidget(new QLabel(i18n("Color"), this), 5, 4);
  grid->addWidget(colData = new KColorButton(ad->color, this), 5, 5);
  colData->setMinimumWidth(50);
  KSeparator* hl = new KSeparator(KSeparator::HLine, this);
  vbox->addWidget(hl);
  hbox = new QHBoxLayout(vbox);
  hbox->addWidget(b = new QPushButton(i18n("Help"), this));
  connect(b, SIGNAL(clicked()), SLOT(slotHelp()));
  hbox->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding));
  hbox->addWidget(b = new QPushButton(i18n("OK"), this));
  b->setDefault(true);
  connect(b, SIGNAL(clicked()), SLOT(slotOK()));
  hbox->addWidget(b = new QPushButton(i18n("Apply"), this));
  connect(b, SIGNAL(clicked()), SLOT(slotApply()));
  hbox->addWidget(b = new QPushButton(i18n("Cancel"), this));
  connect(b, SIGNAL(clicked()), SLOT(reject()));
  setSpinBoxes(ad);
}

ArrayDlg::~ArrayDlg()
{
  delete adt;
}

void ArrayDlg::setSpinBoxes(ArrayItem* ad)
{
  int iColMax = QMAX(0, ad->ncols - 1);
  ix->setRange(0, iColMax);
  ix->setValue(ad->ix);
  iy->setRange(0, iColMax);
  iy->setValue(ad->iy);
  ie->setRange(0, iColMax);
  ie->setValue(ad->ie);
  iStart->setRange(0, QMAX(0, ad->nrows - 1));
  iStart->setValue(ad->istart);
  n->setRange(0, ad->nrows);
  n->setValue(ad->n);
}

bool ArrayDlg::load(KURL &url)
{
  bool success = !url.isEmpty();
  if (success) {
    QString u;
    if (url.isLocalFile())
      u = url.path();
    else
      u = url.url();
      delete adt;
      adt = new ArrayItem;
      adt->url = url;
      success = adt->readFile();
      if (success) {
        int maxCol = adt->ncols - 1;
        adt->ix = QMIN(ad->ix, maxCol);
        adt->iy = QMIN(ad->iy, maxCol);
        adt->ie = QMIN(ad->ie, maxCol);
        adt->istart = QMIN(ad->istart, adt->nrows - 1);
        adt->n = adt->nrows - adt->istart;
        if (ad->n)
          adt->n = QMIN(ad->n, adt->n);
        setSpinBoxes(adt);
      } else {
        delete adt;
        adt = 0;
        KMessageBox::error(this, i18n("while loading the file"));
      }
  }
  return success;
}

void ArrayDlg::getValues(bool ok)
{
  if (adt) {
    *ad = *adt;
    delete adt;
    adt = 0;
  }
  ad->fx = efx->value();
  ad->fy = efy->value();
  ad->color = colData->color().rgb();
  ad->ix = ix->interpretedValue();
  ad->iy = iy->interpretedValue();
  ad->ie = ie->interpretedValue();
  ad->symb = symb->interpretedValue();
  ad->istart = iStart->interpretedValue();
  ad->n = n->interpretedValue();
  ad->url = fileName->text();
  ad->errbars = err->isChecked();
  ad->istart = QMAX(ad->istart, 0);
  ad->n = QMIN(ad->n, ad->nrows - ad->istart);
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
  else
    setSpinBoxes(ad);
}

void ArrayDlg::slotStartChanged(int ia)
{
  int nmax = ad->nrows - ia;
  if (n->interpretedValue() > nmax)
    n->setValue(nmax);
}

void ArrayDlg::slotNChanged(int na)
{
  int imin = ad->nrows - na;
  if (iStart->interpretedValue() > imin)
    iStart->setValue(imin);
}

void ArrayDlg::slotFile()
{
  KURL url = KFileDialog::getOpenURL(m->currentDir(), "*.dat *.DAT\n*");
  if (load(url)) {
    fileName->setText(url.isLocalFile() ? url.path() : url.url());
    m->setCurrentDir(url);
    bReload->setDisabled(fileName->text().isEmpty());
  }
}

void ArrayDlg::slotReload()
{
  KURL url(fileName->text());
  load(url);
}

void ArrayDlg::slotOK()
{
  getValues(true);
}

void ArrayDlg::slotApply()
{
  getValues(false);
}

void ArrayDlg::slotHelp()
{
  kapp->invokeHelp("SEC-ARRAY");
}
