/***************************************************************************
                          knetinet.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include	<stdio.h>
#include	<stdarg.h>
#include	<errno.h>
#include	<unistd.h>

#include	<pthread.h>

#include	<qarray.h>

#include	<netdb.h>
#include	<sys/socket.h>
#include	<sys/ioctl.h>
#include	<net/if.h>
#include	<netinet/in.h>

#include	"ksnuffle.h"

typedef	struct	ifconf	IFCONF	;	/* Interface configuration	*/
typedef	struct	ifreq	IFREQ	;	/* Interface request		*/
typedef	struct	hostent	HOSTENT	;	/* Host entry			*/
typedef	struct	servent	SERVENT	;	/* Service entry		*/

#define	STRERR	strerror(errno)

extern	bool	d_addr	;	/* Decode IP addresses to names		*/
extern	bool	d_port	;	/* Decode IP ports to service names	*/
extern	bool	j_priv	;	/* Just list privileged ports/services	*/

/*  BubbleSort2	: Sort all but first element in list			*/
/*  list	: SArray *	: Array of string pointers		*/
/*  (returns)	: SArray *	: The array				*/

static	SArray	*BubbleSort2
	(	SArray	*list
	)
{
	for (unsigned int idx1 = 2 ; idx1 < list->size() ; idx1 += 1)
		for (unsigned int idx2 = idx1 ; idx2 > 1 ; idx2 -= 1)
			if (strcmp ((*list)[idx2], (*list)[idx2-1]) < 0)
			{	char	*tproto	= (*list)[idx2-1] ;
				(*list)[idx2-1]	= (*list)[idx2  ] ;
				(*list)[idx2  ]	= tproto ;
			}
			else	break	;
	return	list	;
}

/*  INTERFACES								*/
/*  ----------								*/

/*  FindAllIface: Find all interfaces					*/
/*  (returns)	: QArray<char *> *	: List of interfaces		*/

QArray<char *>	*FindAllIface ()
{
	
	IFCONF	ifconf		;
	IFREQ	ifreq		;
	IFREQ	*ifr		;
	char	buff[BUFSIZ]	;
	char	*cp		;
	int	sockfd		;
	int	cnt		= 0	;
	SArray	*list		= new QArray<char *> (64) ;

	if ((sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
		ErrExit	("can't create rx socket: %s", STRERR) ;

	ifconf.ifc_len	= sizeof(buff)	;
	ifconf.ifc_buf	= buff		;
	if (ioctl (sockfd, SIOCGIFCONF, (char *)&ifconf) < 0)
		ErrExit	("ioctl SIOCGIFCONF failed: %s", STRERR) ;

	/* The loop scans all interfaces, rejecting any that are not up	*/
	/* and that do not support broadcast; also skip any loopback	*/
	/* interfaces. For each that is OK, add an interface record to	*/
	/* the list.							*/
	for (cp = buff ; cp < buff + ifconf.ifc_len ; cp += sizeof(ifr->ifr_name) + sizeof(ifr->ifr_addr))
	{
		ifr	= (IFREQ *)cp	;
		ifreq	= *ifr		;

		if (ioctl (sockfd, SIOCGIFFLAGS, (char *)&ifreq) < 0)
			ErrExit	("ioctl SIOCGIFFLAGS failed: %s", STRERR) ;
		if ((ifreq.ifr_flags & IFF_UP) == 0)
			continue ;

		(*list)[cnt++] = strdup (ifr->ifr_name) ;
	}

	close	(sockfd)   ;
	list->resize (cnt) ;
	return	list	   ;
}


/*  ETHERNET PROTOCOLS							*/
/*  ------------------							*/

/*  FindAllEProto: Load all ethernet protocols					*/
/*  (returns)	 : QArray<char *> *	: List of protocols		*/

QArray<char *>	*FindAllEProto ()
{
	SArray	*list	= new QArray<char *> (32) ;
	int	cnt	;

	/* The list of ethernet protocols recognised by libpcap is held	*/
	/* in an array of (char *, u_short) structures, named eproto_db	*/
	/* Scan this to get our list.					*/
	struct	EProto
	{	char	*name	;
		u_short	proto	;
	}	;

	extern	EProto	eproto_db[] ;

	(*list)[0] = "" ;
	cnt	   = 1  ;

	for (EProto *ep = &eproto_db[0] ; ep->name != NULL ; ep += 1)
		(*list)[cnt++] = ep->name	;

	list->resize (cnt) ;

	/* Sort the protocols into alphabetic order, except that we	*/
	/* leave the blank entry at the top. This is then returned.	*/
	return	BubbleSort2 (list) ;
}


/*  INTERNET PROTOCOLS							*/
/*  ------------------							*/

/*  FindAllIProto: Load all internet protocols					*/
/*  (returns)	 : QArray<char *> *	: List of protocols		*/

QArray<char *>	*FindAllIProto ()
{
	/* Libpcap seems to know only about TCP/IP and UDP/IP, so there	*/
	/* is no point scanning /etc/protocols.				*/
	SArray	*list	= new QArray<char *> (3) ;

	(*list)[0]	= ""	;
	(*list)[1]	= "tcp"	;
	(*list)[2]	= "udp"	;

	return	list	;
}


/*  SERVICES								*/
/*  ---------								*/

/*  As well as generating a list of all services, we maintain a hash	*/
/*  table of service names, hashed on the port number, so that we can	*/
/*  convert ports back to service names.				*/

typedef	struct	_phash	PHash	   ;

struct	_phash
{	struct	_phash	*next	   ;
	int		port	   ;
	char		*serv	   ;
}	;

#define	PHSIZE	256
static	PHash		*phashtab[PHSIZE] ;

/*  FindAllService: Load all services					*/
/*  (returns)	  : QArray<char *> *	: List of services		*/

QArray<char *>	*FindAllService ()
{
	SERVENT	*service ;
	char	*serv	 ;
	int	port	 ;
	int	hashv	 ;
	PHash	*phash	 ;
	int	cnt	 ;
	SArray	*list	 = new QArray<char *> (256) ;

	setservent (1)	 ;

	(*list)[0] = ""	 ;
	cnt	   = 1	 ;

	while ((service = getservent ()) != NULL)
	{
		(*list)[cnt++]  = serv = strdup (service->s_name) ;
		port		= ntohs 	(service->s_port) ;

		phash		= new PHash		;
		hashv		= port & (PHSIZE-1)	;

		phash->port	= port			;
		phash->serv	= (char *) malloc (strlen (serv) + 2) ;
		phash->serv[0]	= '.' 			;
		strcpy	(&phash->serv[1], serv)		;

		phash->next	= phashtab[hashv]	;
		phashtab[hashv]	= phash			;
	}

	endservent ()  ;
	list->resize (cnt) ;

	/* Sort the service array (as above, leaving the first entry in	*/
	/* place), and return it.					*/
	return	BubbleSort2 (list) ;
}

/*  FindService	: Look up a service					*/
/*  port	: int		: Port number				*/
/*  (returns)	: char *	: Service name				*/

char	*FindService
	(	int	port
	)
{
	static	char	ptext[32] ;

	if (d_port)
		for (PHash *php = phashtab[port&(PHSIZE-1)] ; php != NULL ; php = php->next)
			if (php->port == port)
				return	php->serv ;

	sprintf	(ptext, "%d", port) ;
	return	ptext	;
}


/*  HOST NAMES								*/
/*  ----------								*/

typedef	struct	_hm	HostMap	 ;

enum	HMState
{	Unmapped,			/* Address never been mapped	*/
	Mapping,			/* Address is being mapped	*/
	Mapped				/* Address mapped to name	*/
}	;

struct	_hm
{	HostMap	*next		 ;	/* Next on hash queue		*/
	long	addr		 ;	/* Binary address value		*/
	HMState	state		 ;	/* Address mapping state	*/
	char	dots[ 20]	 ;	/* Dot-notation address		*/
	char	name[128]	 ;	/* Host name when mapped	*/
}	;

#define	HMAPSIZ	256
static	HostMap	*hostmap[HMAPSIZ];	/* Name lookup cache		*/
static	int	nslsock		 ;	/* Socket to name lookup thread	*/

/*  FindHost	: Get host name from address				*/
/*  _addr	: char *	: AF_INET type address			*/
/*  (returns)	: char *	: Name					*/

char	*FindHost
	(	char	*_addr
	)
{
	long	addr	= *(long *)_addr	;
	long	addr2	= ntohl (addr)		;
	int	hash	= addr & (HMAPSIZ - 1)	;
	HostMap	*hmap	;

	/* Scan the hash table to see if the name is already there, in	*/
	/* which case we do not need to do a lookup and can return the	*/
	/* name or dot address at once.					*/
	for (hmap = hostmap[hash] ; hmap != NULL ; hmap = hmap->next)
		if (hmap->addr == addr)
		{	if ((hmap->state == Unmapped) && d_addr)
			{	write	(nslsock, &addr, sizeof(addr)) ;
				hmap->state = Mapping ;
			}
			return	d_addr ? hmap->name : hmap->dots ;
		}


	/* Now allocate a new hash table entry, and initially set the	*/
	/* dot address in place of the name. This should later be	*/
	/* replaced by the real name.					*/
	hmap		= (HostMap *)malloc(sizeof(HostMap)) ;
	hmap->next	= hostmap[hash] ;
	hmap->addr	= addr		;
	hostmap[hash]	= hmap		;

	sprintf	(hmap->name, "%ld.%ld.%ld.%ld",
			     (addr2 >> 24) & 0xff,
			     (addr2 >> 16) & 0xff,
			     (addr2 >>  8) & 0xff,
			     (addr2 >>  0) & 0xff) ;

	assert	(strlen (hmap->name) < sizeof(hmap->dots)) ;
	strcpy	(hmap->dots, hmap->name) ;

	/* If mapping addresses, write the IP address into the name	*/
	/* lookup request queue, to be processed in the name lookup	*/
	/* thread.							*/
	if (d_addr)
	{	write	(nslsock, &addr, sizeof(addr)) ;
		hmap->state = Mapping  ;
	}
	else	hmap->state = Unmapped ;

	/* Return the dot address. The user will have to make do with	*/
	/* this for the time being.					*/
	return	hmap->dots ;
}

/*  threadStart	: Start routine for nameserver lookup thread		*/
/*  arg		: void *	: Actually communications socket	*/
/*  (returns)	: void *	:					*/

static	void	*threadStart
	(	void	*arg
	)
{
	long	addr	  ;
	HOSTENT	*host	  ;
	char	name[128] ;

	/* The thread sits in a loop receiving addresses, looking up	*/
	/* the corresponding name, and returning. Exit at any point	*/
	/* that there is a read or write error.				*/
	while (read ((int)arg, &addr, sizeof(addr)) == sizeof(addr))
	{
		if ((host = gethostbyaddr ((char *)&addr, sizeof(addr), AF_INET)) != NULL)
		{	assert	(strlen (host->h_name) < sizeof(name)) ;
			strcpy	(name, host->h_name) ;
		}
		else	strcpy	(name, "") ;

		if ( (write ((int)arg, &addr, sizeof(addr)) != sizeof(addr)) ||
		     (write ((int)arg, name,  sizeof(name)) != sizeof(name)) )
			break	;
	}

	close	((int)arg) ;
	return	NULL	   ;
}

/*  KNDApp								*/
/*  sockRead	: Handle nameserver lookup reply			*/
/*  ksock	: KSocket *	: Connection to nameserver thread	*/
/*  (returns)	: void		:					*/

void	KNDApp::sockRead
	(	KSocket	*ksock
	)
{
	long	addr	  ;
	char	name[128] ;

	/* Read the address and the corresponding name, which comes in	*/
	/* a fixed size field. If the lookup failed then do no more as	*/
	/* the hash table will contain the address in dot format.	*/
	if ( (read (ksock->socket(), &addr, sizeof(addr)) != sizeof(addr)) ||
	     (read (ksock->socket(), name,  sizeof(name)) != sizeof(name)) )
		ErrExit	("Ksnuffle socket read error", STRERR) ;

	if (name[0] == 0) return ;

	/* Scan the hash table for the address, and update with the	*/
	/* name. This should not fail, not that it matters.		*/
	int	hash	= addr & (HMAPSIZ - 1)	;
	HostMap	*hmap	;

	for (hmap = hostmap[hash] ; hmap != NULL ; hmap = hmap->next)
		if (hmap->addr == addr)
		{	assert	(strlen(name) < sizeof(hmap->name)) ;
			strcpy	(hmap->name, name) ;
			hmap->state = Mapped	   ;
			break	;
		}
}

/*  KNDApp								*/
/*  startNsl	: Start the nameserver lookup thread			*/
/*  (returns)	: void		:					*/

void	KNDApp::startNsl ()
{
	int		sockp[2]	;
	KSocket		*ksock		;
	pthread_t	thread		;

	/* Create a socket pair. This will be used to communicate	*/
	/* between the main process and the name lookup thread. Then	*/
	/* build a KSocket to be used in the main process to handle	*/
	/* returning results.						*/
	if (socketpair (AF_UNIX, SOCK_STREAM, 0, sockp) < 0)
		ErrExit	("KSnuffle socketpair error", STRERR) ;

	ksock	= new KSocket (sockp[1]) ;
	ksock->enableRead (true) ;
	connect	(ksock, SIGNAL(readEvent(KSocket *)), this, SLOT(sockRead(KSocket *))) ;

	/* Next create the thread which will actually handle the	*/
	/* nameserver lookups ...					*/
	if (pthread_create (&thread, NULL, threadStart, (void *)sockp[0]) != 0)
		ErrExit	("KSnuffle thread error", STRERR) ;

	nslsock	= sockp[1] ;
}
