/***

kifstats.c	- the interface statistics module
Written by Walt A. Boring IV and
Gerard Paul Java (iptraf)
Copyright (c) Gerard Paul Java 1997

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.
	
***/

#include <kmsgbox.h>
#include <qsocketnotifier.h>
#include <qlistview.h>
#include <qstring.h>
#include <qtimer.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if_arp.h>
#include <linux/if_packet.h>
#include <net/if.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <linux/if_ether.h>

#include "ifaces.h"
#include "packet.h"
//#include "instances.h"
#include "options.h"

#include "kifstats.moc"

void initiftab(struct iftab *table);
void positionptr(struct iftab *table, struct iflist **ptmp, char *ifname);
void printifentry(struct iflist *ptmp, unsigned int idx);
void destroyiflist(struct iflist *list);
char *ltrim(char *buf);
int ifinlist(struct iflist *list, char *ifname);

//Constructor for General Interface stats
IfStatsGeneral::IfStatsGeneral(QWidget *parent, struct OPTIONS *appoptions)

{
  starttime = 0;
  statbegin = 0;
  now = 0;
  unow = 0;
  startlog = 0;

  options = appoptions;

  gencolumns.numcols=1;
  gencolumns.total=0;
  gencolumns.total_ip=0;
  gencolumns.non_ip=0;
  gencolumns.bad_ip=0;
  gencolumns.activity=0;

  mylayout = new QGridLayout(this, 1,1);

  mylistview = new QListView(parent, "NetListView");
  mylistview->setGeometry(0,56,480,240);
//  parent->setView(mylistview);

  //mylayout->addWidget(mylistview);
  mylistview->addColumn("Iface",-1);
  mylistview->setColumnWidth(0,50);

  if (options->genopts.total)
    {
      mylistview->addColumn("Total",-1);
      gencolumns.total=1;
      mylistview->setColumnWidth(gencolumns.total,65);
      gencolumns.numcols++;
    }
  if (options->genopts.total_ip)
    {
      mylistview->addColumn("IP",-1);
      gencolumns.total_ip = gencolumns.numcols;
      mylistview->setColumnWidth(gencolumns.total_ip,65);
      gencolumns.numcols++;
    }
  if (options->genopts.non_ip)
    {
      mylistview->addColumn("NonIP",-1);
      gencolumns.non_ip = gencolumns.numcols;
      mylistview->setColumnWidth(gencolumns.non_ip,50);
      gencolumns.numcols++;
    }
  if (options->genopts.bad_ip)
    {
      mylistview->addColumn("BadIP",-1);
      gencolumns.bad_ip = gencolumns.numcols;
      mylistview->setColumnWidth(gencolumns.bad_ip,50);
      gencolumns.numcols++;
    }
  if (options->genopts.activity)
    {
      mylistview->addColumn("Activity",-1);
      gencolumns.activity = gencolumns.numcols;
      mylistview->setColumnWidth(gencolumns.activity,120);
  //    gencolumns.numcols++;
    }
  mylistview->triggerUpdate();
  mylistview->setColumnWidthMode(1,QListView::Maximum);
  mylistview->setAllColumnsShowFocus(1);
  mylistview->setSorting(-1, FALSE);
  mylistview->show();

  initiflist(&(table.head));

  if (table.head != NULL) {
    open_socket(&fd);

    if (fd < 0)
      return;

    gettimeofday(&tv, NULL);
    starttime = startlog = statbegin = tv.tv_sec;

    tempstr = new QString();
    tempstr2 = new QString();
    mycallback = new QSocketNotifier(fd, QSocketNotifier::Read);
    connect(mycallback, SIGNAL(activated(int)), SLOT(dataReceived(int)));

    mytimer = new QTimer(this, "General Stats Timer");
    connect( mytimer, SIGNAL(timeout()), SLOT(timerDone()) );
    mytimer->start( 1000, FALSE );                 // 2 seconds single-shot
  }
}

IfStatsGeneral::~IfStatsGeneral()
{
  destroyiflist(table.head);
  mytimer->stop();
  delete(tempstr);
  delete(tempstr2);
  delete(mytimer);
  delete(mycallback);
  delete (mylistview);
  //close(fd);
}


void
IfStatsGeneral::updateOptions(struct OPTIONS *appoptions)
{
// options->promisc = appoptions->promisc;
// options->actmode = appoptions->actmode;
}


//This actually processes our packet
void IfStatsGeneral::dataReceived(int fd)
{
  char buf[8192];
  struct sockaddr_pkt fromaddr;
  struct iflist *ptmp = NULL;
  int ch;
  int br;
  char ifname[10];
  unsigned short linktype;
  char *packet;
  char aligned_buf[120];
  struct iphdr *ipacket;
  char *tpacket;
  int iphlen;
  int ipck, ipcsum;

  getpacket(fd, buf, &fromaddr, &ch, &br);
      
    if (br > 0) 
      {
        strcpy(ifname, (const char *)fromaddr.spkt_device);
        positionptr(&table, &ptmp, ifname);

        ptmp->total++;

        ptmp->spanbr += br;
        ptmp->br += br;

        if (fromaddr.spkt_protocol == ntohs(ETH_P_IP)) {
           (ptmp->iptotal)++;

           linktype = getlinktype(fromaddr.spkt_family, fromaddr.spkt_device);

           adjustpacket(buf, linktype, &packet, aligned_buf, &br);

           if (packet == NULL)
              return;

           ipacket = (struct iphdr *) packet;
           iphlen = ipacket->ihl * 4;
           tpacket = packet + iphlen;

           /* recalculate checksum */                                                                                                                                      ipcsum = ipacket->check;
           ipacket->check = 0;
           ipck = in_cksum((u_short *) ipacket, iphlen);
           if ((ipcsum != ipck) || (!ipck)) {                                                  (ptmp->badtotal)++;
               return;                                                 
           }
          } else {
              (ptmp->noniptotal)++;                                                       }
          if (gencolumns.total)
            {
	      tempstr->setNum(ptmp->total);
              ptmp->listviewitem->setText(gencolumns.total,tempstr->data()); 
            }
          if (gencolumns.total_ip)
            {
	      tempstr->setNum(ptmp->iptotal);
              ptmp->listviewitem->setText(gencolumns.total_ip,tempstr->data()); 
            }
          if (gencolumns.non_ip)
            {
	      tempstr->setNum(ptmp->noniptotal);
              ptmp->listviewitem->setText(gencolumns.non_ip,tempstr->data()); 
            }
          if (gencolumns.bad_ip)
            {
	      tempstr->setNum(ptmp->badtotal);
              ptmp->listviewitem->setText(gencolumns.bad_ip,tempstr->data()); 
            }
   }
}


void
IfStatsGeneral::timerDone()
{
  struct iflist *ptmp = NULL;

  gettimeofday(&tv, NULL);
  now = tv.tv_sec;
  unow = tv.tv_sec * 1e+6 + tv.tv_usec;

  ptmp = table.head;
  
  while (ptmp!=NULL)    		//Make sure list isn't empty
    {
      if (options->actmode == KBITS) {
      ptmp->rate = ((float) (ptmp->spanbr * 8 / 1000)) / ((float) (now - ptmp->starttime));
      tempstr2->sprintf("%2.2f kbits/sec",ptmp->rate);
    } else {
      ptmp->rate = ((float) (ptmp->spanbr / 1024)) / ((float) (now - ptmp->starttime));
      tempstr2->sprintf("%2.2f kbytes/sec",ptmp->rate);
    }

    ptmp->listviewitem->setText(gencolumns.activity,tempstr2->data()); 
    ptmp->starttime = now;
    ptmp->spanbr = 0;

    ptmp = ptmp->next_entry;
    }
}



/********************************************
  initiflist(struct iflist **list):
    This grabs the known interfaces and 
    creates the linked list in memory,
    then adds the interface entries to 
    the user display.
*********************************************/
void 
IfStatsGeneral::initiflist(struct iflist **list)
{
    FILE *fd;
    char buf[161];
    char ifname[8];
    struct iflist *itmp = NULL;
    struct iflist *tail = NULL;
    unsigned int index = 0;

    *list = NULL;

    fd = fopen("/proc/net/dev", "r");
    if (fd == NULL) {
	return;
    }
    fgets(buf, 160, fd);	/* dummy header lines */
    fgets(buf, 160, fd);

    while (!(feof(fd))) {
	strcpy(buf, "");
	fgets(buf, 160, fd);
	if (strcmp(buf, "") != 0) {
	    strcpy(ifname, ltrim(strtok(buf, ":")));

	    if (!(iface_supported(ifname)))
		continue;

	    if (ifinlist(*list, ifname))	/* ignore entry if already in */
		continue;	/* interface list */


	    itmp = (struct iflist *)malloc(sizeof(struct iflist));
	    bzero(itmp, sizeof(struct iflist));
	    strcpy(itmp->ifname, ifname);
	    index++;
	    itmp->index = index;

	    if (*list == NULL) {
		*list = itmp;
		itmp->prev_entry = NULL;
	    } else {
		tail->next_entry = itmp;
		itmp->prev_entry = tail;
	    }

	    tail = itmp;
	    itmp->next_entry = NULL;
            switch (gencolumns.numcols) {
                case 1:
                  itmp->listviewitem = new QListViewItem(mylistview, ifname);
                  break;

                case 2:
                  itmp->listviewitem = new QListViewItem(mylistview, ifname, "0");
                  break;
                case 3:
                  itmp->listviewitem = new QListViewItem(mylistview, ifname, "0", "0");
                  break;
                case 4:
                  itmp->listviewitem = new QListViewItem(mylistview, ifname, "0", "0", "0");
                  break;

                case 5:
                  itmp->listviewitem = new QListViewItem(mylistview, ifname, "0", "0", "0", "0");
                  break;
  
               default:
                  break;
            }
	}
    }
    fclose(fd);
}


/*************************************
  Detailed Stats Class area
*************************************/

//Constructor for Detailed Interface stats
IfStatsDetailed::IfStatsDetailed(QWidget *parent, char *iface, struct OPTIONS *appoptions)
{

  starttime = 0;
  statbegin = 0;
  now = 0;
  unow = 0;
  startlog = 0;
  updtime = 0;
  updtime_usec = 0;
  spanbr = 0;
  spanpkt = 0;
  activity = 0;
  peakactivity = 0;
  pps = 0;
  peakpps = 0;

  rcvstr = new QString();
  timerstr = new QString();

  qstrcpy((char*) &ifname, iface);

  options = appoptions;

  detcolumns.numcols=1;
  detcolumns.pkts=0;
  detcolumns.bytes=0;
  detcolumns.activity=0;

  mylistview = new QListView(parent, "IfaceListView");
  mylistview->setGeometry(0,56,480,244);
  mylistview->addColumn(ifname,-1);
  mylistview->setColumnWidth(0,90);

  if (options->detopts.pkts)
    {
      mylistview->addColumn("Packets",-1); 
      detcolumns.pkts = 1;
      mylistview->setColumnWidth(detcolumns.pkts,100);
      detcolumns.numcols++;
    }
  if (options->detopts.bytes)
    {
      mylistview->addColumn("Bytes",-1);
      detcolumns.bytes = detcolumns.numcols;
      mylistview->setColumnWidth(detcolumns.bytes,100);
      detcolumns.numcols++;
    }
  if (options->detopts.activity)
    {
      mylistview->addColumn("Activity",-1);
      detcolumns.activity = detcolumns.numcols;
      mylistview->setColumnWidth(detcolumns.activity,110);
    }
  
  mylistview->setColumnWidthMode(1,QListView::Maximum);
  mylistview->setAllColumnsShowFocus(1);
  mylistview->setSorting(-1, FALSE);
  mylistview->show();

  switch (detcolumns.numcols) {
      case 1:
        ipxitem = new QListViewItem(mylistview, "IPX:");
        rarpitem = new QListViewItem(mylistview, "RARP:");
        arpitem = new QListViewItem(mylistview, "ARP:");
        nonipitem = new QListViewItem(mylistview, "NON-IP:");
        otheripitem = new QListViewItem(mylistview, "OTHER-IP:");
        icmpitem = new QListViewItem(mylistview, "ICMP:");
        udpitem = new QListViewItem(mylistview, "UDP:");
        tcpitem = new QListViewItem(mylistview, "TCP:");
        ipitem = new QListViewItem(mylistview, "IP:");
        totalsitem = new QListViewItem(mylistview, "Total:");
        break;

      case 2:
        ipxitem = new QListViewItem(mylistview, "IPX:", "0");
        rarpitem = new QListViewItem(mylistview, "RARP:", "0");
        arpitem = new QListViewItem(mylistview, "ARP:", "0");
        nonipitem = new QListViewItem(mylistview, "NON-IP:", "0");
        otheripitem = new QListViewItem(mylistview, "OTHER-IP:", "0");
        icmpitem = new QListViewItem(mylistview, "ICMP:", "0");
        udpitem = new QListViewItem(mylistview, "UDP:", "0");
        tcpitem = new QListViewItem(mylistview, "TCP:", "0");
        ipitem = new QListViewItem(mylistview, "IP:", "0");
        totalsitem = new QListViewItem(mylistview, "Total:", "0");
        break;

      case 3:
        ipxitem = new QListViewItem(mylistview, "IPX:", "0", "0");
        rarpitem = new QListViewItem(mylistview, "RARP:", "0", "0");
        arpitem = new QListViewItem(mylistview, "ARP:", "0", "0");
        nonipitem = new QListViewItem(mylistview, "NON-IP:", "0", "0");
        otheripitem = new QListViewItem(mylistview, "OTHER-IP:", "0", "0");
        icmpitem = new QListViewItem(mylistview, "ICMP:", "0", "0");
        udpitem = new QListViewItem(mylistview, "UDP:", "0", "0");
        tcpitem = new QListViewItem(mylistview, "TCP:", "0", "0");
        ipitem = new QListViewItem(mylistview, "IP:", "0", "0");
        totalsitem = new QListViewItem(mylistview, "Total:", "0", "0");
        break;

      case 4:
        ipxitem = new QListViewItem(mylistview, "IPX:", "0", "0", "");
        rarpitem = new QListViewItem(mylistview, "RARP:", "0", "0", "");
        arpitem = new QListViewItem(mylistview, "ARP:", "0", "0", "");
        nonipitem = new QListViewItem(mylistview, "NON-IP:", "0", "0", "");
        otheripitem = new QListViewItem(mylistview, "OTHER-IP:", "0", "0", "");
        icmpitem = new QListViewItem(mylistview, "ICMP:", "0", "0", "");
        udpitem = new QListViewItem(mylistview, "UDP:", "0", "0", "");
        tcpitem = new QListViewItem(mylistview, "TCP:", "0", "0", "");
        ipitem = new QListViewItem(mylistview, "IP:", "0", "0", "");
        totalsitem = new QListViewItem(mylistview, "Total:", "0", "0", "0");
        break;

      default:
        break;
  }

  if (!iface_supported(ifname)) {
      err_iface_unsupported();
      return;
    }

  bzero(&totals, sizeof(struct iftotals));

  if (initialize_brackets(ifname, &totals) != 0) {
     close(fd);
     return;
  }

  open_socket(&fd);

  if (fd < 0)
    return;

  gettimeofday(&tv, NULL);
  starttime = startlog = statbegin = tv.tv_sec;

  mycallback = new QSocketNotifier(fd, QSocketNotifier::Read);
  connect(mycallback, SIGNAL(activated(int)), SLOT(dataReceived(int)));

  if (options->detopts.activity)
    {
      mytimer = new QTimer(this, "General Stats Timer");
      connect( mytimer, SIGNAL(timeout()), SLOT(timerDone()) );
      mytimer->start( 1000, FALSE );                 // 2 seconds single-shot
    }
}

IfStatsDetailed::~IfStatsDetailed()
{
  if (options->detopts.activity)
    {
      mytimer->stop();
      delete(mytimer);
    }
  delete(mycallback);
  delete (mylistview);
  delete (rcvstr);
//  close(fd);
}

void
IfStatsDetailed::updateOptions(struct OPTIONS *appoptions)
{
// options.promisc = appoptions->promisc;
// options.actmode = appoptions->actmode;
}


//This actually processes our packet
void IfStatsDetailed::dataReceived(int fd)
{
  char buf[8192];
  struct sockaddr_pkt fromaddr;
  int ch;
  int br;
  unsigned short linktype;
  char *packet;
  char aligned_buf[120];
  struct iphdr *ipacket;
  char *tpacket;
  int iphlen;
  int ipck, ipcsum;
  unsigned int iplen;
  int framelen = 0;


  getpacket(fd, buf, &fromaddr, &ch, &br);

  if (br > 0) {
    if (strcmp(ifname, fromaddr.spkt_device) != 0)
	return;

      framelen = br;
      totals.total++;
      totals.bytestotal += br;

      spanbr += br;
      spanpkt++;

     if (fromaddr.spkt_protocol == ntohs(ETH_P_IP)) {
       totals.iptotal++;
		
       linktype = getlinktype(fromaddr.spkt_family, fromaddr.spkt_device);
		
       adjustpacket(buf, linktype, &packet, aligned_buf, &br);

       if (packet == NULL)
         return;

       ipacket = (struct iphdr *) packet;
       iphlen = ipacket->ihl * 4;
       tpacket = packet + iphlen;

       ipcsum = ipacket->check;
       ipacket->check = 0;
       ipck = in_cksum((u_short *) ipacket, iphlen);
       if ((ipcsum != ipck) || (!ipck)) {
         totals.badtotal++;
         return;
       }

       iplen = ntohs(ipacket->tot_len);
       totals.ipbtotal += iplen;
       
       if (detcolumns.pkts)
         {
           rcvstr->setNum(totals.iptotal);
           ipitem->setText(detcolumns.pkts,rcvstr->data()); 
         }
       if (detcolumns.bytes)
         {
           rcvstr->setNum(totals.ipbtotal);
           ipitem->setText(detcolumns.bytes,rcvstr->data()); 
         }

       switch (ipacket->protocol) {
          case IPPROTO_TCP:
            totals.tcptotal++;
            totals.tcpbtotal += iplen;

            if (detcolumns.pkts)
              {
                rcvstr->setNum(totals.tcptotal);
                tcpitem->setText(detcolumns.pkts,rcvstr->data()); 
              }
            if (detcolumns.bytes)
              {
                rcvstr->setNum(totals.tcpbtotal);
                tcpitem->setText(detcolumns.bytes,rcvstr->data()); 
              }
            break;
          case IPPROTO_UDP:
            totals.udptotal++;
            totals.udpbtotal += iplen;

            if (detcolumns.pkts)
              {
                rcvstr->setNum(totals.udptotal);
                udpitem->setText(detcolumns.pkts,rcvstr->data()); 
              }
            if (detcolumns.bytes)
              {
                rcvstr->setNum(totals.udpbtotal);
                udpitem->setText(detcolumns.bytes,rcvstr->data()); 
              }
            break;
          case IPPROTO_ICMP:
            totals.icmptotal++;
            totals.icmpbtotal += iplen;

            if (detcolumns.pkts)
              {
                rcvstr->setNum(totals.icmptotal);
                icmpitem->setText(detcolumns.pkts,rcvstr->data()); 
              }
            if (detcolumns.bytes)
              {
                rcvstr->setNum(totals.icmpbtotal);
                icmpitem->setText(detcolumns.bytes,rcvstr->data()); 
              }
	    break;
          default:
            totals.othtotal++;
            totals.othbtotal += iplen;

            if (detcolumns.pkts)
              {
                rcvstr->setNum(totals.othtotal);
                otheripitem->setText(detcolumns.pkts,rcvstr->data()); 
              }
            if (detcolumns.bytes)
              {
                rcvstr->setNum(totals.othbtotal);
                otheripitem->setText(detcolumns.bytes,rcvstr->data()); 
              }
            break;
       }
     } 
     else if (fromaddr.spkt_protocol == ntohs(ETH_P_ARP)) 
     {
       totals.arptotal++;
       totals.arpbtotal += iplen;

       if (detcolumns.pkts)
         {
           rcvstr->setNum(totals.arptotal);
           arpitem->setText(detcolumns.pkts,rcvstr->data()); 
         }
       if (detcolumns.bytes)
         {
           rcvstr->setNum(totals.arpbtotal);
           arpitem->setText(detcolumns.bytes,rcvstr->data()); 
         }
     }
     else if (fromaddr.spkt_protocol == ntohs(ETH_P_RARP))
     {
       totals.rarptotal++;
       totals.rarpbtotal += iplen;

       if (detcolumns.pkts)
         {
           rcvstr->setNum(totals.rarptotal);
           rarpitem->setText(detcolumns.pkts,rcvstr->data()); 
         }
       if (detcolumns.bytes)
         {
           rcvstr->setNum(totals.rarpbtotal);
           rarpitem->setText(detcolumns.bytes,rcvstr->data()); 
         }
     }
     else if (fromaddr.spkt_protocol == ntohs(ETH_P_IPX))
     {
       totals.ipxtotal++;
       totals.ipxbtotal += iplen;

       if (detcolumns.pkts)
         {
           rcvstr->setNum(totals.ipxtotal);
           ipxitem->setText(detcolumns.pkts,rcvstr->data()); 
         }
       if (detcolumns.bytes)
         {
           rcvstr->setNum(totals.ipxbtotal);
           ipxitem->setText(detcolumns.bytes,rcvstr->data()); 
         }
     }
     else {
       totals.noniptotal++;
       totals.nonipbtotal += br;

       if (detcolumns.pkts)
         {
           rcvstr->setNum(totals.noniptotal);
           nonipitem->setText(detcolumns.pkts,rcvstr->data()); 
         }
       if (detcolumns.bytes)
         {
           rcvstr->setNum(totals.nonipbtotal);
           nonipitem->setText(detcolumns.bytes,rcvstr->data()); 
         }
     }

       if (detcolumns.pkts)
         {
           rcvstr->setNum(totals.total);
           totalsitem->setText(detcolumns.pkts,rcvstr->data()); 
         }
       if (detcolumns.bytes)
         {
           rcvstr->setNum(totals.bytestotal);
           totalsitem->setText(detcolumns.bytes,rcvstr->data()); 
         }
   }
}

void
IfStatsDetailed::timerDone()
{

  gettimeofday(&tv, NULL);
  now = tv.tv_sec;
  unow = tv.tv_sec * 1e+6 + tv.tv_usec;

  if (now - starttime >= 1) {
    if (options->actmode == KBITS)
      {
        activity = (float) (spanbr * 8 / 1000) / (float) (now - starttime);
        timerstr->sprintf("%2.2f kbits/sec",activity);
      }
    else
      {
         activity = (float) (spanbr / 1024) / (float) (now - starttime);
         timerstr->sprintf("%2.2f kbytes/sec",activity);
      }

    totalsitem->setText(detcolumns.activity,timerstr->data()); 
	    
    pps = (float) (spanpkt) / (float) (now - starttime);
    spanbr = 0;
    spanpkt = 0;
    starttime = now;

    timerstr->sprintf("%2.2f pkts/sec",pps);
    ipitem->setText(detcolumns.activity,timerstr->data()); 

    if (activity > peakactivity)
      peakactivity = activity;

    if (pps > peakpps)
      peakpps = pps;
 }
}



/****************************************
  General purpose Routines for stats
****************************************/

int 
IfStatsDetailed::initialize_brackets(char *ifname, struct iftotals *ttmp)
{
    struct ifreq ifr;
    int fd;
    int istat;
    unsigned int interval;
    int i;
    
    strcpy(ifr.ifr_name, ifname);
    
    fd = socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL));
    
    if (fd < 0) {
        //errbox("Unable to open socket for MTU determination", ANYKEY_MSG, &resp);
       {
         KMsgBox *mybox;
         mybox = new KMsgBox(this, "Error", "Unable to open Socket for MTU Determination.  You should run Ksniffer as root", 2,"ok",0,0,0);
         mybox->show();
         delete(mybox);
        }
        return 1;
    }
    
    istat = ioctl(fd, SIOCGIFMTU, &ifr);
    
//    close(fd);
    if (istat < 0) {
        //errbox("Unable to obtain interface MTU", ANYKEY_MSG, &resp);
   fprintf(stderr, "Unable to obtain interface MTU\n");
       {
         KMsgBox *mybox;
         mybox = new KMsgBox(this, "Error", "Unable to obtain interface's MTU.  You should run Ksniffer as root", 2,"ok",0,0,0);
         mybox->show();
         delete(mybox);
        }
	return 1;
    }
            
    ttmp->interval = interval = ifr.ifr_mtu / 16;    /* There are 16 packet size brackets */
    
    for (i = 0; i <= 15; i++) {
        ttmp->brackets[i].floor = interval * i + 1;
        ttmp->brackets[i].ceil = interval * (i + 1);
    }
    
    ttmp->brackets[15].ceil = ifr.ifr_mtu;
    return 0;
}


void positionptr(struct iftab *table, struct iflist **ptmp, char *ifname)
{
    struct iflist *plast = NULL;
    int ok = 0;

    *ptmp = table->head;

    while ((*ptmp != NULL) && (!ok)) {
	ok = (strcmp((*ptmp)->ifname, ifname) == 0);

	if (!ok) {
	    if ((*ptmp)->next_entry == NULL)
		plast = *ptmp;

	    *ptmp = (*ptmp)->next_entry;
	}
    }

    if (*ptmp == NULL) {
	*ptmp = (struct iflist *)malloc(sizeof(struct iflist));
	bzero(*ptmp, sizeof(struct iflist));
	(*ptmp)->index = plast->index + 1;
	plast->next_entry = *ptmp;
	(*ptmp)->prev_entry = plast;
	(*ptmp)->next_entry = NULL;
	strcpy((*ptmp)->ifname, ifname);
	
	/*if ((*ptmp)->index <= LINES - 4)
	   table->lastvisible = *ptmp; */
    }
}

void destroyiflist(struct iflist *list)
{
    struct iflist *ctmp;
    struct iflist *ptmp;

    if (list != NULL) {
	ptmp = list;
	ctmp = ptmp->next_entry;

	do {
	    free(ptmp);
	    ptmp = ctmp;
	    if (ctmp != NULL)
		ctmp = ctmp->next_entry;
	} while (ptmp != NULL);
    }
}

char *ltrim(char *buf)
{
    char *tmp = buf;

    while ((*tmp == ' ') || (*tmp == '\t'))
	tmp++;

    strcpy(buf, tmp);
    return buf;
}

/*
 * Function to check if an interface is already in the interface list.
 * This eliminates duplicate interface entries due to aliases
 */

int ifinlist(struct iflist *list, char *ifname)
{
    struct iflist *ptmp = list;
    int result = 0;

    while ((ptmp != NULL) && (result == 0)) {
	result = (strcmp(ifname, ptmp->ifname) == 0);
	ptmp = ptmp->next_entry;
    }

    return result;
}
