#include "smb.h"
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <stdlib.h>
#include "nmbhdr.h"
#include "protmgr.h"
#include <kapp.h>

bool SMBClientInterface(const char *_cmd, const char *word, SMBManager *obj)
{
	QString		tempFile, cmd;
	tempFile.sprintf("/tmp/SMB%d",getpid());
	cmd.sprintf("%s > %s",_cmd,tempFile.data());
	int	result(0);
cout << cmd.data() << endl;
	if (system(cmd.data()) != 0) result = 1;
	else {
		QFile	f(tempFile.data());
		// opening result file
		if (!f.open(IO_ReadOnly)) result = 1;
		else {
			QTextStream	t(&f);
			QString		Buffer, Word;
			bool		ok = false;
			// search for line beginning with "Server"
			while (!ok && !t.eof()) {
				Buffer = t.readLine();
				QTextStream	tstr(Buffer,IO_ReadOnly);
				tstr >> Word;
				if (Word == word) ok = true;
			}
			t.readLine();	// skip next line
			ok = false;
			while (!ok && !t.eof()) {
				Buffer = t.readLine().stripWhiteSpace();
				if (Buffer.isEmpty()) {
					ok = true;
					continue;
				}
				obj->processBuffer(Buffer);
			}
			if (t.eof()) result = 1;
		}
	}
	::remove(tempFile.data());
	return (result == 0);
}

QString processForShell(const QString& str)
{
// process a string for use in shell command, insert a backslash before any special character
	QString	out = str.data();
	const int	n = 8;
	QString	str1[n] = {" ","!","<",">","*","\\$","(",")"};
	QString	str2[n] = {"\\ ","\\!","\\<","\\>","\\*","\\$","\\(","\\)"};
	for (int i=0;i<n;i++) out.replace(QRegExp(str1[i].data()),str2[i].data());
	return out;
}

//----------------------------------------------------------------------------------------------------

SMBManager::SMBManager()
	: Protocol(Protocol::SmbMgr)
{
	Prefix = "smb:";
	GroupDict.setAutoDelete(true);
	MachineList = new QList<SMBMachine>;
	MachineList->setAutoDelete(false);
	UpdateFlag = true;
	CurrentGroup = "";
}

SMBManager::~SMBManager()
{
	delete MachineList;
}

bool SMBManager::matchURL(KURL url)
{
	if (strcmp(url.protocol(),"smb") == 0) return true;
	return false;
}

bool SMBManager::setPath(const char *pathname)
{
	CurrentGroup = pathname;
	CurrentGroup.replace(QRegExp("/"),"");
	return true;
}

const FileInfoList* SMBManager::entryInfoList(bool hidden, bool dirsOnly, bool showArchive)
{
	if (UpdateFlag) {
		sendMessage("Scanning network...");
		if (!getMachineList()) return 0;
		translateMachineList();
	}
	return GroupDict.find(CurrentGroup.isEmpty() ? "__GROUPDICT__" : CurrentGroup.data());
}

bool SMBManager::cleanup()
{
	UpdateFlag = true;
	return true;
}

bool SMBManager::getMachineList()
{
	// clear MachineList
	while (!MachineList->isEmpty()) {
		MachineList->first()->cleanProtocols();
		((ProtocolMgr*)Manager)->deleteProtocol(MachineList->first());
		MachineList->removeFirst();
	}

	// first retrieve IP list of connected computers
	QStrList	IPList, BCIPList;
	bool		result = true;
	IPList.setAutoDelete(true);
	BCIPList.setAutoDelete(true);
	kapp->getConfig()->setGroup("Configuration");
	// first use broadcast IP's
	kapp->getConfig()->readListEntry("BroadCastIP",BCIPList);
	if (BCIPList.isEmpty()) result = getIPList(IPList);
	else {
		QStrListIterator	sit(BCIPList);
		for (;sit.current() && result;++sit) result = (result && getIPList(IPList,sit.current()));
	}
	if (!result) return false;
	// Then use computer IP's
	BCIPList.clear();
	kapp->getConfig()->readListEntry("ComputerIP",BCIPList);
	for (BCIPList.first();BCIPList.current();BCIPList.next()) IPList.append(BCIPList.current());

	// get node info for each found IP. Timeout is 0.1s !
	QStrListIterator	it(IPList);
	for (;it.current();++it) {
		QString		Name(16), Work(16);
		unsigned long	IP = inet_addr(it.current());
		if (IP == INADDR_NONE) continue;	// wrong IP
		getNMBInfo(IP,Name,Work);
		cout << it.current() << "\t" << Name << "\t" << Work << endl;
		if (!Name.isEmpty()) {	// create new Machine instance
			MachineList->append((SMBMachine*)(((ProtocolMgr*)Manager)->createProtocol(Protocol::SmbMach,Name.data(),Work.data(),it.current())));
		}
	}

	// output
	UpdateFlag = false;
	return true;
}

void SMBManager::translateMachineList()
{
	// clear all entries
	GroupDict.clear();
	FileInfoList	*groupList = new FileInfoList;
	groupList->setAutoDelete(true);
	GroupDict.insert("__GROUPDICT__",groupList);

	// reconstruct all the entries
	QString		str("drwxrwxrwx 1 root root 0 Jan 1 00:00:00 "), str2;
	QListIterator<SMBMachine>	it(*MachineList);
	for (;it.current();++it) {
		// craete the appropriate entry
		str2 = str + it.current()->machine();
		FileInfo	*fi = new FileInfo(str2.data(),6);
		fi->setDescription(i18n("SMB Machine"));
		str2.sprintf("smb://%s/",it.current()->machine());
		fi->setPath(str2.data());
		fi->setIconName("mini-term.xpm");
		// find the group where to add it
		FileInfoList	*list = GroupDict.find(it.current()->workgroup());
		if (!list) {	// we don't have such a group, just create it
			str2 = str + it.current()->workgroup();
			FileInfo	*gfi = new FileInfo(str2.data(),6);
			gfi->setDescription(i18n("SMB Domain"));
			str2.sprintf("smb:/%s/",it.current()->workgroup());
			gfi->setPath(str2.data());
			gfi->setIconName("group.xpm");
			// add this new group to the group list
			GroupDict.find("__GROUPDICT__")->append(gfi);
			// create a new entry list and add it to the GroupDict
			list = new FileInfoList;
			list->setAutoDelete(true);
			GroupDict.insert(it.current()->workgroup(),list);
		}
		list->append(fi);
	}
}

void SMBManager::getNMBInfo(ulong ip, QString& machine, QString& workgroup)
{
  struct sockaddr_in sin_dst;
  struct nmbhdr *nmb;
  char *data;
  struct typez *typz;
  char buffer[1024];
  int socket_client;
  int longueur = sizeof(struct sockaddr_in);
  int count;
  struct _nameinfo {
    char name[15];
    char type;
    char res;
    char fill;
  } *ninf;
  struct timeval  timeout = {0, 100000};
  fd_set rfds;
  bool canread = true;


  bzero(buffer, sizeof(buffer));
  nmb = (struct nmbhdr *) buffer;
  data = (char *) (buffer + NMBHDRSIZE);
  typz = (struct typez *) (buffer + NMBHDRSIZE + 33);

  sin_dst.sin_family = AF_INET;
  sin_dst.sin_port = htons(137);
  sin_dst.sin_addr.s_addr = ip;

  memcpy(data, "CKAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", 34);

  nmb->id = 0x600;
  nmb->R = 0;			/* 0 for question 1 for response */
  nmb->opcode = 0;		/* 0 = query */
  nmb->que_num = htons(1);	/* i have only 1 question :) */
  nmb->namelen = 0x20;
  typz->type = 0x2100;
  typz->type2 = 0x0100;

  socket_client = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

  sendto(socket_client, buffer, 50, 0, (struct sockaddr *) &sin_dst, longueur);
  FD_ZERO(&rfds);
  FD_SET(socket_client,&rfds);
  if (select(socket_client+1,&rfds,0,0,&timeout) < 0 || !FD_ISSET(socket_client,&rfds))
    // there nothing to read, timeout reached.
    canread = false;

  bzero(buffer, sizeof(buffer));
  nmb = (struct nmbhdr *) (buffer);

  if (canread) {
    if (recvfrom(socket_client, buffer, sizeof(buffer), 0, (struct sockaddr *) &sin_dst, (socklen_t*)&longueur) != -1) {
      if (nmb->rep_num != 0) {
	/*set to first nameblock after header */
	count = 0;
	ninf = (struct _nameinfo *) (buffer + 57 + (count++ * sizeof(struct _nameinfo)));
	while (machine.isEmpty() && (57 + (count * sizeof(struct _nameinfo)) < 1024)) {
	  if (ninf->type == 0) {
	    QString	tmp(ninf->name,15);
	    machine = tmp.data();
	  }
	  ninf = (struct _nameinfo *) (buffer + 57 + (count++ * sizeof(struct _nameinfo)));
	}
	if (machine.isEmpty()) {
	  close(socket_client);
	  return;
	}
	while (workgroup.isEmpty() && (57 + (count * sizeof(struct _nameinfo)) < 1024)) {
	  if (ninf->type == 0) {
	    QString	tmp(ninf->name,15);
	    workgroup = tmp.data();
	  }
	  ninf = (struct _nameinfo *) (buffer + 57 + (count++ * sizeof(struct _nameinfo)));
	}
      }
    }
  }
  close(socket_client);
  return;
}

void SMBManager::processBuffer(const QString& Buffer)
{
	QTextStream	tstr(Buffer,IO_ReadOnly);
	char		buf[15];
	tstr.readRawBytes(buf,15);
	QString		NameStr(buf,15);
	NameStr = NameStr.stripWhiteSpace();
	MachineList->append((SMBMachine*)(((ProtocolMgr*)Manager)->createProtocol(Protocol::SmbMach,NameStr.data(),"","")));
}

/*bool SMBManager::getMachineListFromWINS()
{
	QString		cmd;
	kapp->getConfig()->setGroup("SMB Config");
	cmd.sprintf("smbclient -L %s -N -I %s",kapp->getConfig()->readEntry("WINS_Server","UNKNOWN").data(),kapp->getConfig()->readEntry("WINS_IP","127.0.0.1").data());
	if (!SMBClientInterface(cmd.data(),"Server",this)) {
		ErrorMsg = i18n("Unable to retrieve machine list from WINS.");
		return false;
	}
	return true;
}

bool SMBManager::getMachineListFromRange()
{
	// retrieve configuration
	kapp->getConfig()->setGroup("SMB Config");
	QStrList	start, stop;
	start.setAutoDelete(true);
	stop.setAutoDelete(true);
	kapp->getConfig()->readListEntry("IPStart",start);
	kapp->getConfig()->readListEntry("IPStop",stop);
	if (start.isEmpty() || stop.isEmpty()) {
		ErrorMsg = i18n("Unable to retrieve machine list from IP range.");
		return false;
	}
	for (start.first(), stop.first();start.current() && stop.current();start.next(), stop.next()) {
		QString	ip1(start.current()), ip2(stop.current());
		inet_aton(ip1,(struct in_addr*)(&IPStart));
		inet_aton(ip2,(struct in_addr*)(&IPStop));
		ulong	step;
		inet_aton("0.0.0.1",(struct in_addr*)(&step));

		// retrieve info for each IP
		for (ulong ip=IPStart;ip<=IPStop;ip+=step) {
			QString		IP = inet_ntoa(*((struct in_addr*)&ip));
			QString		Name(16), Workgroup(16);
			getNMBInfo(ip,Name,Workgroup);
			cout << IP << "\t" << Name << "\t" << Workgroup << endl;
			if (!Name.isEmpty()) {	// create new Machine instance
				MachineList->append((SMBMachine*)(((ProtocolMgr*)Manager)->createProtocol(Protocol::SmbMach,Name.data(),Workgroup.data(),IP.data())));
			}
		}
	}
	return true;
}*/

bool SMBManager::getIPList(QStrList& list, const char *broadcastIP)
{
	QString		cmd("nmblookup "), tempFile;
	bool		result = true;
	tempFile.sprintf("/tmp/SMB%d",getpid());
	if (broadcastIP) {
		cmd += "-B ";
		cmd += broadcastIP;
		cmd += " ";
	}
	cmd += ("\\* | grep \\<00\\> | sed 's/\\*<00>//' > " + tempFile);
	if (system(cmd.data()) != 0) {
		ErrorMsg = i18n("Can't retrieve IP list. Check that \"nmblookup\" is in your path\nor that you entered valid broadcast IP's");
		result = false;
	}
	else {
		QFile	f(tempFile.data());
		if (!f.open(IO_ReadOnly)) {
			ErrorMsg = i18n("I/O error");
			result = false;
		}
		else {
			list.setAutoDelete(true);
			QString	Buffer;
			QTextStream	t(&f);
			while (!t.eof()) {
				t >> Buffer;
				if (list.find(Buffer.data()) == -1) list.append(Buffer.data());
			}
		}
		f.close();
	}
	::remove(tempFile.data());
	return result;
}

//----------------------------------------------------------------------------------------------------

SMBMachine::SMBMachine(const char *name, const char *work, const char *ip)
	: SMBManager()
{
	Type = Protocol::SmbMach;
	Name = QString(name).stripWhiteSpace();
	Workgroup = QString(work).stripWhiteSpace();
	IP = ip;
	Entries = new FileInfoList;
	Entries->setAutoDelete(true);
	FilteredEntries = new FileInfoList;
	FilteredEntries->setAutoDelete(false);
	ShareList = new QList<SMBShare>;
	ShareList->setAutoDelete(false);
	Prefix = "smb://" + Name + "/";
	UpdateFlag = true;
}

SMBMachine::~SMBMachine()
{
	delete Entries;
}

bool SMBMachine::matchURL(KURL url)
{
	if (strcmp(url.protocol(),"smb") == 0 && url.host() == Name) return true;
	return false;
}

const FileInfoList* SMBMachine::entryInfoList(bool hidden, bool dirsOnly, bool showArchive)
{
	if (UpdateFlag) {
		sendMessage("Scanning shares...");
		if (!getShareList()) return 0;
		translateShareList();
	}
	filterEntries(dirsOnly);
	return FilteredEntries;
}

bool SMBMachine::cleanup()
{
	UpdateFlag = true;
	return true;
}

void SMBMachine::cleanProtocols()
{
	while (!ShareList->isEmpty()) {
		((ProtocolMgr*)Manager)->deleteProtocol(ShareList->first());
		ShareList->removeFirst();
	}

}

void SMBMachine::processBuffer(const QString& Buffer)
{
	QTextStream	tstr(Buffer,IO_ReadOnly);
	QString		TypeStr, CommentStr;
	char		buf[15];
	tstr.readRawBytes(buf,15);
	QString		NameStr(buf,15);
	NameStr = NameStr.stripWhiteSpace();
	tstr >> TypeStr >> CommentStr;
	QString		MountPoint, Service;
	MountPoint.sprintf("%s/share/apps/kruiser/mnt/%s_%s/",KApplication::localkdedir().data(),Name.data(),NameStr.data());
	MountPoint.replace(QRegExp(" "),"_");
	Service.sprintf("//%s(%s)/%s",Name.data(),Workgroup.data(),NameStr.data());
	ShareList->append((SMBShare*)(((ProtocolMgr*)Manager)->createProtocol(Protocol::SmbShare,MountPoint.data(),Service.data(),User.data(),Passwd.data(),IP.data())));
	ShareList->last()->setShareInfo(NameStr.data(),TypeStr.data());
}

bool SMBMachine::getShareList()
{
	// clear ShareList
	cleanProtocols();

	// use "smbclient" to list all shares
	QString		cmd;
	cmd.sprintf("smbclient -L %s -W %s",processForShell(Name).data(),processForShell(Workgroup).data());
	if (User.isEmpty()) cmd += " -N";
	else cmd += (" -U " + processForShell(User) + "%" + processForShell(Passwd));
	if (!IP.isEmpty()) cmd += (" -I " + IP);
	if (!SMBClientInterface(cmd.data(),"Sharename",this)) {
		ErrorMsg = i18n("Unable to retrieve share list");
		User = "";
		Passwd = "";
		return false;
	}
	UpdateFlag = false;
	return true;
}

void SMBMachine::translateShareList()
{
	// clear all entries
	Entries->clear();

	// reconstruct entries
	kapp->getConfig()->setGroup("Configuration");
	bool	showHidden(kapp->getConfig()->readBoolEntry("ShowHiddenShares",false));
	QListIterator<SMBShare>	it(*ShareList);
	for (;it.current();++it) {
		QString		str(it.current()->shareName());
		if (str.right(1) == "$" && !showHidden) continue;
		if (strcmp(it.current()->typestr(),"Disk") == 0) str = "d";
		else if (strcmp(it.current()->typestr(),"Printer") == 0) str = "-";
		else continue;
		str += "rwxrwxrwx 1 root root 0 Jan 1 00:00:00 ";
		str += it.current()->shareName();
		FileInfo	*fi = new FileInfo(str.data(),6);
		fi->setDescription(i18n("SMB Share Device"));
		str.sprintf("file:%s",it.current()->mountp());
		fi->setPath(str.data());
		fi->setIconName(fi->isDir() ? "harddrive_unmount.xpm" : "printer.xpm");
		Entries->append(fi);
	}
}

void SMBMachine::filterEntries(bool dirsOnly)
{
	FilteredEntries->clear();
	FileInfoListIterator	it(*Entries);
	for (;it.current();++it)
		if (it.current()->isDir() || !dirsOnly) FilteredEntries->append(it.current());
}

//----------------------------------------------------------------------------------------------------

SMBShare::SMBShare(const char *mpoint, const char *service, const char *user, const char *passwd, const char *ip)
	: SmbProtocol(mpoint,service,user,passwd,ip)
{
	Type = Protocol::SmbShare;
	::mkdir(mpoint,0700);
}

SMBShare::~SMBShare()
{
	if (umount()) ::rmdir(mountp());
}

const FileInfoList* SMBShare::entryInfoList(bool hidden, bool dirsOnly, bool showArchive)
{
	if (!isMounted() && !mount()) {
		ErrorMsg = i18n("Unable to mount SMB share");
		return 0;
	}
	return FileProtocol::entryInfoList(hidden,dirsOnly,showArchive);
}

bool SMBShare::setPath(const char *pathname)
{
	if (!isMounted() && !mount()) {
		ErrorMsg = i18n("Unable to mount SMB share");
		return false;
	}
	return FileProtocol::setPath(pathname);
}

bool SMBShare::matchURL(KURL url)
{
  if (strcmp(url.protocol(),"smb") != 0) return FileProtocol::matchURL(url);
  QString str = url.url().mid(4,url.url().length());
  if (str.find(Service.data()) != -1) return true;
  else return false;
}
