#include <config.h>
#include <iostream.h>
#include <qmessagebox.h>
#include <qdragobject.h>
#include <qapplication.h>
#include <qkeycode.h>
#include <strings.h>
#include "dirview.h"
#include <kurl.h>

QString createPath(int n,...)
{
	QString	str;
	va_list	args;
	char*	c;
	va_start(args,n);
	for (int i=0;i<n;i++)
	{
		c = va_arg(args,char*);
		str.append(c);
		str.append("/");
	}
	va_end(args);
	return str;
}

//-------------------------------------------------------------------------------------------------------

DirViewToolTip::DirViewToolTip(QWidget *parent)
	: QToolTip(parent)
{}

void DirViewToolTip::maybeTip(const QPoint& pos)
{
	QString	tipStr;
	QRect	r = ((DirView*)parentWidget())->tip(pos,tipStr);
	if (r.isValid()) {
		tip(r,tipStr.data());
	}
}

//-------------------------------------------------------------------------------------------------------

DirViewItem::DirViewItem(DirViewItem *parent, const char *label, const char *addr, QPixmap *icon, QPixmap *iconSel, bool hasChilds, int type)
	: Label(label), Address(addr), ParentItem(parent), Opened(FALSE), Icon(icon), IconSel(iconSel), ChildsScanned(FALSE), HasChilds(hasChilds), Type(type)
{
	Childs = new DirViewItemList;
	Childs->setAutoDelete(TRUE);
	if (ParentItem) ParentItem->addChildItem(this);
	else {
		Depth = 0;
		ParentDirView = 0;
	}
	Index = 0;
	Sorting = FALSE;
}

DirViewItem::~DirViewItem()
{ if (ParentDirView) ParentDirView->removeItem(this); delete Childs;}

void DirViewItem::clear()
{ Childs->clear();}

void DirViewItem::addChildItem(DirViewItem *item)
{
	if (Sorting && Childs->count() > 0) Childs->inSort(item);
	else Childs->append(item);
	item->ParentItem = this;
	item->Depth = Depth+1;
	item->ParentDirView = ParentDirView;
	if (ParentDirView) ParentDirView->addItem(item);
}

void DirViewItem::removeChildItem(DirViewItem *item)
{
	Childs->removeRef(item);
}

DirViewItem* DirViewItem::getFirstParent()	// on depth = 1
{
	if (Depth < 1) return 0;
	DirViewItem	*current = this;
	while (current->ParentItem->ParentItem) current = current->ParentItem;
	return current;
}

QList<DirViewItem> DirViewItem::getParentList()
{
	QList<DirViewItem>	list;
	list.setAutoDelete(FALSE);
	DirViewItem	*current = this;
	while (current) {
		list.insert(0,current);
		current = current->ParentItem;
	}
	return list;
}

QString DirViewItem::getPath()
{
	DirViewItem	*item = this;
	QString		result(Label + "/");
	while (item->ParentItem) {
		item = item->ParentItem;
		result.insert(0,'/');
		result.insert(0,item->Label.data());
	}
	return result;
}

DirViewItem* DirViewItem::findChild(const char *name)
{
	if (Childs->count() < 1) return 0;
	DirViewItem	*item = Childs->first();
	while (item) {
		if (item->Label == name) break;
		item = Childs->next();
	}
	return item;
}

DirViewItem* DirViewItem::getNext()
{
	if (ParentItem) {
		ParentItem->Childs->findRef(this);
		return ParentItem->Childs->next();
	}
	else return 0;
}

//-------------------------------------------------------------------------------------------------------

DirView::DirView(QWidget *parent, const char *name)
	: QTableView(parent,name), QDropSite(this)
{
	setTableFlags(Tbl_autoHScrollBar | Tbl_autoVScrollBar | Tbl_smoothScrolling | Tbl_clipCellPainting);
	setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
//	setAutoUpdate(true);
	setBackgroundMode(PaletteBase);
	setFocusPolicy(QWidget::StrongFocus);
	setNumCols(1);
	setNumRows(1);
	setCellHeight(ICON_HEIGHT);
//	setCellWidth(100);
	MaxWidth = -1;
	LockPixmap = MountPixmap = LinkPixmap = DNDIcon = 0;
	Combo = 0;
	pressed = dragging = false;
	Root = new DirViewItem(0,"Desktop",0);
	Root->ParentDirView = this;
	Root->Opened = TRUE;
	Root->ChildsScanned = TRUE;
	CurrentItem = Root;
	PopupItem = DragItem = TargetItem = 0;
	ViewList.setAutoDelete(FALSE);
	ItemList.setAutoDelete(FALSE);
	ItemList.append(Root);

	ToolTip = new DirViewToolTip(this);

	updateViewList();
}

DirView::~DirView()
{
	delete Root;
	delete ToolTip;
}

void DirView::setRootIcon(QPixmap *icon)
{ Root->Icon = Root->IconSel = icon;}

void DirView::addItem(DirViewItem *parent, const char *name, const char *addr, QPixmap *icon, QPixmap *iconSel, bool sorting, bool hasChilds, int type, bool updateF)
{
	DirViewItem	*item, *Parent = (parent ? parent : Root);
	item = new DirViewItem(Parent,name,addr,icon,iconSel,hasChilds,type);
	item->Sorting = sorting;
	if (updateF) {
		emit updateHasChilds(Parent);
		if (Parent->Opened) {
			updateViewList();
			update();
		}
	}
	else updateCell(Parent);
}

void DirView::addItem(const char *path, const char *name, const char *addr, QPixmap *icon, QPixmap *iconSel, bool sorting, bool hasChilds, int type, bool updateF)
{
	DirViewItem	*Parent = findItem(path);
	if (Parent == 0) return;
	DirViewItem	*item;
	item = new DirViewItem(Parent,name,addr,icon,iconSel,hasChilds,type);
	item->Sorting = sorting;
	if (updateF) emit updateHasChilds(Parent);
	if (Parent->Opened) {
		updateViewList();
		update();
	}
	else updateCell(Parent);
}

void DirView::addItemUrl(const char *url, const char *name, const char *addr, QPixmap *icon, QPixmap *iconSel, bool toBeShown)
{
	QListIterator<DirViewItem>	it(ItemList);
	for (;it.current();++it)
		if (it.current()->Address == url)
			if (it.current()->ChildsScanned && toBeShown) addItem(it.current(),name,addr,icon,iconSel,TRUE,FALSE);
			else emit updateHasChilds(it.current());
}

void DirView::removeItem(DirViewItem *parent, const char *name)
{
	DirViewItem	*item = parent->findChild(name);
	if (item == 0) return;
	parent->removeChildItem(item);
	emit updateHasChilds(parent);
	updateViewList();
	update();
	if (ViewList.findRef(CurrentItem) == -1) {
		changeSelection(Root);
		emit selectionChanged(CurrentItem);
	}
}

void DirView::removeItem(const char *path, const char *name)
{
	DirViewItem	*Parent = findItem(path);
	if (Parent != 0) removeItem(Parent,name);
}

void DirView::removeItem(DirViewItem *item)
{ ItemList.removeRef(item);}

void DirView::addItem(DirViewItem *item)
{ if (ItemList.findRef(item) == -1) ItemList.append(item);}

void DirView::removeItem(const char *url)
{
	DirViewItem	*item;
	while ((item = findItemFromURL(url)) != 0) {
cout << "removing item " << item->Address << endl;
		removeItem(item->ParentItem,item->Label.data());
	}
}

void DirView::renameItem(const char *urlsrc, const char *urldest)
{
	DirViewItem	*item;
	QString		dest(urldest);
	if (dest[dest.length()-1] == '/') dest.truncate(dest.length()-1);
	QString		name(dest.right(dest.length()-dest.findRev('/')-1));
	KURL::decodeURL(name);
	while ((item = findItemFromURL(urlsrc)) != 0) {
		DirViewItem	*parent = item->ParentItem;
		if (parent->findChild(name.data()))	// we're renaming an item to something already
							// existing, then remove the existing item
			removeItem(parent,item->Label.data());
		else {
			parent->Childs->setAutoDelete(FALSE);
			removeItem(parent,item->Label.data());
			item->Label = name.data();
			item->Address = urldest;
			parent->addChildItem(item);
			parent->Childs->setAutoDelete(TRUE);
		}
	}
	QListIterator<DirViewItem>	it(ItemList);
	for (;it.current();++it)
		if (it.current()->Address.find(urlsrc) >= 0)
			it.current()->Address.replace(0,strlen(urlsrc),urldest);
	updateViewList();
	update();
}

void DirView::removePointFiles()
{
	setUpdatesEnabled(FALSE);
	bool	done = FALSE;
	while (!done) {
		done = TRUE;
		QListIterator<DirViewItem>	it(ItemList);
		for (;it.current();++it)
			if (it.current()->Label[0] == '.') {
				done = FALSE;
				removeItem(it.current()->ParentItem,it.current()->Label.data());
			}
	}
	setUpdatesEnabled(TRUE);
	repaint();
}

void DirView::removeArchiveFiles(SubProtList *list)
{
	setUpdatesEnabled(FALSE);
	bool	done = FALSE;
	while (!done) {
		done = TRUE;
		int	pos;
		QListIterator<DirViewItem>	it(ItemList);
		for (;it.current();++it)
			if ((pos=it.current()->Address.findRev('#')) != -1 && pos == (int)(it.current()->Address.length())-5
			    && (list == 0 || matchFileName(it.current()->Label.data(),list) == 0)) {
				done = FALSE;
				removeItem(it.current()->ParentItem,it.current()->Label.data());
			}
	}
	setUpdatesEnabled(TRUE);
	repaint();
}

DirViewItem* DirView::findItem(const char *path)
{
	QString	tmp(path);
	char	*c = strtok(tmp.data(),"/");
	if (c) c = strtok(0,"/");
	else return 0;
	DirViewItem	*item = Root;
	while (c && item) {
		item = item->findChild(c);
		c = strtok(0,"/");
	}
	return item;
}

DirViewItem* DirView::findItemFromURL(const char *url, const char *tree)
{
	QListIterator<DirViewItem>	it(ItemList);
	for (;it.current();++it)
		if (it.current()->Address == url) {
			if (!tree || it.current()->getPath().find(tree) != -1) return it.current();
		}
	return 0;
}

void DirView::paintCell(QPainter *p, int row, int col)
{
	DirViewItem	*current = ViewList.at(row);
	QFontMetrics	fm(p->fontMetrics());
	int	w = current->Depth*ICON_HEIGHT;
	QPen	normal(foregroundColor(),0,SolidLine), dashed(foregroundColor(),0,DotLine);
	p->setPen(normal);
	bool	target = (current == TargetItem);
	if (current == CurrentItem || target) {
		if (hasFocus() || target) {
			p->setBrush(QApplication::winStyleHighlightColor());
			p->drawRect(w+ICON_HEIGHT,1,fm.width(current->Label.data())+8,ICON_HEIGHT-2);
			p->setBrush(NoBrush);
		}
		if (current == CurrentItem && current->IconSel) p->drawPixmap(w+1,1,*(current->IconSel));
		else if (current->Icon) p->drawPixmap(w+1,1,*(current->Icon));
		p->drawRect(w+ICON_HEIGHT,1,fm.width(current->Label.data())+8,ICON_HEIGHT-2);
		p->setPen((hasFocus() || target ? QPen(white) : normal));
		p->drawText(w+ICON_HEIGHT+4,0,fm.width(current->Label.data()),ICON_HEIGHT,AlignLeft | AlignVCenter,current->Label.data());
		p->setPen(normal);
	}
	else {
		if (current->Icon) p->drawPixmap(w+1,1,*(current->Icon));
		p->drawText(w+ICON_HEIGHT+4,0,fm.width(current->Label.data()),ICON_HEIGHT,AlignLeft | AlignVCenter,current->Label.data());
	}
	switch(current->Type) {
	   case DirNormal: break;
	   case DirLink: if (LinkPixmap) p->drawPixmap(w+1,1,*LinkPixmap);
		      	 break;
	   case DirMount: if (MountPixmap) p->drawPixmap(w,ICON_HEIGHT-MountPixmap->height(),*MountPixmap);
		          break;
	   case DirLock: if (LockPixmap) p->drawPixmap(w,ICON_HEIGHT-LockPixmap->height(),*LockPixmap);
		         break;
	   default: break;
	}
	if (current->ParentItem) {
		p->setPen(dashed);
		p->drawLine(w-ICON_HEIGHT/2,ICON_HEIGHT/2,w,ICON_HEIGHT/2);
		if (current->getNext()) p->drawLine(w-ICON_HEIGHT/2,0,w-ICON_HEIGHT/2,ICON_HEIGHT);
		else p->drawLine(w-ICON_HEIGHT/2,0,w-ICON_HEIGHT/2,ICON_HEIGHT/2);
		DirViewItem	*prev = current->ParentItem;
		while (prev != Root) {
			if (prev->getNext()) p->drawLine(prev->Depth*ICON_HEIGHT-ICON_HEIGHT/2,0,prev->Depth*ICON_HEIGHT-ICON_HEIGHT/2,ICON_HEIGHT);
			prev = prev->ParentItem;
		}
		p->setPen(normal);
		if (current->HasChilds || current->Childs->count() > 0 || current->Depth == 2) {
			p->drawRect(w-ICON_HEIGHT/2-4,ICON_HEIGHT/2-4,9,9);
			p->fillRect(w-ICON_HEIGHT/2-3,ICON_HEIGHT/2-3,7,7,backgroundColor());
			p->drawLine(w-ICON_HEIGHT/2-2,ICON_HEIGHT/2,w-ICON_HEIGHT/2+2,ICON_HEIGHT/2);
			if (!(current->Opened))
				p->drawLine(w-ICON_HEIGHT/2,ICON_HEIGHT/2-2,w-ICON_HEIGHT/2,ICON_HEIGHT/2+2);
		}
	}
}

void DirView::updateViewList()
{
	bool	updates = isUpdatesEnabled();
	setAutoUpdate(false);
	ViewList.clear();
	DirViewItem	*current = Root->Childs->first();
	int	index = 1;
	MaxWidth = -1;
	ViewList.append(Root);
	if (Root->Opened) {
		while (current) {
			ViewList.append(current);
			MaxWidth = QMAX(MaxWidth,(current->Depth+1)*cellHeight()+fontMetrics().width(current->Label.data())+8);
			current->Index = index;
			if (current->Opened && current->Childs->first()) {
				current = current->Childs->first();
			}
			else if (current->getNext() == 0) {
				while (current->Depth > 0 && current->getNext() == 0) {
					current = current->ParentItem;
				}
				current = current->getNext();
			}
			else current = current->getNext();
			index++;
		}
	}
	setNumRows(ViewList.count());
	setCellWidth(QMAX(MaxWidth,viewWidth()));
	setAutoUpdate(true);
	setUpdatesEnabled(updates);
}

void DirView::resizeEvent(QResizeEvent *)
{
	QTableView::resizeEvent(0);
	setCellWidth(QMAX(MaxWidth,viewWidth()));
}

void DirView::mousePressEvent(QMouseEvent *e)
{
	canDrag = canSelect = false;
	DragItem = 0;
	pressed = true;
	int	row = findRow(e->y());
	if (row < 0) return;
	DirViewItem	*current = ViewList.at(row);
	int		p(e->x()+xOffset());
	if (e->button() == LeftButton) {
		if (p > current->Depth*cellHeight() && p < (current->Depth+1)*cellHeight()+fontMetrics().width(current->Label.data())+8) {
			canSelect = true;
			if (current->Depth > 2) {
				canDrag = true;
				press_x = e->x();
				press_y = e->y();
			}
			DragItem = current;
		}
		else if (expandRect(current).contains(e->pos())) {
			if (current->Opened) collapseItem(current);
			else expandItem(current);
			pressed = false;
		}
	}
	else if (e->button() == RightButton && p > current->Depth*cellHeight()) {
		PopupItem = current;
		emit rightButtonPressed(current);
	}
	else if (e->button() == MidButton && p > current->Depth*cellHeight()) {
		if (current->Opened) collapseItem(current);
		else expandItem(current);
		pressed = false;
	}
}

void DirView::mouseMoveEvent(QMouseEvent *e)
{
	if (canDrag && DragItem) {
		if (abs(e->x() - press_x) + abs(e->y() - press_y) > 10) {
			QStrList	List;
			List.append(DragItem->Address.data());
			QUrlDrag	*object = new QUrlDrag(List,this);
			if (DNDIcon) object->setPixmap(*DNDIcon);
			dragging = true;
			object->drag();
			dragging = canDrag = pressed = false;
			DragItem = 0;
			press_x = press_y = 0;
		}
	}
	else {
		int	row = findRow(e->y());
		DirViewItem	*item = (row >= 0 ? ViewList.at(row) : 0);
		int		p(e->x()+xOffset());
		if (item && p > item->Depth*cellHeight() && p < (item->Depth+1)*cellHeight()+fontMetrics().width(item->Label.data())+8
		    && !pressed)
			setCursor(upArrowCursor);
		else setCursor(arrowCursor);
	}
}

void DirView::mouseReleaseEvent(QMouseEvent *e)
{
	canDrag = pressed = false;
	if (canSelect) {
		if (CurrentItem != DragItem) {
			changeSelection(DragItem);
			emit selectionChanged(CurrentItem);
		}
	}
	canSelect = false;
	DragItem = 0;
}

void DirView::mouseDoubleClickEvent(QMouseEvent *e)
{
	if (e->button() == LeftButton) {
		int	row = findRow(e->y());
		if (row < 0) return;
		DirViewItem	*current = ViewList.at(row);
		int		p(e->x()+xOffset());
		if (p > current->Depth*cellHeight() && p < (current->Depth+1)*cellHeight()+fontMetrics().width(current->Label.data())+8) {
			if (current->Opened) collapseItem(current);
			else expandItem(current);
		}
	}
}

void DirView::changeSelection(DirViewItem *newItem)
{
	if (newItem == CurrentItem) return;
	DirViewItem	*old = CurrentItem;
	CurrentItem = newItem;
	if (old) QTableView::updateCell(old->Index,0);
	if (CurrentItem && (CurrentItem == Root || CurrentItem->ParentItem->Opened)) {
		QTableView::updateCell(CurrentItem->Index,0);
		scrollInView();
	}
	if (Combo) updateCombo();
}

QRect DirView::expandRect(DirViewItem *item)
{       if (item->HasChilds || item->Childs->count() > 0 || item->Depth <= 2) {
		int	x,y;
		rowYPos(item->Index,&y);
		colXPos(0,&x);
		return QRect(x+item->Depth*ICON_HEIGHT-ICON_HEIGHT/2-3,y+cellHeight()/2-3,9,9);
	}
	else return QRect(0,0,0,0);
}

void DirView::expandItem(DirViewItem *item, bool redraw)
{
	if (!item->Opened) {
		item->Opened = TRUE;
		setUpdatesEnabled(false);
		if (!(item->ChildsScanned)) {
			emit requestItemChilds(item);
/*item->ChildsScanned = TRUE;
			if (!item->ChildsScanned) return;*/
		}
		setUpdatesEnabled(true);
		if (redraw) {
			updateViewList();
			if (!item->Childs->isEmpty()) {
				// try to set vertical offset so that all subitems ar shown
				int	i1(item->Index), i2(item->Childs->last()->Index), offset(-1);
				if ((i2+1)*ICON_HEIGHT-yOffset() > height()-2*lineWidth()) {
					offset = ((i2-i1)*ICON_HEIGHT < height() ? (i2+1)*ICON_HEIGHT-height()+2*lineWidth() : i1*ICON_HEIGHT);
					setYOffset(offset);
				}
			}
			update();
		}
	}
}

void DirView::collapseItem(DirViewItem *item, bool redraw)
{
	if (item->Opened) {
		item->Opened = FALSE;
		DirViewItem	*next = item;
		bool		needEmit = FALSE;
		while (next && next->getNext() == 0) next = next->ParentItem;
		if (next) next = next->getNext();
		if ((CurrentItem && next && CurrentItem->Index > item->Index && CurrentItem->Index < next->Index) ||
		    (CurrentItem && next == 0 && CurrentItem->Index > item->Index)) {
			changeSelection(item);
			needEmit = TRUE;
		}
		if (redraw) {
			updateViewList();
			update();
		}
		if (needEmit) emit selectionChanged(item);
	}
}

void DirView::updateCombo()
{
	int	index = (CurrentItem ? 0 : -1);
	Combo->clear();
	Combo->insertItem("Desktop",Root->Icon,0);
	if (CurrentItem && CurrentItem != Root && Root->Childs->count() > 0) {
		DirViewItem	*item = CurrentItem->getFirstParent();
		DirViewItemListIterator		it(*(Root->Childs));
		for (; it.current() && it.current() != item; ++it)
			Combo->insertItem(it.current()->Label.data(),it.current()->Icon,1);
		if (item) {
			Combo->insertItem(item->Label.data(),item->Icon,1);
			QList<DirViewItem>	list;
			list.setAutoDelete(FALSE);
			DirViewItem	*tmp = CurrentItem;
			while (tmp) {
				list.insert(0,tmp);
				tmp = tmp->ParentItem;
			}
			QListIterator<DirViewItem>	it2(list);
			for (int i=0;it2.current() && i<2;++it2,i++) ;
			for (int i=2;it2.current();++it2,i++) Combo->insertItem(it2.current()->Label.data(),it2.current()->Icon,i);
		}
		index = Combo->count()-1;
		for (++it; it.current(); ++it)
			Combo->insertItem(it.current()->Label.data(),it.current()->Icon,1);
	}
	Combo->setCurrentItem(index);
}

void DirView::cdUp()
{
	if (CurrentItem && CurrentItem != Root) {
		changeSelection(CurrentItem->ParentItem);
		emit selectionChanged(CurrentItem);
	}
}

#include <kapp.h>
bool DirView::cdDown(const char *dir)
{
cout << "looking " << dir << endl;
	bool	Scanned = CurrentItem->ChildsScanned, Rescanned = FALSE;
	DirViewItem	*item;
	while (1) {
		if (Scanned) {
			item = CurrentItem->Childs->first();
			while (item && item->Label != dir) item = CurrentItem->Childs->next();
			if (item) break;
			else if (Rescanned) {
				QString		msg;
				msg.sprintf(i18n("Can't find directory %s."),dir);
				QMessageBox::critical(this,i18n("Error"),msg.data(),QMessageBox::Ok | QMessageBox::Default,0);
				break;
			}
			else {
				CurrentItem->Childs->clear();
				emit requestItemChilds(CurrentItem);
				Rescanned = TRUE;
				CurrentItem->Opened = FALSE;
			}
		}
		else {
			emit requestItemChilds(CurrentItem);
			Scanned = Rescanned = TRUE;
		}
	}
	if (item) {
		if (!CurrentItem->Opened) {
			CurrentItem = item;
			kapp->getConfig()->setGroup("Configuration");
			if (kapp->getConfig()->readBoolEntry("TreeFollow",true)) {
				CurrentItem->ParentItem->Opened = true;
				updateViewList();
				update();
			}
			if (Combo) updateCombo();
		}
		else changeSelection(item);
		scrollInView();
		if (isUpdatesEnabled()) emit selectionChanged(item);
		return TRUE;
	}
	else return FALSE;
}

void DirView::goToDir(const char *path, bool relative)
{
	setUpdatesEnabled(FALSE);
	QString		Path(path);
	if (Path[Path.length()-1] != '/') Path += '/';
	char	*last, *c = strtok_r(Path.data(),"/",&last);
	if (!relative) {
		changeSelection(Root);
		c = strtok_r(0,"/",&last);	// to strip root item
	}
	while (c) {
		if (!cdDown(c)) break;
		c = strtok_r(0,"/",&last);
	}
	setUpdatesEnabled(TRUE);
	scrollInView();
	update();
	emit selectionChanged(CurrentItem);
}

void DirView::comboChanged(int index)
{
	if (index == 0) {
		changeSelection(Root);
		emit selectionChanged(Root);
		return;
	}
	DirViewItem	*item = CurrentItem;
	while (item->Depth > 0 && item->Label != Combo->text(index)) item = item->ParentItem;
	if (item->Label != Combo->text(index)) {
		item = Root->Childs->first();
		while (item && item->Label != Combo->text(index)) item = Root->Childs->next();
	}
	if (item) {
		changeSelection(item);
		emit selectionChanged(item);
	}
}

void DirView::dropEvent(QDropEvent *e)
{
	int	row = findRow(e->pos().y());
	if (row < 0) return;
	emit dropAccepted(e,ViewList.at(row));
	DirViewItem	*old = TargetItem;
	TargetItem = 0;
	updateCell(old,true);
}

void DirView::dragEnterEvent(QDragEnterEvent *e)
{
	if (QUrlDrag::canDecode(e)) e->accept();
}

void DirView::dragMoveEvent(QDragMoveEvent *e)
{
	if (!QUrlDrag::canDecode(e)) { e->accept(); return;}
	int	row = findRow(e->pos().y());
	DirViewItem	*oldItem = TargetItem;
	TargetItem = (row >= 0 ? ViewList.at(row) : 0);
	if (TargetItem && (TargetItem->Depth < 2 ||
	    e->pos().x() < TargetItem->Depth*cellHeight() ||
	    e->pos().x() > (TargetItem->Depth+1)*cellHeight()+fontMetrics().width(TargetItem->Label.data())+8)) TargetItem = 0;
	bool	Accept(false);
	if (TargetItem)
		if (dragging) { if (TargetItem != DragItem) Accept = true;}
		else if (TargetItem != CurrentItem) Accept = true;;
	if (Accept) e->accept();
	else { e->ignore(); TargetItem = 0;}
	if (oldItem != TargetItem) {
		if (oldItem) updateCell(oldItem,true);
		if (TargetItem) updateCell(TargetItem,true);
	}
}

void DirView::dragLeaveEvent(QDragLeaveEvent *e)
{
	if (TargetItem) {
		DirViewItem	*old(TargetItem);
		TargetItem = 0;
		updateCell(old,true);
	}
}

void DirView::refresh()
{
	repaint(TRUE);
	if (Combo) Combo->repaint(TRUE);
}

void DirView::clearItem(DirViewItem *item, bool changeIfCurrent, bool propagate)
{
	item->Childs->clear();
	item->Opened = false;
	item->ChildsScanned = false;
	updateViewList();
	update();
	if ((changeIfCurrent && CurrentItem == item) || (CurrentItem && ViewList.findRef(CurrentItem) == -1)) {
		changeSelection(Root);
		emit selectionChanged(CurrentItem);
	}
	if (propagate) {
		QString		url = item->Address.data();
		QListIterator<DirViewItem>	it(ItemList);
		for (;it.current();++it)
			if (it.current()->Address == url && it.current() != item) clearItem(it.current(),true,false);
	}
}

void DirView::rescanChilds(DirViewItem *item, bool openIt)
{
	if (item->Depth < 2 && item->Address.isEmpty()) return;
	setUpdatesEnabled(FALSE);
//	item->Childs->clear();
//	updateViewList();
//	item->ChildsScanned = false;
	emit requestItemChilds(item);
	if (openIt) item->Opened = TRUE;
	updateViewList();
	setUpdatesEnabled(TRUE);
	repaint();
}

void DirView::refreshItem(DirViewItem *item)
{
	if (item->ChildsScanned) rescanChilds(item,false);
	else emit updateHasChilds(item);
}

void DirView::refreshItem(const char *url)
{
	QListIterator<DirViewItem>	it(ItemList);
	for (;it.current();++it) if (it.current()->Address == url) refreshItem(it.current());
}

void DirView::updateCell(DirViewItem *item, bool redraw)
{
	int	w = (item->Depth+1)*cellHeight()+fontMetrics().width(item->Label.data())+8;
	if (w > cellWidth()) {
		setCellWidth(w);
		update();
	}
	else QTableView::updateCell(item->Index,0,redraw);
}

void DirView::setCurrentItem(const char *path, bool emitF)
{ setCurrentItem(findItem(path),emitF);}

void DirView::focusInEvent(QFocusEvent *)
{ if (ViewList.findRef(CurrentItem) != -1) QTableView::updateCell(CurrentItem->Index,0);}

void DirView::focusOutEvent(QFocusEvent *)
{ if (ViewList.findRef(CurrentItem) != -1) QTableView::updateCell(CurrentItem->Index,0);}

void DirView::keyPressEvent(QKeyEvent *e)
{
	DirViewItem	*newItem(0);
	int		state(0);
	if (ViewList.findRef(CurrentItem) != -1) ViewList.at(CurrentItem->Index);
	else ViewList.first();
	switch (e->key()) {
	   case Key_Down: newItem = ViewList.next(); state = 1; break;
	   case Key_Up: newItem = ViewList.prev(); state = 1; break;
	   case Key_Right: expandItem(CurrentItem,true); break;
	   case Key_Left: collapseItem(CurrentItem,true); break;
	   case Key_PageUp : moveUp(CurrentItem); break;
	   case Key_PageDown : moveDown(CurrentItem); break;
	   case Key_Enter:
	   case Key_Return:
		if (CurrentItem && CurrentItem->Opened) collapseItem(CurrentItem,true);
		else if (CurrentItem) expandItem(CurrentItem,true);
		break;
	   case Key_Backspace: cdUp(); break;
	}
	if (state) {
		if (!newItem) newItem = CurrentItem;
		changeSelection(newItem);
		emit selectionChanged(CurrentItem);
	}
}

QRect DirView::tip(const QPoint& pos, QString& tipStr)
{
	int	row = findRow(pos.y());
	if (row >= 0) {
		DirViewItem	*item = ViewList.at(row);
		int	x1 = (item->Depth+1)*cellHeight()+minViewX(), x2 = x1+fontMetrics().width(item->Label.data())+4;
		if (x2-xOffset() > maxViewX()) {
			int	y;
			rowYPos(row,&y);
			tipStr = item->Label.data();
			return QRect(x1-xOffset(),y,x2-x1-xOffset(),cellHeight());
		}
	}
	return QRect(0,0,-1,-1);
}

void DirView::moveUp(DirViewItem *item)
{
	if (item->Depth == 2) {
		int	pos = item->ParentItem->Childs->findRef(item);
		if (pos > 0) {
			item->ParentItem->Childs->insert(pos-1,item);
			item->ParentItem->Childs->take(pos+1);
			updateViewList();
			update();
		}
	}
}

void DirView::moveDown(DirViewItem *item)
{
	if (item->Depth == 2) {
		int	pos = item->ParentItem->Childs->findRef(item);
		if (pos < (int)(item->ParentItem->Childs->count()-1)) {
			item->ParentItem->Childs->insert(pos+2,item);
			item->ParentItem->Childs->take(pos);
			updateViewList();
			update();
		}
	}
}
