/* 
 * pgpconfig.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <iostream.h>
#include <stdlib.h>

#include <qdir.h>
#include <qlayout.h>
#include <qfiledlg.h>
#include <qfileinf.h>
#include <qbttngrp.h>

#include <kmsgbox.h>

#include "pgpconfig.h"

// appearance dialog
KPGPConfig::~KPGPConfig ()
{
}

QWidget* KPGPConfig::fileSettings(QPushButton* &btn, QString name, QLineEdit * &ledit)
{
  QGroupBox   *group = new QGroupBox(name, this, "group");
  QBoxLayout  *line  = new QBoxLayout(group, QBoxLayout::LeftToRight, 10);
  
  ledit = new QLineEdit(group);
  ledit->adjustSize();
  ledit->setMinimumSize(100, ledit->height());
  ledit->setMaximumSize(1000, ledit->height());
  
  line->addWidget(ledit, 0, AlignBottom);
  line->addSpacing(10);
  
  btn = new QPushButton("...", group);
  btn->setFixedSize(btn->sizeHint().width(), 20);
  
  line->addWidget(btn, 0, AlignBottom);
  
  group->adjustSize();
  group->setMinimumSize(150, ledit->height()+30);
  
  return group;
};

QWidget* KPGPConfig::getCheckSettings()
{
  QGroupBox   *group = new QGroupBox("PGP Variables", this, "group_check");
  QBoxLayout  *row   = new QBoxLayout(group, QBoxLayout::Down, 10);
  
  row->addSpacing(10);
  
  checkTEXTMODE = new QCheckBox("TEXTMODE", group);
  checkTEXTMODE->adjustSize();
  checkTEXTMODE->setChecked(TEXTMODE);
  row->addWidget(checkTEXTMODE);
  
  checkARMOR = new QCheckBox("ARMOR", group);
  checkARMOR->adjustSize();
  checkARMOR->setChecked(ARMOR);
  row->addWidget(checkARMOR);
  
  checkKEEPBINARY = new QCheckBox("KEEPBINARY", group);
  checkKEEPBINARY->adjustSize();
  checkKEEPBINARY->setChecked(KEEPBINARY);
  row->addWidget(checkKEEPBINARY);
  
  checkCOMPRESS = new QCheckBox("COMPRESS", group);
  checkCOMPRESS->adjustSize();
  checkCOMPRESS->setChecked(COMPRESS);
  row->addWidget(checkCOMPRESS);
  
  checkSHOWPASS = new QCheckBox("SHOWPASS", group);
  checkSHOWPASS->adjustSize();
  checkSHOWPASS->setChecked(SHOWPASS);
  row->addWidget(checkSHOWPASS);
  
  checkCLEARSIG = new QCheckBox("CLEARSIG", group);
  checkCLEARSIG->adjustSize();
  checkCLEARSIG->setChecked(CLEARSIG);
  row->addWidget(checkCLEARSIG);
  
  checkENCRYPTTOSELF = new QCheckBox("ENCRYPTTOSELF", group);
  checkENCRYPTTOSELF->adjustSize();
  checkENCRYPTTOSELF->setChecked(ENCRYPTTOSELF);
  row->addWidget(checkENCRYPTTOSELF);
  
  checkINTERACTIVE = new QCheckBox("INTERACTIVE", group);
  checkINTERACTIVE->adjustSize();
  checkINTERACTIVE->setChecked(INTERACTIVE);
  row->addWidget(checkINTERACTIVE);
  
  group->adjustSize();
  
  return group;
};

QWidget* KPGPConfig::getTrustSettings()
{
  QLabel    *label;
  QLineEdit *completeEdit;
  QLineEdit *marginalEdit;
  QLineEdit *depthEdit;

  QGroupBox  *group = new QGroupBox(this, "group_trust");
  QBoxLayout *box   = new QBoxLayout(group, QBoxLayout::Down, 7);
  QBoxLayout *line;
  QString     str;
     
  
  line = new QBoxLayout(QBoxLayout::LeftToRight, 7);
  
     box->addLayout(line);
     
     label = new QLabel( "Number of Completly Trusted Introducers Needed", group );
     label->adjustSize();
     label->setMinimumSize(label->width(), label->height());
     line->addWidget(label);
     
     completeEdit = new QLineEdit( group );
     line->addWidget(completeEdit);
     
     str.setNum(COMPLETES_NEEDED);
     completeEdit->setText( str );
     completeEdit->setFixedSize(30, completeEdit->sizeHint().height());
     
     connect( completeEdit, SIGNAL( textChanged( const char * ) ),
	     SLOT( slotCompleteNeededChanged( const char * ) ) );
     line->addSpacing(10);
        
   
  line = new QBoxLayout(QBoxLayout::LeftToRight, 7);
  
     box->addLayout(line);
  
     label = new QLabel( "Number of Marginally Trusted Introducers Needed", group );
     label->adjustSize();
     label->setMinimumSize(label->width(), label->height());
     line->addWidget(label);
  
     marginalEdit = new QLineEdit( group );
     str.setNum(MARGINALS_NEEDED);
     marginalEdit->setText( str );
     marginalEdit->setFixedSize(30, marginalEdit->sizeHint().height());
     line->addWidget(marginalEdit);
     
     connect( marginalEdit, SIGNAL( textChanged( const char * ) ),
	     SLOT( slotMarginalNeededChanged( const char * ) ) );  
     line->addSpacing(10);
     
  line = new QBoxLayout(QBoxLayout::LeftToRight, 7);
  
     box->addLayout(line);
  
     label = new QLabel( "How Deep May Introducers be Nested", group );
     line->addWidget(label);
     label->adjustSize();
     label->setMinimumSize(label->width(), label->height());
     
     depthEdit = new QLineEdit( group );
     str.setNum(CERT_DEPTH);
     depthEdit->setText( str );
     depthEdit->setFixedSize(30, marginalEdit->sizeHint().height());
     line->addWidget(depthEdit);
     
     connect( depthEdit, SIGNAL( textChanged( const char * ) ),
  	  SLOT( slotCertDepthChanged( const char * ) ) );
     line->addSpacing(10);
  
  group->adjustSize();
  group->setMinimumSize(group->width(), 105);
  
  return group;
};

QWidget* KPGPConfig::getCommentSettings()
{
  QGroupBox   *group = new QGroupBox("Default PGP Comment", this, "grp_comment");
  QBoxLayout  *line  = new QBoxLayout(group, QBoxLayout::LeftToRight, 10);
  
  commentEdit = new QLineEdit(group);
  commentEdit->setText(COMMENT);
  commentEdit->adjustSize();
  commentEdit->setMinimumSize(100, commentEdit->height());
  commentEdit->setMaximumSize(1000, commentEdit->height());
  
  line->addWidget(commentEdit, 0, AlignBottom);
  
  group->adjustSize();
  group->setMinimumSize(120, commentEdit->height()+30);
  
  return group;
};

KPGPConfig::KPGPConfig (QWidget * parent, const char *name)
  : KConfigWidget (parent, name)
{
  QPushButton *btn;
  
  config_found = FALSE;
  loadSettings();
  
  if (config_found)
    {
      QBoxLayout *base  = new QBoxLayout(this, QBoxLayout::Down, 5);
      QBoxLayout *line  = new QBoxLayout(QBoxLayout::LeftToRight);
      QBoxLayout *files = new QBoxLayout(QBoxLayout::Down);
      
      base->addLayout(line);
      
       line->addWidget(getCheckSettings());
       line->addLayout(files);
      
        files->addWidget(fileSettings(btn, "Public Key File", pubringEdit));
	pubringEdit->setText(PUBRING);
	connect(btn, SIGNAL(clicked()), this, SLOT(slotChoosePubring()));
  
        files->addWidget(fileSettings(btn, "Secret Key File", secringEdit));
        secringEdit->setText(SECRING);
	connect(btn, SIGNAL(clicked()), this, SLOT(slotChooseSecring()));
  
        files->addWidget(fileSettings(btn, "Seed File", seed_fileEdit));
        seed_fileEdit->setText(RANDSEED);
	connect(btn, SIGNAL(clicked()), this, SLOT(slotChooseSeedFile()));
  
        // files->addWidget(getNameSettings());
      
      base->addWidget(getTrustSettings());
      base->addWidget(getCommentSettings());
    }
}

void KPGPConfig::resizeEvent(QResizeEvent *)
{
}

 
void KPGPConfig::slotChoosePubring()
{
  QFileDialog *d = new QFileDialog(pgpPath, "*", this, NULL, TRUE);
  
  d->setCaption("Choose Public PGP Keyring File");
  if (d->exec()) 
    pubringEdit->setText(d->selectedFile());
  
  delete d;
}

void KPGPConfig::slotChooseSecring()
{
  QFileDialog *d = new QFileDialog(pgpPath, "*", this, NULL, TRUE);
  
  d->setCaption("Choose Secret PGP Keyring File");
  if (d->exec()) 
    secringEdit->setText(d->selectedFile());
  
  delete d;
}

void KPGPConfig::slotChooseSeedFile()
{
  QFileDialog *d = new QFileDialog(pgpPath, "*", this, NULL, TRUE);
  
  d->setCaption("Choose ranseed File");
  if (d->exec()) 
    seed_fileEdit->setText(d->selectedFile());
  
  delete d;
}

void KPGPConfig::slotCertDepthChanged( const char *depth )
{
  int i = atoi( depth );
  
  if ((i > 0) && (i < 9))
    CERT_DEPTH = i;
};
  
void KPGPConfig::slotCompleteNeededChanged( const char *needed )
{
  int i = atoi( needed );
  
  if (i > 0)
    COMPLETES_NEEDED = i;
};
  
void KPGPConfig::slotMarginalNeededChanged( const char *needed )
{
  int i = atoi( needed );
  
  if (i > 0)
    MARGINALS_NEEDED = i;
};

bool KPGPConfig::openConfig(QString name)
{
  QFileInfo finfo(name);
  
  if (finfo.isReadable() && finfo.isWritable())
    {
      config = new KSimpleConfig(name);
      config_found = TRUE;
      cout << name << " exists and set" << endl;	    
    }
  else
    {
      KMsgBox::message(this, "PGP Config Error", "Can't read/write config file " + name);
	
      config_found = FALSE;
    }
    
  return config_found;	
}

void KPGPConfig::applySettings()
{
  if (config_found)
    {
      config->writeEntry("PUBRING", pubringEdit->text());
      config->writeEntry("SECRING", secringEdit->text());
      config->writeEntry("RANDSEED", seed_fileEdit->text());
      
      config->writeEntry("COMMENT", commentEdit->text());
      
      if (checkINTERACTIVE->isChecked())
	config->writeEntry("INTERACTIVE", "on");
      else
	config->writeEntry("INTERACTIVE", "off");
      
      if (checkCOMPRESS->isChecked())
	config->writeEntry("COMPRESS", "on");
      else
	config->writeEntry("COMPRESS", "off");
      
      if (checkKEEPBINARY->isChecked())
	config->writeEntry("KEEPBINARY", "on");
      else
	config->writeEntry("KEEPBINARY", "off");
      
      if (checkARMOR->isChecked())
	config->writeEntry("ARMOR", "on");
      else
	config->writeEntry("ARMOR", "off");
      
      if (checkTEXTMODE->isChecked())
	config->writeEntry("TEXTMODE", "on");
      else
	config->writeEntry("TEXTMODE", "off");
      
      if (checkSHOWPASS->isChecked())
	config->writeEntry("SHOWPASS", "on");
      else
	config->writeEntry("SHOWPASS", "off");
      
      if (checkCLEARSIG->isChecked())
	config->writeEntry("CLEARSIG", "on");
      else
	config->writeEntry("CLEARSIG", "off");
      
      if (checkENCRYPTTOSELF->isChecked())
	config->writeEntry("ENCRYPTTOSELF", "on");
      else
	config->writeEntry("ENCRYPTTOSELF", "off");
      
      
      config->writeEntry("COMPLETES_NEEDED", COMPLETES_NEEDED);
      config->writeEntry("MARGINALS_NEEDED", MARGINALS_NEEDED);
      config->writeEntry("CERT_DEPTH", CERT_DEPTH);
      config->writeEntry("VERBOSE", VERBOSE);
      
      config->sync();
    }
    
}

void KPGPConfig::loadSettings()
{
  QDir       pgp;
  
  
  // find pgp configuration file
    pgpPath = getenv("PGPPATH");
    
  // first test $PGPPATH 
    if ((pgpPath != NULL) && (pgpPath != ""))
      {
	if (pgp.cd(pgpPath))       // PGPPATH exists
	  openConfig(pgp.absPath()+"/config.txt");
	else
	  KMsgBox::message(this, "PGP Config Error", "$PGPPATH directory does not exist!");
      }
    else
      {
	if (pgp.exists(QDir::homeDirPath() + "/.pgprc"))
	  {
	    openConfig(QDir::homeDirPath() + "/.pgprc");  // open .pgprc
	    
	    pgp.cd(QDir::homeDirPath() + "/.pgp");        // set pgp_dir to ~/.pgp
	  }
	else
	  {
	    if (pgp.cd(QDir::homeDirPath() + "/.pgp"))
	      {
		openConfig(pgp.absPath()+"/config.txt");
	      }
	    else
	      {
		KMsgBox::message(this, "PGP Config Error", "No PGP config file found!");
	      }
	  }
      }
    
  if (config_found)
    {
      PUBRING = config->readEntry("PUBRING");
      if (PUBRING == NULL)
	if (pgp.exists("pubring.pgp"))
	  PUBRING = pgp.absPath() + "/pubring.pgp";
      
      SECRING = config->readEntry("SECRING");
      if (SECRING == NULL)
	if (pgp.exists("secring.pgp"))
	  SECRING = pgp.absPath() + "/secring.pgp";
  
      RANDSEED = config->readEntry("RANDSEED");
      if (RANDSEED == NULL)
	if (pgp.exists("randseed.bin"))
	  RANDSEED = pgp.absPath() + "/randseed.bin";
  
      COMMENT = config->readEntry("COMMENT");
      
      if (config->readEntry("INTERACTIVE") == "on")
	INTERACTIVE = TRUE;
      else
	INTERACTIVE = FALSE;
      
      if (config->readEntry("COMPRESS") == "off")
	COMPRESS = FALSE;
      else
	COMPRESS = TRUE;
      
      if (config->readEntry("KEEPBINARY") == "on")
	KEEPBINARY = TRUE;
      else
	KEEPBINARY = FALSE;
      
      if (config->readEntry("ARMOR") == "on")
	ARMOR = TRUE;
      else
	ARMOR = FALSE;
      
      if (config->readEntry("TEXTMODE") == "on")
	TEXTMODE = TRUE;
      else
	TEXTMODE = FALSE;
      
      if (config->readEntry("CLEARSIG") == "off")
	CLEARSIG = FALSE;
      else
	CLEARSIG = TRUE;
      
      if (config->readEntry("SHOWPASS") == "on")
	SHOWPASS = TRUE;
      else
	SHOWPASS = FALSE;
      
      if (config->readEntry("ENCRYPTTOSELF") == "on")
	ENCRYPTTOSELF = TRUE;
      else
	ENCRYPTTOSELF = FALSE;
  
      
      
      VERBOSE  = config->readNumEntry("VERBOSE");
      if (VERBOSE == 0)
	VERBOSE = 1;
      
      MARGINALS_NEEDED = config->readNumEntry("MARGINALS_NEEDED");
      if (MARGINALS_NEEDED == 0)
	MARGINALS_NEEDED = 2;
      
      COMPLETES_NEEDED = config->readNumEntry("COMPLETES_NEEDED");
      if (COMPLETES_NEEDED == 0)
	COMPLETES_NEEDED = 1;
  
      CERT_DEPTH = config->readNumEntry("CERT_DEPTH");
      if (CERT_DEPTH == 0)
	CERT_DEPTH = 4;
    }
}


#include "pgpconfig.moc"