/* 
 * keyring.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>

#include <qdir.h>
#include <qfileinf.h>

#include <kmsgbox.h>
#include <ksimpleconfig.h>

#include "keyring.h"

#define PUB_KEY          1
#define PUB_KEY_SIGN     2
#define PUB_KEY_EXTRACT  3
#define PUB_KEY_EDIT     4
#define PUB_KEY_DELETE   5

#define SEC_KEY          1
#define SEC_KEY_EXTRACT  12
#define SEC_KEY_EDIT     13
#define SEC_KEY_DELETE   14
#define SEC_KEY_DEFAULT  15

#define INFO             1
#define INFO_BITLEN      2
#define INFO_KEYID       3
#define INFO_FP          4

#define USERID           1
#define USERID_EDIT      2
#define USERID_DELETE    3

#define SIG              1
#define SIG_EDIT         2
#define SIG_DELETE       3




// appearance dialog
KPGPKeyring::~KPGPKeyring ()
{
  delete list;
}
   
KPGPKeyring::KPGPKeyring (QWidget * parent, const char *name)
  : KConfigWidget (parent, name)
{
  config_name = "";
  
  // Create key popup menu
  keyRightPopup = new QPopupMenu;
    keyRightPopup->insertItem("Public Key", PUB_KEY); 
    keyRightPopup->setItemEnabled(PUB_KEY, FALSE); 
    keyRightPopup->insertSeparator();
    keyRightPopup->insertItem("Sign", PUB_KEY_SIGN); 
    keyRightPopup->setItemEnabled(PUB_KEY_SIGN, FALSE); 
    keyRightPopup->insertItem("Extract", PUB_KEY_EXTRACT); 
    keyRightPopup->setItemEnabled(PUB_KEY_EXTRACT, FALSE); 
    keyRightPopup->insertItem("Edit", PUB_KEY_EDIT); 
    keyRightPopup->setItemEnabled(PUB_KEY_EDIT, FALSE); 
    keyRightPopup->insertItem("Delete", PUB_KEY_DELETE); 
    keyRightPopup->setItemEnabled(PUB_KEY_DELETE, FALSE); 
    connect (keyRightPopup, SIGNAL(activated(int)), this, SLOT(slotKeyPopup(int)));
   
  // Create key popup menu
  seckeyRightPopup = new QPopupMenu;
    seckeyRightPopup->insertItem("Private Key", SEC_KEY); 
    seckeyRightPopup->setItemEnabled(SEC_KEY, FALSE); 
    seckeyRightPopup->insertSeparator();
    seckeyRightPopup->insertItem("Extract", SEC_KEY_EXTRACT); 
    seckeyRightPopup->setItemEnabled(SEC_KEY_EXTRACT, FALSE); 
    seckeyRightPopup->insertItem("Edit", SEC_KEY_EDIT); 
    seckeyRightPopup->setItemEnabled(SEC_KEY_EDIT, FALSE); 
    seckeyRightPopup->insertItem("Delete", SEC_KEY_DELETE); 
    seckeyRightPopup->setItemEnabled(SEC_KEY_DELETE, FALSE); 
    seckeyRightPopup->insertSeparator();
    seckeyRightPopup->insertItem("Make Default", SEC_KEY_DEFAULT); 
    connect (seckeyRightPopup, SIGNAL(activated(int)), this, SLOT(slotKeyPopup(int)));
   
  // Create key popup menu
  keyLeftPopup = new QPopupMenu;
    keyLeftPopup->insertItem("Key Info", 1); 
    keyLeftPopup->setItemEnabled(1, FALSE); 
    keyLeftPopup->insertSeparator();
    keyLeftPopup->insertItem("Bitlen:", 2); 
    keyLeftPopup->insertItem("Key ID:", 3); 
    keyLeftPopup->insertItem("Created:", 4); 
    keyLeftPopup->insertSeparator();
    keyLeftPopup->insertItem("FINGERPRINT", 5); 
   
  
  list = new MyTreeList(this, "");
  list->move(10, 10);
  connect(list, SIGNAL(rightButtonHighlighted(int, QPoint)), this, SLOT(slotRightPopupMenu(int, QPoint)));
  connect(list, SIGNAL(leftButtonHighlighted(int, QPoint)), this, SLOT(slotLeftPopupMenu(int, QPoint)));
      
  // Create userID popup menu
  userPopup = new QPopupMenu;
    userPopup->insertItem("User ID", USERID); 
    userPopup->setItemEnabled(USERID, FALSE); 
    userPopup->insertSeparator();
    userPopup->insertItem("Edit", USERID_EDIT); 
    userPopup->setItemEnabled(USERID_EDIT, FALSE); 
    userPopup->insertItem("Delete", USERID_DELETE); 
    userPopup->setItemEnabled(USERID_DELETE, FALSE); 
    connect (userPopup, SIGNAL(activated(int)),SLOT(slotUserPopup(int)));
   
  
  // Create signature popup menu
  sigPopup = new QPopupMenu;
    sigPopup->insertItem("Signature", SIG); 
    sigPopup->setItemEnabled(SIG, FALSE); 
    sigPopup->insertSeparator();
    sigPopup->insertItem("Edit", SIG_EDIT); 
    sigPopup->setItemEnabled(SIG_EDIT, FALSE); 
    sigPopup->insertItem("Delete", SIG_DELETE); 
    sigPopup->setItemEnabled(SIG_DELETE, FALSE); 
    connect (sigPopup, SIGNAL(activated(int)),SLOT(slotSigPopup(int)));
    
    loadSettings();
}

void KPGPKeyring::resizeEvent(QResizeEvent *)
{
  list->resize(width()-20, height()-20);
}

void KPGPKeyring::slotKeyPopup(int menu)
{
  KPGPKey *key;
  
  switch (menu)
    {
      case 15:  key = (KPGPKey*) list->itemAt(0);
		// delete old default key settings
		while (key != 0)
		  {
		    if (key->priv() && key->getDefault())
		      key->setDefault(FALSE);
		    key = (KPGPKey*) key->getSibling();
		  }
	        key = (KPGPKey*) list->getCurrentItem();
		MYNAME = key->userID();
		key->setDefault(TRUE);
		list->update();
		break;
    }
};

void KPGPKeyring::slotUserPopup(int)
{
  // cout << "User Menu selected: " << menu << endl;
};

void KPGPKeyring::slotSigPopup(int)
{
  // cout << "Sig Menu selected: " << menu << endl;
};

void KPGPKeyring::slotRightPopupMenu(int index, QPoint pos)
{
  KPGPKey       *key;
  
  switch (list->itemAt(index)->getBranch())
    {
      case 0: key = (KPGPKey*) list->itemAt(index);
	      if (key->priv())
		{
		  seckeyRightPopup->move(list->mapToGlobal(pos));
		  seckeyRightPopup->show();
		}
	      else
		{
		  keyRightPopup->move(list->mapToGlobal(pos));
		  keyRightPopup->show();
		}
	      break;
      case 1: userPopup->move(list->mapToGlobal(pos));
	      userPopup->show();
	      break;
      case 2: sigPopup->move(list->mapToGlobal(pos));
	      sigPopup->show();
	      break;
    }
};

void KPGPKeyring::slotLeftPopupMenu(int index, QPoint pos)
{
  KPGPKey       *key;
  
  switch (list->itemAt(index)->getBranch())
    {
      case 0: key = (KPGPKey*) list->itemAt(index);
	      keyLeftPopup->changeItem("Bitlen:\t" + key->bitlen(), 2);
	      keyLeftPopup->changeItem("Key ID:\t" + key->keyID(), 3);
	      keyLeftPopup->changeItem("Created:\t" + key->created(), 4);
	      keyLeftPopup->changeItem("Fingerprint:\t" + key->fingerPrint(), 5);
	      keyLeftPopup->move(list->mapToGlobal(pos));
	      keyLeftPopup->show();
	      break;
      case 1: break;
      case 2: break;
    }
};

bool KPGPKeyring::openConfig(QString name)
{
  QFileInfo finfo(name);
  
  if (finfo.isReadable() && finfo.isWritable())
    {
      config = new KSimpleConfig(name);
      config_name = name;
      
      return TRUE;
    }
  else
    {
      KMsgBox::message(this, "PGP Keyring Error", "Can't read/write config file " + name);
      
      return FALSE;
    }	
}

void KPGPKeyring::applySettings()
{
  if (config_found)
    {
      config_found = openConfig(config_name);
	
      config->writeEntry("MYNAME", MYNAME);
      config->sync();
      
      delete config;
    }
  
  
  list->clear();
  loadSettings();
}

void KPGPKeyring::loadSettings( void )
{ 
  QString          PUBRING;
  QString          pgpPath;
  QFile            file;
  QDir             pgp;
  
  config_found = FALSE;
  
  // find pgp configuration file
    pgpPath = getenv("PGPPATH");
  
  // first test $PGPPATH 
    if ((pgpPath != NULL) && (pgpPath != ""))
      {
	if (pgp.cd(pgpPath))       // PGPPATH exists
	  config_found = openConfig(pgp.absPath()+"/config.txt");
	else
	  KMsgBox::message(this, "PGP Config Error", "$PGPPATH directory does not exist!");
      }
    else
      {
	if (pgp.exists(QDir::homeDirPath() + "/.pgprc"))
	  {
	    config_found = openConfig(QDir::homeDirPath() + "/.pgprc");  // open .pgprc
	    
	    pgp.cd(QDir::homeDirPath() + "/.pgp");        // set pgp_dir to ~/.pgp
	  }
	else
	  {
	    if (pgp.cd(QDir::homeDirPath() + "/.pgp"))
	      config_found = openConfig(pgp.absPath()+"/config.txt");
	    else
	      KMsgBox::message(this, "PGP Keyring Error", "No PGP config file found!");
	  }
      }
    
      
  if (config_found)
    {
      PUBRING = config->readEntry("PUBRING");
      if (PUBRING == NULL)
	PUBRING = pgp.absPath() + "/pubring.pgp";
      
      MYNAME = config->readEntry("MYNAME");
      
      file.setName(PUBRING);
      
      if (!file.exists())
	{
	  KMsgBox::message(this, "PGP Keyring Error", "Could not find pubring.pgp.\nSelect pubring.pgp first!");
	  config_found = FALSE;
	}
      else
	readKeyList(file);  
      
      delete config;
    }
}

void KPGPKeyring::readKeyList(QFile &file)
{
  QFileInfo        finfo;
  QDataStream      keyring;
  	  
  KPGPKey          *key;
  KTreeListItem    *keyItem;
  KTreeListItem    *userID;
  KTreeListItem    *signature;
  QString          keyID;
 
  finfo.setFile(file);
  
  if (finfo.isReadable())
    { 
      file.open( IO_ReadOnly );
      
      keyring.setDevice(&file);
      
      // parse pubring first time to read 
      // keys, userIDs, signatures
      while (!keyring.eof())
	{
	  key = new KPGPKey(keyring);
	  keyDict.insert(key->keyID(), key);
	  
	  if (key->priv())
	    {
	      if ((MYNAME == "") || (MYNAME == NULL) || 
		   (key->userID().find(MYNAME) != -1))
		{
		  MYNAME = key->userID();
		  key->setDefault(TRUE);
		} 
	    }
	  
	  list->insertItem(key);
	}
      
      file.close();
      
      // read keyList the second time to change KeyIDs
      // into real userID names
      keyItem = list->itemAt(0);
      while (keyItem != 0)
	{
	  userID = keyItem->childAt(0);
	  while (userID != 0)
	    {
	      signature = userID->childAt(0);
	      
	      while (signature != 0)
		{
		  keyID = signature->getText();
		  key = keyDict.find(keyID);
		  
		  if (key != NULL)
		    signature->setText(keyDict.find(keyID)->userID());
		  else
		    signature->setText(keyID + "  (Unknown signator, can't be checked)");
		  
		  signature = signature->getSibling();
		}
	      
	      userID = userID->getSibling();
	    }
	  
	  keyItem = keyItem->getSibling();
	}
    }
}
  
#include "keyring.moc"