//==========================================================================
//
//   View.cpp
//
//==========================================================================

#include <math.h>
#include <stdlib.h>

#include <qapp.h>
#include <qcursor.h>
#include <qpainter.h>

#include <kdebug.h>

#include "view.h"
#include "kmag.h"

//--------------------------------------------------------------------------
//   Construction
//--------------------------------------------------------------------------

View::View(QWidget *parent) :
    QFrame(parent),
    zoom(1),
    selRect(0, 0, 250, 120),
    mouseMode(Normal)
{
    setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
    setLineWidth(0);
    setBackgroundColor(QColor("white"));

    connect(&selRect, SIGNAL(updated()), SLOT(slotUpdatePixmap()));
    connect(&timer,   SIGNAL(timeout()), SLOT(slotUpdatePixmap()));

    timer.start(100);
}
//--------------------------------------------------------------------------
//   Events
//--------------------------------------------------------------------------

void View::mousePressEvent (QMouseEvent *e)
{
    QPoint p = mapToGlobal(e->pos());

    switch(e->button() + mouseMode)
    {
    case 1 + Normal:
	
	mouseMode = MoveSelection2;
	grabMouse(crossCursor);
	
	selRect.moveBottomRight(p);
	selRect.show();

	break;

    case 1 + StartSelect:

	mouseMode = ResizeSelection1;	

	selRect.moveTopLeft(p);
	selRect.setSize(QSize(0, 0)); 
	selRect.show();

	break;

    case 2 + ResizeSelection1:

	mouseMode = MoveSelection1;
	break;

    case 2 + MoveSelection2:
	
	mouseMode = ResizeSelection2;
	break;

    case 2 + MoveSelection3:
	
	mouseMode = ResizeSelection3;
	oldPoint = mapToGlobal(e->pos());
	oldSize = selRect.size();
	break;

    case 4 + Normal:
	mouseMode = MoveSelection3;
	grabMouse(sizeAllCursor);

	oldPoint =  mapToGlobal(e->pos());
	oldPos = selRect.topLeft();
	selRect.show();
	break;

    default:

	; // do nothing
    }
}

void View::mouseMoveEvent (QMouseEvent *e)
{
    QPoint p = mapToGlobal(e->pos());

    switch(mouseMode)
    {
    case MoveSelection1:
    case MoveSelection2:
	
	selRect.moveBottomRight(p);
	selRect.update();

	break;

    case MoveSelection3:

	selRect.moveTopLeft(oldPos - (p - oldPoint)/zoom);
	selRect.update();

	break;

    case ResizeSelection1:
    case ResizeSelection2:

	selRect.setRight(p.x());
	selRect.setBottom(p.y());
	
	selRect.update();

	break;

    case ResizeSelection3:
    {
	QPoint sizeDiff(oldPoint - p);
	QSize newSize(oldSize.width() + sizeDiff.x(),
		      oldSize.height() + sizeDiff.y());
	
	selRect.setSize(newSize);
	
	selRect.update();


	break;
    }
    default:

	; // do nothing
    }
}

void View::mouseReleaseEvent (QMouseEvent *e)
{
    switch(e->button() + mouseMode)
    {
    case 1 + MoveSelection1:
    case 1 + MoveSelection2:
    case 1 + ResizeSelection1:
    case 1 + ResizeSelection2:
    case 4 + MoveSelection3:
    case 4 + ResizeSelection3:

	mouseMode = Normal;
	    
	selRect.hide();
	releaseMouse();

	break;

    case 2 + MoveSelection1:
	
	mouseMode = ResizeSelection1;
	break;

    case 2 + ResizeSelection2:
	
	mouseMode = MoveSelection2;
	break;

    case 2 + ResizeSelection3:
	
	mouseMode = MoveSelection3;
	oldPoint = mapToGlobal(e->pos());
	oldPos = selRect.topLeft();
	break;
	
    default:
	
	; // do nothing
    }
}

void View::paintEvent(QPaintEvent *e)
{
    SelRectHider hider(selRect);
    QFrame::paintEvent(e);
    
    QRect pRect(pixmapRect());
    QRect uRect(e->rect());
    QRect wRect(contentsRect());
    
    QPainter p;
    p.begin(this);
    p.setClipRect(wRect);

    QRect r;
    
    // top
    
    r = wRect;
    r.setBottom(pRect.top()-1);
    r = r.intersect(uRect);
    if(!r.isEmpty())
	p.eraseRect(r);

    // bottom
    
    r = wRect;
    r.setTop(pRect.bottom()+1);
    r = r.intersect(uRect);
    if(!r.isEmpty())
	p.eraseRect(r);

    // left
    
    r = wRect;
    r.setTop(pRect.top());
    r.setBottom(pRect.bottom());
    r.setRight(pRect.left()-1);
    r = r.intersect(uRect);
    if(!r.isEmpty())
	p.eraseRect(r);

    // right
    
    r = wRect;
    r.setTop(pRect.top());
    r.setBottom(pRect.bottom());
    r.setLeft(pRect.right()+1);
    r = r.intersect(uRect);
    if(!r.isEmpty())
	p.eraseRect(r);

    // Pixmap

    p.drawPixmap(pRect.x(), pRect.y(), pixmap);
    p.end();
}

//--------------------------------------------------------------------------
//   Slots
//--------------------------------------------------------------------------

void View::slotUpdatePixmap()
{
    QRect r = pixmapRect();

    QRect sr(selRect.normalize());
    QRect grabRect(sr.left(), sr.top(),
		   min(sr.width(),  (width()  + zoom-1)/zoom),
		   min(sr.height(), (height() + zoom-1)/zoom)
		   );

    pixmap = QPixmap::grabWindow(QApplication::desktop()->winId(),
				 grabRect.x(), grabRect.y(),
				 grabRect.width(), grabRect.height());
    // magnify
    
    QWMatrix m;
    m.scale(zoom, zoom);
    
    pixmap = pixmap.xForm(m);

    // update window
    
    repaint(r, false);
}

void View::slotSetZoom(int _zoom)
{
    if(zoom != _zoom)
    {
	zoom = _zoom;
	slotUpdatePixmap();

	emit zoomChanged(zoom);
    }
}

void View::slotIncZoom()
{
    slotSetZoom(zoom+1);
}

void View::slotDecZoom()
{
    if(zoom > 1)
	slotSetZoom(zoom-1);
}

void View::slotReplace()
{
    grabMouse(crossCursor);
    mouseMode = StartSelect;
}

//--------------------------------------------------------------------------
//   Utilities
//--------------------------------------------------------------------------

QPoint View::pixmapPosition()
{
    int free_x = width()  - pixmap.width();
    int free_y = height()  - pixmap.height();

    return QPoint((free_x > 0) ? (free_x / 2) : 0,
		  (free_y > 0) ? (free_y / 2) : 0);
}

QRect View::pixmapRect()
{
    QRect r(pixmap.rect());
    r.moveTopLeft(pixmapPosition());
    return r;
}

//--------------------------------------------------------------------------
//   end of file
//--------------------------------------------------------------------------

