/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <netdb.h>
#include <stdarg.h>

#include "server.h"
#include "inter.h"


#define i18n(s)	s


static char short_hostname_[100] = "darkstar";
const char * const short_hostname = short_hostname_;
static char fq_hostname_[100] = "darkstar";
const char * const fq_hostname = fq_hostname_;


static int busy = 0;

static const struct server *server = NULL;


struct child_pipe {
	int fd;
	char *buf;
	int buf_size;
	int buf_len;
};

static struct {
	pid_t pid;
	int status;
	struct child_pipe out;
	struct child_pipe err;
} child = { 0, 0, { -1, NULL, 0, 0 }, { -1, NULL, 0, 0 } };


void iwrite_fail(const char *s)
{
	iwrite_char('f');
	iwrite_string(s);
}


void *xmalloc(size_t size)
{
	void *ptr;

	if (!(ptr = malloc(size))) {
		errno = ENOMEM;
		perror("klp: xmalloc");
		exit(EXIT_FAILURE);
	}
	return ptr;
}


void *xrealloc(void *ptr, size_t size)
{
	if (!(ptr = realloc(ptr, size))) {
		errno = ENOMEM;
		perror("klp: xrealloc");
		exit(EXIT_FAILURE);
	}
	return ptr;
}


char *xstrdup(const char *s)
{
	char *dup;
	if (!(dup = strdup(s))) {
		errno = ENOMEM;
		perror("klp: xstrdup");
		exit(EXIT_FAILURE);
	}
	return dup;
}


/*
 *	This thing is also registered with atexit(), so we use abort() on
 *	failure.
 */
static void cleanup_child(void)
{
	if (child.pid) {
		if (kill(child.pid, SIGKILL) == -1) {
			perror("klp: kill");
			abort();
		}
		if (waitpid(child.pid, NULL, 0) != child.pid) {
			perror("klp: waitpid");
			abort();
		}
		child.pid = 0;
	}
	if (child.out.fd != -1) {
		close(child.out.fd);
		child.out.fd = -1;
	}
	if (child.err.fd != -1) {
		close(child.err.fd);
		child.err.fd = -1;
	}
	xfree(child.out.buf);
	xfree(child.err.buf);
	busy = 0;
}


/*
 *	spawn_child
 *
 *	This acts like a popen() call.
 *
 */
static int spawn_child(char a, ...)
{
	int fd_out[2];
	int fd_err[2];

	if ((pipe(fd_out) == -1) || (pipe(fd_err) == -1)) {
		perror("klp: pipe");
		goto fail;
	}
	if ((child.pid = fork()) == -1) {
		perror("klp: fork");
		close(fd_out[0]);
		close(fd_out[1]);
		close(fd_err[0]);
		close(fd_err[1]);
		child.pid = 0;
		goto fail;
	}
	if (!child.pid) {
		if ((dup2(fd_out[1], 1) == -1) || (dup2(fd_err[1], 2) == -1))
			exit(EXIT_FAILURE);
		close(fd_out[0]);
		close(fd_out[1]);
		close(fd_err[0]);
		close(fd_err[1]);
		switch (a) {
			case 'q':
				exit(server->q_main());
			case 'r': {
				va_list ap;
				const char **jobids;
				va_start(ap, a);
				jobids = va_arg(ap, const char **);
				va_end(ap);
				exit(server->r_main(jobids));
			}
			case 'p': {
				va_list ap;
				const char **files;
				va_start(ap, a);
				files = va_arg(ap, const char **);
				va_end(ap);
				exit(server->p_main(files));
			}
		}
		abort();
	}
	close(fd_out[1]);
	close(fd_err[1]);
	child.out.fd = fd_out[0];
	child.err.fd = fd_err[0];
	busy = a;
	return 1;
fail:
	iwrite_fail(i18n("Could not start child process"));
	return 0;
}


static void set_printer(void)
{
	int type;
	char *printer;
	char *host = NULL;
	int local;
	char *fail;

	if (server) {
		server->set_printer(NULL, NULL);
		server = NULL;
	}
	iread_int(&type);
	iread_string(&printer);
	iread_int(&local);
	if (!local)
		iread_string(&host);
	if ((type < 0) || (type >= SERVER_COUNT)) {
		iwrite_fail(i18n("Invalid server type"));
		goto cleanup;
	}
	server = servers + type;
	if (!(local ? server->local : server->remote)) {
		iwrite_fail(i18n("Invalid server type"));
		server = NULL;
		goto cleanup;
	}
	if ((fail = server->set_printer(host, printer))) {
		iwrite_fail(fail);
		free(fail);
		server = NULL;
		goto cleanup;
	}
	iwrite_char('s');

cleanup:
	free(printer);
	free(host);
}


static void read_frontend(void)
{
	char c;

	iread_char(&c);
	if (!strchr("asprq", c)) {
		fprintf(stderr, "klp: frontend out of sync\n");
		exit(EXIT_FAILURE);
	}
	if (busy && (c != 'a')) {
		fprintf(stderr, "klp: server got command while busy\n");
		exit(EXIT_FAILURE);
	}
	if (!server && (c != 's')) {
		iwrite_fail(i18n("No printer selected"));
		return;
	}
		
	switch (c) {
		case 'a':
			cleanup_child();
			iwrite_char('a');
			break;
		case 's':
			set_printer();
			break;
		case 'q':
			spawn_child('q', NULL);
			break;
		case 'r': {
			char **jobids;
			int count;
			int i;
			iread_int(&count);
			jobids = xmalloc((count + 1) * sizeof(char *));
			for ( i = 0 ; i < count ; ++i)
				iread_string(&jobids[i]);
			jobids[count] = NULL;
			spawn_child('r', jobids);
			for ( i = 0 ; i < count ; ++i)
				xfree(jobids[i]);
			xfree(jobids);
			break;
		}
		case 'p': {
			char **files;
			int count;
			int i;
			
			iread_int(&count);
			files = xmalloc((count + 1) * sizeof(char *));
			for ( i = 0 ; i < count ; ++i)
				iread_string(&files[i]);
			files[count] = NULL;
			spawn_child('p', files);
			for ( i = 0 ; i < count ; ++i)
				xfree(files[i]);
			xfree(files);
			break;
		}
	}
}


static void queue_to_frontend(struct queue *queue)
{
	int i;

	iwrite_char('q');
	iwrite_int(queue->no_print);
	iwrite_int(queue->no_queue);
	iwrite_int(queue->item_count);
	for (i = 0 ; i < queue->item_count ; ++i) {
		iwrite_int(queue->items[i]->rank);
		iwrite_string(queue->items[i]->job_name);
		iwrite_string(queue->items[i]->user);
		iwrite_string(queue->items[i]->job_id);
		iwrite_int(queue->items[i]->size);
	}
}


static void handle_child()
{
	if (!busy || child.pid || (child.out.fd != -1) || (child.err.fd != -1))
		return;
	fprintf(stderr, "%s", child.err.buf);
	if (!WIFEXITED(child.status)) {
		if (WIFSIGNALED(child.status))
			fprintf(stderr, "klp: child got signal %d\n", WTERMSIG(child.status));
		iwrite_fail(i18n("Child process ended abnormally"));
	} else {
		int r = WEXITSTATUS(child.status);
		switch (busy) {
			case 'q': {
				struct queue *queue;
				if ((queue = server->q_parse(child.out.buf, child.err.buf, r))) {
					queue_to_frontend(queue);
					free_queue(queue);
				}
				break;
			}
			case 'r':
				server->r_parse(NULL, child.out.buf, child.err.buf, r);
				break;
			case 'p':
				server->p_parse(NULL, child.out.buf, child.err.buf, r);
				break;
			default:
				exit(EXIT_FAILURE);
		}
	}
	cleanup_child();
}


static void read_child(struct child_pipe *cp)
{
	int r;

	if (!cp->buf) {
		cp->buf = xmalloc(1024);
		cp->buf[0] = '\0';
		cp->buf_size = 1024;
		cp->buf_len = 0;
	} else if (cp->buf_len == cp->buf_size - 1) {
		cp->buf_size *= 2;
		cp->buf = xrealloc(cp->buf, cp->buf_size);
	}
	r = read(cp->fd, cp->buf + cp->buf_len, cp->buf_size - cp->buf_len - 1);
	switch(r) {
		case 0:
			close(cp->fd);
			cp->fd = -1;
			handle_child();
			break;
		case -1:
			perror("klp: read");
			iwrite_fail(i18n("Communication error with printer server"));
			cleanup_child();
			break;
		default:
			cp->buf_len += r;
			cp->buf[cp->buf_len] = '\0';
	}
}


static volatile int max_fd;


static void handle_SIGCHLD(int signum)
{
	max_fd = -1;
}


int server_main(void)
{
	struct utsname utsn;
	struct hostent *he;
	struct sigaction act;
	sigset_t sig_child;

	tmp_drop_root();
	if (atexit(cleanup_child)) {
		perror("klp: atexit");
		return EXIT_FAILURE;
	}

	sigemptyset(&act.sa_mask);

	act.sa_handler = handle_SIGCHLD;
	act.sa_flags = SA_NOCLDSTOP;
	sigaction(SIGCHLD, &act, NULL);

	act.sa_handler = SIG_IGN;
	act.sa_flags = 0;
	sigaction(SIGPIPE, &act, NULL);

	sigemptyset(&sig_child);
	sigaddset(&sig_child, SIGCHLD);
	if (sigprocmask(SIG_BLOCK, &sig_child, NULL) == -1) {
		perror("klp: sigprocmask");
		return EXIT_FAILURE;
	}

	if (uname(&utsn))
		perror("klp: uname");
	else {
		strncpy(short_hostname_, utsn.nodename, 99);
		short_hostname_[99] = '\0';
		if (!(he = gethostbyname(utsn.nodename)))
			perror("klp: gethostbyname");
		else {
			strncpy(fq_hostname_, he->h_name, 99);
			fq_hostname_[99] = '\0';
		}
	}

	while (1) {
		fd_set readfds;

		FD_ZERO(&readfds);
		FD_SET(pipe_r, &readfds);
		max_fd = pipe_r;
		if (child.out.fd != -1) {
			FD_SET(child.out.fd, &readfds);
			if (child.out.fd > max_fd)
				max_fd = child.out.fd;
		}
		if (child.err.fd != -1) {
			FD_SET(child.err.fd, &readfds);
			if (child.err.fd > max_fd)
				max_fd = child.err.fd;
		}
		++max_fd;
		if (sigprocmask(SIG_UNBLOCK, &sig_child, NULL) == -1) {
			perror("klp: sigprocmask");
			return EXIT_FAILURE;
		}
		if (select(max_fd, &readfds, NULL, NULL, NULL) == -1) {
			if ((errno != EINVAL) && (errno != EINTR)) {
				perror("klp: select");
				return EXIT_FAILURE;
			}
			FD_ZERO(&readfds);
		}
		if (sigprocmask(SIG_BLOCK, &sig_child, NULL) == -1) {
			perror("klp: sigprocmask");
			return EXIT_FAILURE;
		}
		if (child.pid) {
			int r = waitpid(child.pid, &child.status, WNOHANG);
			switch(r) {
				case -1:
					perror("klp: waitpid");
					return EXIT_FAILURE;
				case 0:
					break;
				default:
					child.pid = 0;
					handle_child();
			}
		}
		if (FD_ISSET(pipe_r, &readfds))
			read_frontend();
		if ((child.out.fd != -1) && FD_ISSET(child.out.fd, &readfds))
			read_child(&child.out);
		if ((child.err.fd != -1) && FD_ISSET(child.err.fd, &readfds))
			read_child(&child.err);
	}
}
