/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include <netdb.h>
#include <limits.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <errno.h>

#include "lpr_remote.h"
#include "server.h"
#include "inter.h"


#define i18n(s) s


static struct sockaddr_in host_addr;
static char *printer;

static char *sequence_file = SEQUENCE_FILE;
static int local_sequence = -1;


static int local_next_sequence(void)
{
	if (++local_sequence == 1000)
		local_sequence = 1;
	return local_sequence;
}


static int next_sequence(void)
{
	FILE *f;
	int seq;

	if (!sequence_file)
		return local_next_sequence();
	tmp_as_root();
	if (!(f = fopen(sequence_file, "r+"))) {
		perror("klp: fopen");
		if (!(f = fopen(sequence_file, "w+"))) {
			tmp_drop_root();
			perror("klp: fopen");
			fprintf(stderr, "klp: Warning: Using private sequence\n");
			sequence_file = NULL;
			return local_next_sequence();
		}
		seq = local_next_sequence();
	} else if ((fscanf(f, "%d", &seq) <= 0) || (seq < 1 || seq > 999)) {
		fprintf(stderr, "klp: Warning: Corrupt sequence file\n");
		seq = local_next_sequence();
	}
	tmp_drop_root();
	if (++seq == 1000)
		seq = 1;
	if (fseek(f, 0, SEEK_SET) == -1) {
		perror("klp: fseek");
		fprintf(stderr, "klp: Warning: Using private sequence\n");
		fclose(f);
		sequence_file = NULL;
		return local_next_sequence();
	}
	if (fprintf(f, "%03d\n", seq) != 4) {
		fprintf(stderr, "klp: Warning: Using private sequence\n");
		fclose(f);
		sequence_file = NULL;
		return local_next_sequence();
	}
	if (fclose(f)) {
		perror("klp: fclose");
		fprintf(stderr, "klp: Warning: Using private sequence\n");
		sequence_file = NULL;
		return local_next_sequence();
	}
	return seq;
}


char *lpr_remote_set_printer(const char *host, const char *p)
{
	struct hostent *he;
	struct servent *se;

	if (local_sequence < 0) {
		srand(getpid());
		local_sequence = 1 + (int)(999.0 * rand() / (RAND_MAX + 1.0));
	}
	xfree(printer);
	if (!host)
		return xstrdup(i18n("Local printers not supported by this server type"));
	if (!(he = gethostbyname(host))
			|| (he->h_addrtype != AF_INET)
			|| (he->h_length != sizeof(host_addr.sin_addr))) {
		if (!he)
			herror("klp: gethostbyname");
		else
			fprintf(stderr, "klp: gethostbyname: no internet address to host\n");
		return xstrdup(i18n("Unable to find host"));
	}
	memcpy(&host_addr.sin_addr, he->h_addr_list[0], he->h_length);
	host_addr.sin_family = AF_INET;
	if (!(se = getservbyname("printer", "tcp"))) {
		fprintf(stderr, "klp: getservbyname: cannot get port number for service `printer' (using 515)\n");
		host_addr.sin_port = htons(515);
	} else
		host_addr.sin_port = se->s_port;
	printer = xstrdup(p);
	return NULL;
}
	

/*
 *	connected_socket
 *
 *	returns a file descriptor to a connected socket
 *	bound to a privileged port (721..731).
 *	exit(EXIT_FAILURE) on failure (sic).
 *
 */
static int connected_socket(void)
{
	int s;
	int i;
	struct sockaddr_in own_addr;
	struct protoent *pe;

	if (!(pe = getprotobyname("tcp"))) {
		fprintf(stderr, "klp: getprotobyname: cannot get protocol number for `tcp'\n");
		goto fail;
	}
	if ((s = socket(AF_INET, SOCK_STREAM, pe->p_proto)) == -1) {
		perror("klp: socket");
		goto fail;
	}
	own_addr.sin_family = AF_INET;
	own_addr.sin_addr.s_addr = INADDR_ANY;
	/*
	 *	as root !!!
	 */
	if (!tmp_as_root()) {
		fprintf(stderr, "klp: root privileges required\n");
		goto fail;
	}
	for (i = 721 ; i <= 731 ; ++i) {
		own_addr.sin_port = htons(i);
		if (bind(s, (struct sockaddr *)&own_addr, sizeof(own_addr)) == -1) {
			if (errno == EADDRINUSE)
				continue;
			perror("klp: bind");
			goto fail;
		} else
			break;
	}
	tmp_drop_root();
	/*
	 *	no more as root !!!
	 */
	if (i > 731) {
		fprintf(stderr, "klp: no free socket\n");
		goto fail;
	}
	if (connect(s, (struct sockaddr *)&host_addr, sizeof(struct sockaddr_in)) == -1) {
		perror("klp: connect");
		goto fail;
	}
	return s;

fail:
	fprintf(stderr, "klp: cannot connect to host\n");
	exit(EXIT_FAILURE);
}


static void xwrite(int fd, const void *buf, size_t count)
{
	int w;

	if ((w = write(fd, buf, count)) != count) {
		if (w == -1)
			perror("klp: write");
		exit(EXIT_FAILURE);
	}
}


int lpr_remote_q_main(void)
{
	char buf[1024];
	ssize_t r;
	int s;

	tmp_drop_root();
	s = connected_socket();
	drop_root();
	xwrite(s, "\003", 1);
	xwrite(s, printer, strlen(printer));
	xwrite(s, " \n", 2);
	while ((r = read(s, buf, 1024)) > 0)
		xwrite(1, buf, r);
	if (r == -1) {
		perror("klp: read");
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}


#define MAX_CLINE (1 + PATH_MAX + 1)


static char *cfile = NULL;
static int cfile_len;
static int cfile_size;


static void cprintf(const char *format, ...)
{
	va_list ap;
	int r;

	if (!cfile) {
		cfile_size = MAX_CLINE + 1;
		cfile = xmalloc(cfile_size);
		cfile_len = 0;
		cfile[0] = '\0';
	} else if (cfile_len + MAX_CLINE >= cfile_size - 1) {
		cfile_size *= 2;
		cfile = xrealloc(cfile, cfile_size);
	}
	va_start(ap, format);
#ifdef HAVE_VSNPRINTF
	if ((r = vsnprintf(cfile + cfile_len, MAX_CLINE + 1, format, ap)) < 0 || (r >= MAX_CLINE + 1))
		abort();
#else
	if ((r = vsprintf(cfile + cfile_len, format, ap)) < 0 || (r >= MAX_CLINE + 1))
		abort();
#endif
	va_end(ap);
	cfile_len += r;
}


int lpr_remote_p_main(const char * const files[])
{
	char buf[1024];
	int r;
	char fail;
	char d = 'd';
	char A = 'A';
	int count = 0;
	int s;
	int seq;

	tmp_drop_root();
	s = connected_socket();

	sprintf(buf, "\002%s\n", printer);
	xwrite(s, buf, strlen(buf));
	if ((read(s, &fail, 1) != 1) || fail)
		return EXIT_FAILURE;

	seq = next_sequence();
	drop_root();

	cprintf("H%.131s\n", fq_hostname);
	cprintf("P%.31s\n", username);
	cprintf("C%.131s\n", fq_hostname);
	cprintf("L%.31s\n", username);

	for ( ; *files ; ++files) {
		int fd = -1;
		int len = 0;
		struct stat st;

		if (((fd = open(*files, O_NOCTTY | O_RDONLY)) == -1)
				|| fstat(fd, &st)
				|| !S_ISREG(st.st_mode)) {
			if (fd != -1)
				close(fd);
			fprintf(stdout, "lpr: cannot access %s\n", *files);
			continue;
		}
		if (st.st_size == 0) {
			close(fd);
			fprintf(stdout, "lpr: %s is an empty file\n", *files);
			continue;
		}
		sprintf(buf, "\003%lu %cf%c%03d%.131s\n", (unsigned long)st.st_size, d, A, seq, fq_hostname);
		if (!count) {
			char *jn;
			jn = strrchr(*files, '/');
			cprintf("J%.99s\n", jn ? jn + 1 : *files);
		}
		cprintf("f%cf%c%03d%.131s\n", d, A, seq, fq_hostname);
		cprintf("U%cf%c%03d%.131s\n", d, A, seq, fq_hostname);
		cprintf("N%.131s\n", *files);
		xwrite(s, buf, strlen(buf));
		if ((read(s, &fail, 1) != 1) || fail)
			return EXIT_FAILURE;
		while ((r = read(fd, buf, 1024)) > 0) {
			xwrite(s, buf, r);
			len += r;
		}
		close(fd);
		if (r || (len != st.st_size))
			return EXIT_FAILURE;
		xwrite(s, "", 1);
		if ((read(s, &fail, 1) != 1) || fail)
			return EXIT_FAILURE;
		if (A == 'Z')
			A = 'a';
		else if (A == 'z') {
			A = 'A';
			if (d == 's') {
				fprintf(stdout, "too many files - break up the job\n");
				return EXIT_FAILURE;
			}
			++d;
		} else
			++A;
		++count;
	}
	if (count) {
		sprintf(buf, "\002%d cfA%03d%.131s\n", cfile_len, seq, fq_hostname);
		xwrite(s, buf, strlen(buf));
		if ((read(s, &fail, 1) != 1) || fail)
			return EXIT_FAILURE;
		xwrite(s, cfile, cfile_len);
		xwrite(s, "", 1);
		if ((read(s, &fail, 1) != 1) || fail)
			return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}


int lpr_remote_r_main(const char * const jobids[])
{
	char buf[1024];
	int r;
	int s;

	tmp_drop_root();
	s = connected_socket();
	drop_root();

	xwrite(s, "\005", 1);
	xwrite(s, printer, strlen(printer));
	xwrite(s, " ", 1);
	xwrite(s, username, strlen(username));
	while (*jobids) {
		xwrite(s, " ", 1);
		xwrite(s, *jobids, strlen(*jobids));
		++jobids;
	}
	xwrite(s, "\n", 1);
	while ((r = read(s, buf, 1024)) > 0)
		xwrite(1, buf, r);
	return r ? EXIT_FAILURE : EXIT_SUCCESS;
}
