/*
 *            klm: A lm_config front end for the KDE project
 *
 * $Id: klm.cpp,v 1.68 1999/02/13 12:17:09 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <unistd.h>
#include <sys/sysctl.h>
#include <sys/wait.h>
#include <linux/sensors.h>
#include <linux/unistd.h>
#include <linux/sysctl.h>

#include <kapp.h>
#include <ktopwidget.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kmsgbox.h>
#include <kiconloader.h>
#include <kwm.h>
#include <klocale.h>
#include <kconfig.h>
#include <qaccel.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qmenubar.h>
#include <qregexp.h>

#include "VertGraph.h"
#include "klm.h"
#include "LMStats.h"
#include "StdTemp.h"
#include "StdVoltage.h"
#include "Lm80Voltage.h"
#include "MinMaxTemp.h"
#include "StdFan.h"
#include "StdVid.h"
#include "Item.h"
#include "Settings.h"

#include <iostream.h>
using namespace std;

// version info
#define KLM_MAJOR 0
#define KLM_MINOR 5
#define KLM_MINOR_MINOR 0

// number of chips, not sensors
#define MAX_SENSOR_CHIPS 20

// config file locations for main app settings.
#define KLM_APP_GROUP	"klm-application"
#define REFRESH_INTERVAL	"_REFRESH_INT"
#define ALARM_BEEP 		"_ALARM_BEEP"
#define ALARM_EXEC		"_ALARM_EXEC"
#define ALARM_SCRIPT		"_ALARM_SCRIPT"

// pointer to the dock window.
extern DockWidget *dock;

Klm::Klm
(
  QWidget* parent,
  const char* name
  )
:
KTopLevelWidget( name )
{
  dockDisplayedPtr = 0;
  numLm75Chips = 0;
  
  // read the klm settings
  getAppSettings();
  
  // create the settings dialog.
  settings = new Settings( this, "Settings" );
  settings->hide();	
  connect(
    settings, SIGNAL( newInterval( int )),
    this, SLOT( newUpdateInterval( int )) );
  
  connect(
    settings, SIGNAL( beepOnFault( bool ) ),
    this, SLOT( alarmBeepToggled( bool )) );
  
  connect(
    settings, SIGNAL( scriptOnFault( bool ) ),
    this, SLOT( alarmScriptToggled( bool )) );
  
  connect(
    settings, SIGNAL( okPressed() ),
    this, SLOT( hideSettings() ) );
  
  // create dock entity
  dock = new DockWidget( 0, "dock" );
  dock->dock();
  connect( dock, SIGNAL(clicked()), SLOT(maxOrMin()));
  connect( dock, SIGNAL(exit_cmd()), SLOT(saveQuit()));	
  connect( dock, SIGNAL(showSettings()), SLOT(showSettings()));
  
  // create file menu
  file = new QPopupMenu();
  
  file->insertItem( 
    "&Reset", 
    this, SLOT(reInitialise()), 
    ALT + Key_R);
  
  file->insertItem(
    "&Settings",
    this, SLOT(showSettings()),
    ALT + Key_S);
  
  file->insertSeparator();
  
  file->insertItem( 
    "E&xit", 
    this, SLOT(saveQuit()), 
    ALT + Key_F4 );
  
  // create about text and help menu
  QString s(512);	
  s.sprintf(
    klocale->translate("klm %d.%d.%d - A hardware health monitor that uses\n"
      "the lm_sensors kernel module for sensor access.\n\n"
      "(c) 1998, 1999 Brendon Humphrey (brendy@swipnet.se)\n\n" 
      "lm_sensors is written by Frodo Looijaard, Philip\n"
      "Edelbrock and Alexander Larsson\n"
      "and can be obtained from:\n\n"
      "     http://www.netroedge.com/~lm78\n\n"
      "This program is published under the GNU General\n" 
      "Public License (GNU GPL). See the file COPYING\n" 
      "for details about this."),
    KLM_MAJOR,
    KLM_MINOR,
    KLM_MINOR_MINOR	);
  
  help = kapp->getHelpMenu(TRUE, s.data());	
  
  kdemenu = new KMenuBar( this );
  kdemenu->insertItem("&File", file);
  kdemenu->insertSeparator();
  kdemenu->insertItem("&Help", help);
  
  // create toolbar
  il = kapp->getIconLoader();
  tb = new KToolBar(this);	
  tb->insertButton(il->loadIcon("exit.xpm"), 
    1,
    SIGNAL(clicked()),
    this,
    SLOT(saveQuit()),
    TRUE,
    "Quit");	
  
  tb->insertSeparator();
  
  tb->insertButton(il->loadIcon("reload.xpm"), 
    1,
    SIGNAL(clicked()),
    this,
    SLOT(reInitialise()),
    TRUE,
    "Reinitialise klm");	
  
  tb->insertSeparator();
  
  // prepare the status bar 
  statusBar = new KStatusBar(this);
  statusBar->insertItem("", 0);
  
  // probe for active sensors on the motherboard
  probeSensors( false );
  
  // add the central data display widget
  lmstats = new LMStats(this, "klm_guts", &machine, statusBar);
  
  connect(
    lmstats, SIGNAL( reloadMe() ),
    this, SLOT( sensorDisabled() ) );
  
  // icon stuff
  if(KWM::isKWMInitialized()) 
  {
    QPixmap pm = il->loadIcon( "klm.xpm" );
    KWM::setIcon( winId(), pm );
    
    pm = il->loadMiniIcon( "klm.xpm" );
    KWM::setMiniIcon( winId(), pm );
  }
  
  // add components to the toplevel window
  setMenu(kdemenu);
  addToolBar(tb);
  setStatusBar(statusBar);	
  setView(lmstats, false);	
  
  // turn on session management (does not seem to work)
  kapp->enableSessionManagement();
  connect( kapp, SIGNAL(saveYourself()), SLOT(save()));
  connect( kapp, SIGNAL(shutDown()), SLOT(saveQuit()));
  
  resize( 
    lmstats->width(),  
    tb->height() + 
    kdemenu->height() + 
    lmstats->height() + 
    statusBar->height() );
  
  setMinimumSize( width(), height() );
  
  updateDisplay();
  
  // arm the refresh timer
  myTimer = new QTimer( this );
  connect( myTimer, SIGNAL(timeout()), SLOT(timerExpired()));	
  myTimer->start( refreshInterval, false ); 
}	


Klm::~Klm()
{
  int i;
  
  myTimer->stop();
  delete myTimer;	
  delete kdemenu;
  delete tb;
  delete lmstats;
  
  // wipe out the sensors.
  save();
  for(i=0; i<machine.num_sensors; i++)
  {
    delete machine.sensors[i];
  }
}

bool Klm::testSensorPresence( int sysctl_addr2, int sysctl_addr3 )
{
  long 	val[10];
  size_t 	len = sizeof(val);
  int		address[4] = { 
    CTL_DEV, 
    DEV_SENSORS, 
    sysctl_addr2, 
    sysctl_addr3 };
  
  if	(sysctl( address, 4, (void *)val, &len, 0, 0)) 
  {		
    return ( false );
  }
  else 
  {	
    return ( true );
  }
}

void Klm::checkDisabled( int &idx, bool override )
{
  if (machine.sensors[idx]->isDisabled())
  {
    if (override)
    {
      machine.sensors[idx]->setDisabled(false);
      idx++;
    }
    else
    {
      delete machine.sensors[idx];
    }
  }
  else
  {
    idx++;
  }
}

void Klm::readSensorDirectory( 
  sensors_chips_data *buf, 
  int len, 
  int &num_sensor_chips )
{	
  size_t	buflen = len;
  int 	name[3] = { CTL_DEV, DEV_SENSORS, SENSORS_CHIPS };
  int 	res; 
  
  res = sysctl(name, 3, (void *)buf, &buflen, NULL, 0);
  num_sensor_chips = buflen/sizeof(sensors_chips_data);
}	

void Klm::addLm75( sensors_chips_data dirEntry, int &sensorCount, bool override )
{	
  if ( testSensorPresence( dirEntry.sysctl_id, LM75_SYSCTL_TEMP ) )
  {
    QString defaultName;
    
    defaultName.sprintf("Temp-%d", numLm75Chips++);
    machine.sensors[sensorCount]=new StdTemp( 
        CTL_DEV,
        DEV_SENSORS,	
        dirEntry.sysctl_id, 
        LM75_SYSCTL_TEMP,
        0, 
        50,
        defaultName.data());
    
    checkDisabled( sensorCount, override );		
  }	
}

void Klm::addLm78( sensors_chips_data dirEntry, int &sensorCount, bool override )
{
  int 	testSensor;
  char 	*defaultVoltNames[]= { 
    "VCore",
    "VCore2",
    "+3.3V",
    "+5V",
    "+12V",
    "-12V",
    "-5V"	};
  
  double 	defaultVoltConv[]= {
    1.0,
    1.0,
    1.0,
    1.6892,
    3.8,
    -3.4768,
    -1.5050 };
  
  if ( testSensorPresence( dirEntry.sysctl_id, LM78_SYSCTL_TEMP ) )
  {
    machine.sensors[sensorCount]=new StdTemp( 
        CTL_DEV,
        DEV_SENSORS,	
        dirEntry.sysctl_id, 
        LM78_SYSCTL_TEMP,
        0, 
        50,
        "Mbd" );
    
    checkDisabled( sensorCount, override );		
  }
  
  // probe for VID input 
  if ( testSensorPresence( dirEntry.sysctl_id, LM78_SYSCTL_VID ) )
  {
    machine.sensors[sensorCount]=new StdVid( 
        CTL_DEV, 
        DEV_SENSORS,
        dirEntry.sysctl_id, 
        LM78_SYSCTL_VID,
        0, 
        14	);		
    
    checkDisabled( sensorCount, override );		
  }
  
  //check the voltage inputs on the LM78	
  for( testSensor=LM78_SYSCTL_IN0; testSensor<=LM78_SYSCTL_IN6; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      machine.sensors[sensorCount]=new StdVoltage( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          14,
          defaultVoltNames[testSensor - LM78_SYSCTL_IN0],
          defaultVoltConv[testSensor - LM78_SYSCTL_IN0]	);		
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for fan sensors	
  for( testSensor=LM78_SYSCTL_FAN1; testSensor<=LM78_SYSCTL_FAN3; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      QString fanName;
      
      fanName.sprintf("Fan%d", testSensor - LM78_SYSCTL_FAN1 );
      
      machine.sensors[sensorCount]=new StdFan( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          7000,
          fanName.data()	);		
      
      checkDisabled( sensorCount, override );		
    }
  }	
}

void Klm::addLm79(  sensors_chips_data dirEntry, int &sensorCount, bool override ) 
{
  // until further notice an LM79 is an LM78...	
  addLm78( dirEntry, sensorCount, override );
}

void Klm::addSis5595(  sensors_chips_data dirEntry, int &sensorCount, bool override ) 
{
  // until further notice an SIS5595 is an LM78...	
  addLm78( dirEntry, sensorCount, override );
}

void Klm::addLm80( sensors_chips_data dirEntry, int &sensorCount, bool override )
{
  int 	testSensor;
  char 	*defaultVoltNames[]= { 
    "+5V",
    "VTT",
    "+3.3V",
    "+Vcore",
    "+12V",
    "-12V",
    "-5V"
  };
  
  double 	defaultVoltConv[]= {
    2.63265,
    1.0,		
    1.73667,
    2.47368,
    5.48179,
    4.48179,
    2.22222 };
  
  //compute in5 (160/35.7)*(@ - in0) + @, (@ + in0 * 160/25.7)/ (1 + 160/25.7)
  //compute in6 (36/16.2)*(@ - in0) + @,  (@ + in0 * 36/16.2) / (1 + 36/16.2)	
  
  // probe for temperature sensors
  if ( testSensorPresence( dirEntry.sysctl_id, LM80_SYSCTL_TEMP ) )
  {
    machine.sensors[sensorCount]=new StdTemp( 
        CTL_DEV,
        DEV_SENSORS,	
        dirEntry.sysctl_id, 
        LM78_SYSCTL_TEMP,
        0, 
        50,
        "Mbd" );
    
    checkDisabled( sensorCount, override );		
  }
  
  // probe for the standard voltage inputs
  for( testSensor=LM80_SYSCTL_IN0; testSensor<=LM80_SYSCTL_IN4; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      machine.sensors[sensorCount]=new StdVoltage( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          14,
          defaultVoltNames[testSensor - LM80_SYSCTL_IN0],
          defaultVoltConv[testSensor - LM80_SYSCTL_IN0]	);		
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for the differential inputs
  for( testSensor=LM80_SYSCTL_IN5; testSensor<=LM80_SYSCTL_IN6; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      machine.sensors[sensorCount]=new Lm80Voltage( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          LM80_SYSCTL_IN0,
          defaultVoltConv[0],
          0, 
          14,
          defaultVoltNames[testSensor - LM80_SYSCTL_IN0],
          defaultVoltConv[testSensor - LM80_SYSCTL_IN0]	);		
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for fans
  for( testSensor=LM80_SYSCTL_FAN1; testSensor<=LM80_SYSCTL_FAN2; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      QString fanName;
      
      fanName.sprintf("Fan%d", testSensor - LM80_SYSCTL_FAN1 );
      
      machine.sensors[sensorCount]=new StdFan( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          7000,
          fanName.data()	);		
      
      checkDisabled( sensorCount, override );		
    }
  }	
  
  
}

void Klm::addW83781d( sensors_chips_data dirEntry, int &sensorCount, bool override )
{
  int 	testSensor;
  char 	*defaultVoltNames[]= { 
    "VCore",
    "VCore2",
    "+3.3V",
    "+5V",
    "+12V",
    "-12V",
    "-5V"	};
  
  double 	defaultVoltConv[]= {
    1.0,
    1.0,
    1.0,
    1.6892,
    3.8,
    -3.4768,
    -1.5050 };
  
  // add the temperature sensors
  for( testSensor=W83781D_SYSCTL_TEMP1; testSensor<=W83781D_SYSCTL_TEMP3; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {
      QString name;
      
      name.sprintf("WTMP-%d", 1+testSensor-W83781D_SYSCTL_TEMP1);
      
      machine.sensors[sensorCount]=new StdTemp( 
          CTL_DEV,
          DEV_SENSORS,	
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          50,
          name.data() );
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for VID input 
  if ( testSensorPresence( dirEntry.sysctl_id, W83781D_SYSCTL_VID ) )
  {
    machine.sensors[sensorCount]=new StdVid( 
        CTL_DEV, 
        DEV_SENSORS,
        dirEntry.sysctl_id, 
        W83781D_SYSCTL_VID,
        0, 
        14	);		
    
    checkDisabled( sensorCount, override );		
  }
  
  //check the voltage inputs on the LM78	
  for( testSensor=W83781D_SYSCTL_IN0; testSensor<=W83781D_SYSCTL_IN6; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      machine.sensors[sensorCount]=new StdVoltage( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          14,
          defaultVoltNames[testSensor - W83781D_SYSCTL_IN0],
          defaultVoltConv[testSensor - W83781D_SYSCTL_IN0]	);		
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for fan sensors	
  for( testSensor=W83781D_SYSCTL_FAN1; testSensor<=W83781D_SYSCTL_FAN3; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      QString fanName;
      
      fanName.sprintf("Fan%d", testSensor - W83781D_SYSCTL_FAN1 );
      
      machine.sensors[sensorCount]=new StdFan( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          7000,
          fanName.data()	);		
      
      checkDisabled( sensorCount, override );		
    }
  }	
}

void Klm::addGl518( sensors_chips_data dirEntry, int &sensorCount, bool override )
{
  int 	testSensor;
  char 	*defaultVoltNames[]= { 
    "+5V",  // from lm_sensors
    "Vin1", // random guess values
    "Vin2",
    "Vin3" };
  
  double 	defaultVoltConv[]= {
    2.5, // from lm_sensors
    1.0, // rest are random guesses.
    1.0,
    1.0,
    1.0 };
  
  // add the temperature sensor
  if ( testSensorPresence( dirEntry.sysctl_id, GL518_SYSCTL_TEMP ) )
  {
    machine.sensors[sensorCount]=new StdTemp( 
        CTL_DEV,
        DEV_SENSORS,	
        dirEntry.sysctl_id, 
        GL518_SYSCTL_TEMP,
        0, 
        50,
        "Mbd" );
    
    checkDisabled( sensorCount, override );		
  }
  
  // probe for VID input 
  if ( testSensorPresence( dirEntry.sysctl_id, GL518_SYSCTL_VID ) )
  {
    machine.sensors[sensorCount]=new StdVid( 
        CTL_DEV, 
        DEV_SENSORS,
        dirEntry.sysctl_id, 
        GL518_SYSCTL_VID,
        0, 
        14	);		
    
    checkDisabled( sensorCount, override );		
  }
  
  //check the voltage inputs on the LM78	
  for( testSensor=GL518_SYSCTL_VDD; testSensor<=GL518_SYSCTL_VIN3; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      machine.sensors[sensorCount]=new StdVoltage( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          14,
          defaultVoltNames[testSensor - GL518_SYSCTL_VDD],
          defaultVoltConv[testSensor - GL518_SYSCTL_VDD]	);		
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for fan sensors	
  for( testSensor=GL518_SYSCTL_FAN1; testSensor<=GL518_SYSCTL_FAN2; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      QString fanName;
      
      fanName.sprintf("Fan%d", 1 + testSensor - W83781D_SYSCTL_FAN1 );
      
      machine.sensors[sensorCount]=new StdFan( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          7000,
          fanName.data()	);		
      
      checkDisabled( sensorCount, override );		
    }
  }	
}

void Klm::addAdm9240( sensors_chips_data dirEntry, int &sensorCount, bool override )
{
  int 	testSensor;
  char 	*defaultVoltNames[]= { 
    "12V",
    "2.5V",
    "3.3V",
    "5V",
    "Vccp1",
    "Vccp2" };
  
  //  The ADM has self scaling voltages. 
  double 	defaultVoltConv[]= {
    1.0,
    1.0,
    1.0,
    1.0,
    1.0,
    1.0 };
  
  if ( testSensorPresence( dirEntry.sysctl_id, ADM9240_SYSCTL_TEMP ) )
  {
    machine.sensors[sensorCount]=new StdTemp( 
        CTL_DEV,
        DEV_SENSORS,	
        dirEntry.sysctl_id, 
        ADM9240_SYSCTL_TEMP,
        0, 
        50,
        "Mbd" );
    
    checkDisabled( sensorCount, override );		
  }
  
  
  //check the voltage inputs on the ADM9240	
  for( testSensor=ADM9240_SYSCTL_IN0; testSensor<=ADM9240_SYSCTL_IN5; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      machine.sensors[sensorCount]=new StdVoltage( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          14,
          defaultVoltNames[testSensor - ADM9240_SYSCTL_IN0],
          defaultVoltConv[testSensor - ADM9240_SYSCTL_IN0]	);		
      
      checkDisabled( sensorCount, override );		
    }
  }
  
  // probe for fan sensors	
  for( testSensor=ADM9240_SYSCTL_FAN1; testSensor<=ADM9240_SYSCTL_FAN2; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {			
      QString fanName;
      
      fanName.sprintf("Fan%d", testSensor - LM78_SYSCTL_FAN1 );
      
      machine.sensors[sensorCount]=new StdFan( 
          CTL_DEV, 
          DEV_SENSORS,
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          7000,
          fanName.data()	);		
      
      checkDisabled( sensorCount, override );		
    }
  }	
}

void Klm::addAdm1021( sensors_chips_data dirEntry, int &sensorCount, bool override )
{	
  int	testSensor;
  
  for( testSensor=ADM1021_SYSCTL_TEMP; testSensor<=ADM1021_SYSCTL_REMOTE_TEMP; testSensor++)
  {
    if ( testSensorPresence( dirEntry.sysctl_id, testSensor ) )
    {
      QString defaultName;
      
      defaultName.sprintf("Temp-%d", numLm75Chips++);
      machine.sensors[sensorCount]=new MinMaxTemp( 
          CTL_DEV,
          DEV_SENSORS,	
          dirEntry.sysctl_id, 
          testSensor,
          0, 
          50,
          defaultName.data());
      
      checkDisabled( sensorCount, override );		
    }	
  }
}

int Klm::getSensorType( sensors_chips_data dirEntry )
{	
  QString sensorName( dirEntry.name );
  
  if ( sensorName.find("lm75") != -1 )
  {
    return LM75;
  }	
  
  if ( sensorName.find("lm78") != -1 )
  {
    return LM78;
  }	
  
  if ( sensorName.find("lm80") != -1 )
  {
    return LM80;
  }	
  
  if ( sensorName.find("lm79") != -1 )
  {
    return LM79;
  }	
  
  if( sensorName.find("sis5595") != -1 )
  {
    return SIS5595;
  }
  
  if ( sensorName.find("gl518sm") != -1 )
  {
    return GL518;
  }	
  
  if ( sensorName.find("w83781d") != -1 )
  {
    return W83781D;
  }	
  
  if ( sensorName.find("adm1021") != -1 )
  {
    return ADM1021;
  }	
  
  if ( sensorName.find("adm9240") != -1 )
  {
    return ADM9240;
  }	
  
  return NONE;
}

bool Klm::isIgnorable( 
  bool ignoreISA, 
  bool ignoreI2C, 
  sensors_chips_data dirEntry )
{
  QString sensorName( dirEntry.name );
  
  if (ignoreISA)
  {
    return ( sensorName.find("isa") != -1 );	
  }
  
  if (ignoreI2C)
  {
    return ( sensorName.find("i2c") != -1 );			
  }
  
  return false;
}

void Klm::probeSensors( bool override )
{
  int 				i;	
  int 				sensorCount = 0;
  int 				num_sensor_chips;
  int					st;
  sensors_chips_data 	directory[MAX_SENSOR_CHIPS]; 
  
  
  // read the directory of sensors 
  readSensorDirectory( 
    directory, 
    sizeof(directory),
    num_sensor_chips );
  
  for (i=0; i<num_sensor_chips; i++)
  {
    // fixme get ignoreISA and ignoreI2C somewhere else.
    if (!isIgnorable( false, false, directory[i] ))
    {
      st = getSensorType( directory[i] );
      switch (st)
      {
        case LM75:
          addLm75( directory[i], sensorCount, override );
          break;				
        case LM78:
          addLm78( directory[i], sensorCount, override );
          break;
        case LM79:
          addLm79( directory[i], sensorCount, override );
          break;
        case LM80:
          addLm80( directory[i], sensorCount, override );
          break;
        case SIS5595:
          addSis5595( directory[i], sensorCount, override );
          break;					
        case W83781D:
          addW83781d( directory[i], sensorCount, override );
          break;
        case GL518:
          addGl518( directory[i], sensorCount, override );
          break;
        case ADM1021:
          addAdm1021( directory[i], sensorCount, override );
          break;
        case ADM9240:
          addAdm9240( directory[i], sensorCount, override );
          break;					
        case NONE:
        default:
          cout << "unknown sensor encountered.\n";
          break;
      }	
    }	
  }
  
  machine.num_sensors = sensorCount;	
}

void Klm::maxOrMin()
{
  if (isVisible())
  {
    hide();
  }
  else
  {
    show();
  }
}

void Klm::saveQuit()
{
  save();	
  kapp->quit();
}

void Klm::save()
{
  int i;
  
  saveAppSettings();
  for( i=0; i<machine.num_sensors; i++)
  {
    machine.sensors[i]->save();
  }
  
}

void Klm::updateDisplay()
{
  int 		i;
  int 		numSearches;
  int 		status;
  bool 		displayed;
  bool 		alarm = false;
  bool 		tmpAlarm;
  QString 	sensorName;
  QString 	brokenSensors;
  SensorValue	val;
  
  brokenSensors = "";
  
  // there may be completed alarm script processes hanging
  // around waiting to complete. Get rid of them.
  while(waitpid(-1, &status, WNOHANG) > 0)
  {
    wait( &status );
  }			
  
  // Refresh all of the bar graphs.	
  for( i=0; i<machine.num_sensors; i++)
  {
    machine.graphs[i]->plot();		
    tmpAlarm = machine.sensors[i]->isAlarm();
    alarm |= tmpAlarm;
    if (tmpAlarm)
    {
      QString name;
      machine.sensors[i]->name(name);
      if( brokenSensors.length() > 0 )
      {
        brokenSensors += " ";
      }			
      brokenSensors += name.data();
    }
    
    // Update the contents of the dock widget, cycling through all of the
    // temperature sensors in the machine.
    displayed = false;
    numSearches = 0;
    
    if (alarm)
    {
      // if beeping enabled do so
      if( beepOnAlarm)
      {
        kapp->beep();
      }
      
      // if script enabled try to exec.
      if( scriptOnAlarm && (scriptCommand.length() > 0))	
      {
        QString cmd;
        
        cmd = scriptCommand;
        cmd.replace(
          QRegExp("%s"), 
          brokenSensors );
        
        execCommand( cmd );
      }	
    }
    
    while( !displayed && ( numSearches < machine.num_sensors ))
    {	
      if ( machine.sensors[dockDisplayedPtr]->isTemp() )
      {
        machine.sensors[dockDisplayedPtr]->name( sensorName );			
        machine.sensors[dockDisplayedPtr]->read( val );
        dock->display( val.value, alarm, sensorName );
        dockDisplayedPtr = ( dockDisplayedPtr + 1 ) % machine.num_sensors;
        displayed = true;		
      }
      else
      {
        dockDisplayedPtr = ( dockDisplayedPtr + 1 ) % machine.num_sensors;
        numSearches++;	
      }
    }
    
  }
}

void Klm::rebuild( bool override )
{
  int i;
  
  // allow the window to shrink to a minimum size so that it can
  // be rebuilt with the vertical item graphs having the correct
  // width rather than being stretched to fill the old wider window.
  setMinimumWidth( 0 );
  
  // reset the dock widget data pointer in case a temperature
  // sensor was deleled - fixes a nasty little crash.
  dockDisplayedPtr = 0;
  numLm75Chips = 0;
  
  // save all existing sensor and application settings
  save();
  
  // removed lmstats widget
  delete lmstats;
  
  // free all sensors
  for( i=0; i<machine.num_sensors; i++)
  {
    delete machine.sensors[i];
  }
  
  //redetect sensors (mark disabled ones as enabled).
  probeSensors( override );
  
  //create new lmstats
  lmstats = new LMStats(this, "klm_guts", &machine, statusBar);	
  
  connect(
    lmstats, SIGNAL( reloadMe() ),
    this, SLOT( sensorDisabled() ) );
  
  setView(lmstats, false);
  
  lmstats->show();
  
  resize( 
    lmstats->width(),  
    tb->height() + 
    kdemenu->height() + 
    lmstats->height() + 
    statusBar->height() );
  
  setMinimumSize( width(), height() );
  
  show();
  
  updateDisplay();	
}

void Klm::reInitialise()
{
  rebuild( true );
}

void Klm::sensorDisabled()
{
  rebuild( false );
}

void Klm::timerExpired()
{
  updateDisplay();
}

void Klm::showSettings()
{
  settings->setInterval(refreshInterval/1000);
  settings->setScriptStr(scriptCommand);
  settings->setBeepOnAlarm(beepOnAlarm);
  settings->setScriptOnAlarm(scriptOnAlarm);
  settings->show();
}

void Klm::hideSettings()
{
  settings->hide();	
  scriptCommand = settings->getScriptStr();
  saveAppSettings();
}

void Klm::newUpdateInterval( int value )
{
  QString str;
  
  str.sprintf(klocale->translate("Update interval set to %d seconds"), value );
  myTimer->changeInterval( 1000 * value );
  refreshInterval = 1000 * value;	
  statusBar->changeItem( str.data(), 0 );	
}

void Klm::getAppSettings()
{
  KConfig *config;
  QString group;
  
  // get the application config file
  config = kapp->getConfig();	
  
  // switch to the main klm group
  group = KLM_APP_GROUP;
  config->setGroup( group );
  
  // get settings
  refreshInterval = config->readLongNumEntry( REFRESH_INTERVAL, 30000 );
  beepOnAlarm = config->readBoolEntry( ALARM_BEEP, true );
  scriptOnAlarm = config->readBoolEntry( ALARM_EXEC, false );
  scriptCommand = config->readEntry( ALARM_SCRIPT, "" );	
}

void Klm::saveAppSettings()
{
  KConfig *config;
  QString group;
  
  // get the application config file
  config = kapp->getConfig();	
  
  // switch to the main klm group
  group = KLM_APP_GROUP;
  config->setGroup( group );
  
  // get settings
  config->writeEntry( REFRESH_INTERVAL, refreshInterval );
  config->writeEntry( ALARM_BEEP, beepOnAlarm );
  config->writeEntry( ALARM_EXEC, scriptOnAlarm );
  config->writeEntry( ALARM_SCRIPT, scriptCommand );
}

void Klm::alarmScriptToggled( bool state )
{	
  scriptOnAlarm = state;
  settings->setScriptOnAlarm(scriptOnAlarm);
  scriptCommand = settings->getScriptStr();
}

void Klm::alarmBeepToggled( bool state )
{
  beepOnAlarm = state;
}

void Klm::execCommand( const QString command )
{
  pid_t pid;
  int ptr = 0;
  char *args[100];	
  
  
  if ( command.length() == 0 )
  {
    return;
  }
  
  // break the command into strings	
  args[ptr++]=strtok( command.data(), " " );
  while((args[ptr++]=strtok( NULL, " " ))!=NULL);
  
  pid = fork();
  
  switch( pid )
  {
    case 0: // we are the child
      _exit(execvp(args[0],args));	
      break;
    case -1: // failure
      {
        QString str;
        
        str.sprintf(klocale->translate("Fork failed (%d)"), errno );
        statusBar->changeItem( str.data(), 0 );	
      }
      break;
    default: // wait for competion elsewhere.
      return;
  }	
}	

#include <klm.moc.cpp>
